﻿using Microsoft.AspNetCore.SignalR;
using Newtonsoft.Json;
using Sonic.Web.Core;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Resources;
using System;
using System.Linq;
using System.Collections.Generic;
using System.Net;
using System.Net.Http;
using System.Reflection;
using System.Text;
using System.IO;
using System.Net.NetworkInformation;
using Microsoft.AspNetCore.Http;
using System.Collections.ObjectModel;
namespace Sonic.Web.Service
{
    public class ChattingService
    {
        private readonly ChattingManager _chattingManager;
        private IHubContext<MessageHub> _hubContext;
        
        public ChattingService(IHubContext<MessageHub> hubContext)
        {
            _hubContext = hubContext;
            _chattingManager = new ChattingManager();
        }

        public GlobalErrors UpdateChattingUsers(ChattingUserModel user, int languageId)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (user != null && user.UserId != -1) // user: user who logged in / out
                {
                    if (!user.AddRemoveAssignment)
                    {
                        List<ChattingUserModel> availableUsers = new List<ChattingUserModel>(); // users to send notification to.
                        List<ChattingUserModel> BOUsers = new List<ChattingUserModel>();
                        List<ChattingUserModel> FOUsers = new List<ChattingUserModel>();

                        result = GetAvailableUsersForChatting(user.UserId, languageId, string.Empty, ref availableUsers);
                        if (result == GlobalErrors.Success && availableUsers != null && availableUsers.Count > 0)
                        {
                            BOUsers = availableUsers.Where(user1 => user1.IsBOUser == true).ToList();
                            FOUsers = availableUsers.Where(user1 => user1.IsBOUser == false && user1.DeviceToken != null && user1.DeviceToken != string.Empty).ToList();
                            
                            if(CheckForInternetConnection())
                            {
                                if (FOUsers != null && FOUsers.Count > 0)
                                {
                                    string deviceToken = string.Empty;
                                    List<string> devices = new List<string>();
                                    foreach (ChattingUserModel fouser in FOUsers)
                                    {
                                        devices.Add(fouser.DeviceToken);
                                    }
                                    SendNotificationWhenUpdatingChattingUsers(user, devices, NotificationTypes.UpdateChattingUserStatus, false);
                                }
                            }
                          
                            if (BOUsers != null && BOUsers.Count > 0)
                            {
                                UpdateChattingUsersForBOUsers(user);
                            }
                        }
                    }
                    else
                    {
                        UpdateChattingUsersForBOUsers(user);
                        result = GlobalErrors.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetUploadedFOUsersInChattingAssignment(int employeeId, bool uploadedOnly, int languageId, ref List<ChattingUserModel> uploadedFOUsers)
        {
            try
            {
                return _chattingManager.GetUploadedFOUsersInChattingAssignment(employeeId, uploadedOnly, languageId, ref uploadedFOUsers);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetAvailableUsersForChatting(int employeeId,int languageId, string rootPath, ref List<ChattingUserModel> availableUsers)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            List<int> data = new List<int>(); //inactive users to remove;
            try
            {
                result = _chattingManager.GetAvailableUsersForChatting(employeeId, languageId, ref availableUsers);
                if (result == GlobalErrors.Success && availableUsers != null && availableUsers.Count > 0)
                {
                    var inactiveUserIds = availableUsers.Where(x => x.Inactive == 1).Select(u => u.UserId).ToList();
                    if (inactiveUserIds != null && inactiveUserIds.Count > 0)
                    {
                        result = CheckInactiveEmployeeChattingHistory(inactiveUserIds, employeeId, ref data);
                        if (result == GlobalErrors.Success && data != null && data.Count > 0)
                        {
                            foreach (int removeId in data)
                            {
                                var idx = availableUsers.FindIndex(u => u.UserId == removeId);
                                if (idx > -1)
                                {
                                    availableUsers.RemoveAt(idx);
                                }
                            }
                        }
                    }
                    if (result == GlobalErrors.Success && !string.IsNullOrEmpty(rootPath))
                    {
                        foreach (ChattingUserModel user in availableUsers)
                        {
                            user.ImagePath = PrepareImagePathForUsers(rootPath, user.UserId);
                        }
                    }
                }
                //CheckInactiveEmployeeChattingHistory()
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public string PrepareImagePathForUsers(string rootPath, int userId)
        {
            string imagePath = string.Empty;
            try
            {
                string fileName = userId + ".jpg";
                string imageUserPath = rootPath + "\\Users-Img\\";
                List<string> filePaths = Directory.GetFiles(imageUserPath, fileName + "*.*").ToList();
                if (filePaths.Count > 0)
                {
                    imagePath = $"wwwroot/Users-Img/{userId}.jpg";
                }
                else
                {
                    imagePath = string.Empty;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return string.Empty;
            }
            return imagePath;
        }

        public GlobalErrors GetChattingHistory(ChattingHistoryModel chatInfo,int languageId, ref List<ChattingHistoryModel> chattingHistory)
        {
            try
            {
                return _chattingManager.GetChattingHistory(chatInfo, languageId, ref chattingHistory);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetChattingUserData(int userId,int languageId, string rootPath, ref ChattingUserModel user)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _chattingManager.GetChattingUserData(userId, languageId, ref user);
                if (result == GlobalErrors.Success && !string.IsNullOrEmpty(rootPath))
                {
                    user.ImagePath = PrepareImagePathForUsers(rootPath, user.UserId);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SendMessage(ChattingHistoryModel message, Collection<IFormFile> attachedFiles, string rootPath, bool isBroadCast,bool hasSupervisorApp)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                message.HasText = string.IsNullOrEmpty(message.MessageBody) ? false : true;

                #region[Save ChattingHistory]
                result = _chattingManager.SaveChattingHistory(message, isBroadCast, null);
                #endregion

                #region[Save Message Media To DB]
                if(result == GlobalErrors.Success && attachedFiles != null && attachedFiles.Count > 0)
                {
                    List<ChattingHistoryMediaModel> attachedMediaFiles = new List<ChattingHistoryMediaModel>();
                    foreach (var file in attachedFiles)
                    {
                        string extension = Path.GetExtension(file.FileName).ToLowerInvariant();

                        ChattingHistoryMediaModel fileObj = new ChattingHistoryMediaModel();
                        fileObj.FileName = file.FileName;
                        fileObj.MessageId = message.MessageId;
                        string contentType = file.ContentType;
                        fileObj.MediaType = GetChattingMediaTypeFromExtension(extension);
                        fileObj.ImagePath = fileObj.MediaType == ChattingMediaTypes.Image ? $"wwwroot/ChattingMedia/{message.MessageId}/{file.FileName}" : string.Empty;
                        attachedMediaFiles.Add(fileObj);
                    }
                    message.MediaList = attachedMediaFiles;
                    message.HasMedia = true;
                    result = _chattingManager.SaveChattingHistoryMedia(message.MessageId,null,attachedMediaFiles, null);
                }
                #endregion

                #region[Get Customer-Outlet Descriptions]
                if (result == GlobalErrors.Success && message.CustomerId > -1 && message.OutletId > -1)
                {
                    List<CustomerOutletDescriptions> descriptions = new List<CustomerOutletDescriptions>();
                    result = _chattingManager.GetCustomerOutletDescriptions(message.CustomerId, message.OutletId, ref descriptions);
                    if(result == GlobalErrors.Success && descriptions != null && descriptions.Count > 0)
                    {
                        message.CustomerOutletDescriptions = descriptions;
                    }
                }
                #endregion

                #region[Prepare Image Path]
                if (result == GlobalErrors.Success && !string.IsNullOrEmpty(rootPath))
                {
                    message.ImagePath = PrepareImagePathForUsers(rootPath, message.SenderId);
                }
                #endregion

                #region[Save Message Media To Folder]
                if (result == GlobalErrors.Success && attachedFiles != null && attachedFiles.Count > 0)
                {
                    List<string> fileNames = new List<string>();
                    string targetFolderPath = rootPath + "\\ChattingMedia" + "\\" + message.MessageId;
                    if (!Directory.Exists(targetFolderPath))
                    {
                        Directory.CreateDirectory(targetFolderPath);
                    }
                    for (int i = 0; i < attachedFiles.Count; i++)
                    {
                        var file = attachedFiles[i];
                        if (file.Length > 0)
                        {
                            //if (file.Name.ToString().ToLower() == "attachedfiles")
                            {
                                string fileName = file.FileName;
                                string filePath = Path.Combine(targetFolderPath, fileName);
                                using (var stream = new FileStream(filePath, FileMode.Create))
                                {
                                    file.CopyTo(stream);
                                }
                            }
                        }
                    }
                }
                #endregion

                #region[Send Message Using SignalR And FireBase ]
                if (result == GlobalErrors.Success)
                {
                    if (message.IsBOReceiver)
                    {
                        if (_hubContext != null)
                        {
                            // Send message to BO user
                            _hubContext.Clients.All.SendAsync("MessageReceived", message);
                        }
                        if (hasSupervisorApp)
                        {
                            SendSuperVisorMessageUsingFirebasePushNotification(message);
                        }
                    }
                    else
                    {
                        // Send message to FO user
                        SendMessageUsingFirebasePushNotification(message);
                    }
                }
                #endregion
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SendBroadcast(ChattingHistoryModel broadcastMessage , Collection<IFormFile> attachedFiles, string rootPath)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dBHelper = null;
            try
            {
                string broadcastId = Guid.NewGuid().ToString();
                broadcastMessage.BroadcastId = broadcastId;
                broadcastMessage.HasText = string.IsNullOrEmpty(broadcastMessage.MessageBody) ? false : true;

                List<ChattingHistoryMediaModel> broadcastMediaFiles = new List<ChattingHistoryMediaModel>();

                List<BroadcastReceiverModel> FOReceiversList = new List<BroadcastReceiverModel>();
                List<BroadcastReceiverModel> BOReceiversList = new List<BroadcastReceiverModel>();

                dBHelper = new DBHelper<int>();
                dBHelper.BeginTransaction();

                if (broadcastMessage.ReceiversList != null && broadcastMessage.ReceiversList.Count > 0)
                {
                    #region[Prepare media for broadcast in general]
                    if (attachedFiles != null && attachedFiles.Count > 0)
                    {
                        broadcastMediaFiles = new List<ChattingHistoryMediaModel>();
                        foreach (var file in attachedFiles)
                        {
                            string extension = Path.GetExtension(file.FileName).ToLowerInvariant();

                            ChattingHistoryMediaModel fileObj = new ChattingHistoryMediaModel();
                            fileObj.FileName = file.FileName;
                            //fileObj.MessageId = broadcastId; // ??
                            string contentType = file.ContentType;
                            fileObj.MediaType = GetChattingMediaTypeFromExtension(extension);
                            fileObj.ImagePath = fileObj.MediaType == ChattingMediaTypes.Image ? $"wwwroot/ChattingMedia/{broadcastId}/{file.FileName}" : string.Empty;
                            broadcastMediaFiles.Add(fileObj);
                        }
                        broadcastMessage.MediaList = broadcastMediaFiles;
                        broadcastMessage.HasMedia = true;
                    }
                    #endregion

                    #region[Save ChattingHistory and ChattingHistoryMedia Per Receiver]
                    foreach (BroadcastReceiverModel receiver in broadcastMessage.ReceiversList)
                    {
                        broadcastMessage.ReceiverId = receiver.ReceiverId;
                        broadcastMessage.ReceiverName = receiver.ReceiverName;
                        broadcastMessage.MessageId = receiver.MessageId;

                        result = _chattingManager.SaveChattingHistory(broadcastMessage, true, dBHelper);
                        if(result == GlobalErrors.Success && broadcastMediaFiles != null && broadcastMediaFiles.Count > 0)
                        {
                            result = _chattingManager.SaveChattingHistoryMedia(broadcastMessage.MessageId, broadcastMessage.MediaExpiryDateModel, broadcastMediaFiles, dBHelper);
                        }
                        if (result != GlobalErrors.Success)
                        {
                            break;
                        }
                    }
                    #endregion
                }
                #region[Prepare Image Path]
                if (result == GlobalErrors.Success && !string.IsNullOrEmpty(rootPath))
                {
                    broadcastMessage.ImagePath = PrepareImagePathForUsers(rootPath, broadcastMessage.SenderId);
                }
                #endregion

                #region[Save Broadcast Media To Folder]
                if (result == GlobalErrors.Success && attachedFiles != null && attachedFiles.Count > 0)
                {
                    List<string> fileNames = new List<string>();
                    string targetFolderPath = rootPath + "\\ChattingMedia" + "\\" + broadcastId;
                    if (!Directory.Exists(targetFolderPath))
                    {
                        Directory.CreateDirectory(targetFolderPath);
                    }
                    for (int i = 0; i < attachedFiles.Count; i++)
                    {
                        var file = attachedFiles[i];
                        if (file.Length > 0)
                        {
                                string fileName = file.FileName;
                                string filePath = Path.Combine(targetFolderPath, fileName);
                                using (var stream = new FileStream(filePath, FileMode.Create))
                                {
                                    file.CopyTo(stream);
                                }
                        }
                    }
                }
                #endregion

                #region[Send Broadcast Using SignalR and Firebase]
                if (result == GlobalErrors.Success)
                {
                    FOReceiversList = broadcastMessage.ReceiversList.Where(user => user.IsBOReceiver == false && user.ReceiverDeviceToken != null &&
                    user.ReceiverDeviceToken != string.Empty).ToList();
                    BOReceiversList = broadcastMessage.ReceiversList.Where(user => user.IsBOReceiver == true).ToList();

                    // Send hub to BO users.
                    if (BOReceiversList != null && BOReceiversList.Count > 0)
                    {
                        var BObroadcastMessage = broadcastMessage;
                        BObroadcastMessage.ReceiversList = BOReceiversList;
                        if (_hubContext != null)
                        {
                            // Send message to BO user
                            _hubContext.Clients.All.SendAsync("MessageReceived", BObroadcastMessage);
                        }
                    }
                    // Send firebase to FO users.
                    if (FOReceiversList != null && FOReceiversList.Count > 0)
                    {
                        List<string> deviceTokens = new List<string>();

                        foreach (BroadcastReceiverModel foReceiver in FOReceiversList)
                        {
                            deviceTokens.Add(foReceiver.ReceiverDeviceToken);

                            broadcastMessage.ReceiverId = foReceiver.ReceiverId;
                            broadcastMessage.ReceiverName = foReceiver.ReceiverName;
                            broadcastMessage.IsBOReceiver = foReceiver.IsBOReceiver;
                            broadcastMessage.ReceiverDeviceToken = foReceiver.ReceiverDeviceToken;
                            broadcastMessage.MessageId = foReceiver.MessageId;
                            // Send message to FO user:
                            SendMessageUsingFirebasePushNotification(broadcastMessage);
                        }
                        //SendBroadcastUsingFirebasePushNotification(broadcastMessage, deviceTokens);
                    }
                }
                #endregion
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dBHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dBHelper.CommitTransaction();
                    }
                    else
                    {
                        dBHelper.RollBackTransaction();
                    }
                    dBHelper.Dispose();
                    dBHelper = null;
                }
            }
        }
        
        public GlobalErrors SetMessagesAsRead(ChattingHistoryModel message)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;

            try
            {
                result = _chattingManager.SetMessagesAsRead(message);
                if(result == GlobalErrors.Success)
                {
                    
                        // If the sender of the current message is BO user, inform the BO sender that his message is read using hub,
                        // else, If the sender of message is FO User, inform the FO sender that his message is read using Firebase, 
                        if (message.IsBOSender || message.IsBOReceiver)
                        {
                            if (_hubContext != null)
                            {
                                // set message as read to BO user
                                _hubContext.Clients.All.SendAsync("MessageRead", message);
                            }
                        }
                        if(!message.IsBOSender)
                        {
                            // set message as read to FO user [send one message only as all messages are set to read by default]
                            //SetMessageAsReadUsingFirebasePushNotification(message);
                            string senderDeviceToken = string.Empty;
                            result = _chattingManager.GetDeviceToken(message.SenderDeviceSerial, ref senderDeviceToken);
                            if (result == GlobalErrors.Success && !string.IsNullOrEmpty(senderDeviceToken))
                            {
                                message.SenderDeviceToken = senderDeviceToken;
                                SendUsingFirebasePushNotification(message, NotificationTypes.SetMessageAsRead, message.SenderDeviceToken);
                            }
                        }

                   
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors SetAttachmentAsDownloaded(ChattingHistoryModel message)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                ChattingHistoryMediaModel mediaFile = message.MediaList.First();
                DateTime downloadDate = DateTime.Now;
                if(mediaFile != null)
                {
                    mediaFile.DownloadDate = downloadDate.ToString();
                    result = _chattingManager.SetAttachmentAsDownloaded(message.MessageId, mediaFile.FileName, downloadDate, mediaFile.MediaType);
                }
                if (result == GlobalErrors.Success)
                {
                    if(message.IsBOSender || message.IsBOReceiver)
                    {
                        if (_hubContext != null)
                        {
                            _hubContext.Clients.All.SendAsync("MediaDownloaded", message);
                        }
                    }
                    if (!message.IsBOSender) // FO Sender
                    {
                        string senderDeviceToken = string.Empty;
                        result = _chattingManager.GetDeviceToken(message.SenderDeviceSerial, ref senderDeviceToken);
                        if(result == GlobalErrors.Success && !string.IsNullOrEmpty(senderDeviceToken))
                        {
                            message.SenderDeviceToken = senderDeviceToken;
                            SendUsingFirebasePushNotification(message, NotificationTypes.ChattingMediaDownloaded, message.SenderDeviceToken);
                        }

                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors SetEmployeeMessagesAsReceived(int employeeId, ChattingHistoryModel message)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                #region[Update ChattingHistory Table]
                result = _chattingManager.SetEmployeeMessagesAsReceived(employeeId);
                #endregion

                #region[Send SignalR or Firebase]
                if(result == GlobalErrors.Success)
                {
                    if (message.IsBOSender || message.IsBOReceiver)
                    {
                        if (_hubContext != null)
                        {
                            _hubContext.Clients.All.SendAsync("SetMessageAsReceived", message);
                        }
                    }
                    if (!message.IsBOSender)
                    {
                        string senderDeviceToken = string.Empty;
                        result = _chattingManager.GetDeviceToken(message.SenderDeviceSerial, ref senderDeviceToken);
                        if (result == GlobalErrors.Success && !string.IsNullOrEmpty(senderDeviceToken))
                        {
                            message.SenderDeviceToken = senderDeviceToken;
                            SendUsingFirebasePushNotification(message, NotificationTypes.SetMessageAsReceived, message.SenderDeviceToken);
                        }
                    }
                }
                #endregion

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetEmployeeUnreadMessages(int employeeId, int languageId, string rootPath, ref List<ChattingHistoryModel> unreadMessages)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _chattingManager.GetEmployeeUnreadMessages(employeeId, languageId, ref unreadMessages);
                if (result == GlobalErrors.Success && !string.IsNullOrEmpty(rootPath))
                {
                    foreach (ChattingHistoryModel msg in unreadMessages)
                    {
                        msg.ImagePath = PrepareImagePathForUsers(rootPath, msg.SenderId);
                    }
                }
                #region[Set Logged In user messages as received]
                if (result == GlobalErrors.Success)
                {
                    result = _chattingManager.SetEmployeeMessagesAsReceived(employeeId);
                }
                #endregion
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors CheckInactiveEmployeeChattingHistory(List<int> inactiveUsersIds, int employeeId,ref List<int> data)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;

            try
            {
                return _chattingManager.CheckInactiveEmployeeChattingHistory(inactiveUsersIds, employeeId,ref data);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetReceiverToken(string deviceSerial, ref string receiverDeviceToken)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _chattingManager.GetDeviceToken(deviceSerial, ref receiverDeviceToken);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetReceiversTokenList(ref List<BroadcastReceiverModel> receivers)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _chattingManager.GetReceiversTokenList(ref receivers);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public async void SendMessageUsingFirebasePushNotification(ChattingHistoryModel message)
        {
            try
            {
                if(message.ReceiverDeviceToken != null && message.ReceiverDeviceToken != string.Empty && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                {
                    var data = new Object();
                    data = new
                    {
                        to = message.ReceiverDeviceToken,
                        data = new
                        {
                            type = NotificationTypes.Chatting.GetHashCode(),
                            ChattingHistory = message
                        }
                    };
                    var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
                    var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);
                    var jsonBody = JsonConvert.SerializeObject(data);
                    using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
                    {
                        httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
                        httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
                        httpRequest.Content = new StringContent(jsonBody, Encoding.UTF8, "application/json");
                        ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
                        using (var httpClient = new HttpClient())
                        {
                            if(CheckForInternetConnection())
                            {
                                try
                                {
                                    var res = await httpClient.SendAsync(httpRequest);
                                    if (res.IsSuccessStatusCode)
                                    {
                                    }
                                    else
                                    {

                                    }
                                }
                                catch (Exception ex)
                                {
                                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                                }

                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
        }
        public async void SendSuperVisorMessageUsingFirebasePushNotification(ChattingHistoryModel message)
        {
            try
            {
                string devToken = "";
                _chattingManager.IsEmployeeHasDeviceTokenExists(message.ReceiverId, ref devToken);
                if (devToken != null && devToken != string.Empty && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                {
                    var data = new Object();
                    data = new
                    {
                        to = devToken,
                        data = new
                        {
                            type = NotificationTypes.Chatting.GetHashCode(),
                            ChattingHistory = message
                        },
                        notification = new
                        {
                            title = "New message from (" + message.SenderName + ")",
                            body = message.MessageBody,
                        }
                    };
                    var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
                    var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);
                    var jsonBody = JsonConvert.SerializeObject(data);
                    using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
                    {
                        httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
                        httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
                        httpRequest.Content = new StringContent(jsonBody, Encoding.UTF8, "application/json");
                        ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
                        using (var httpClient = new HttpClient())
                        {
                            if (CheckForInternetConnection())
                            {
                                try
                                {
                                    var res = await httpClient.SendAsync(httpRequest);
                                    if (res.IsSuccessStatusCode)
                                    {
                                    }
                                    else
                                    {

                                    }
                                }
                                catch (Exception ex)
                                {
                                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                                }

                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
        }
        public async void SendBroadcastUsingFirebasePushNotification(ChattingHistoryModel message, List<string> deviceTokens)
        {
            try
            {
                if (deviceTokens != null && deviceTokens.Count > 0 && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                {
                    var data = new Object();
                    data = new
                    {
                        registration_ids = deviceTokens,
                        data = new
                        {
                            type = NotificationTypes.Chatting.GetHashCode(),
                            ChattingHistory = message
                        }
                    };
                    var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
                    var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);
                    var jsonBody = JsonConvert.SerializeObject(data);
                    using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
                    {
                        httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
                        httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
                        httpRequest.Content = new StringContent(jsonBody, Encoding.UTF8, "application/json");
                        ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
                        using (var httpClient = new HttpClient())
                        {
                            if (CheckForInternetConnection())
                            {
                                try
                                {
                                    var res = await httpClient.SendAsync(httpRequest);
                                    if (res.IsSuccessStatusCode)
                                    {
                                    }
                                    else
                                    {

                                    }
                                }
                                catch (Exception ex)
                                {
                                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                                }

                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
        }

        //public async void SetMessageAsReadUsingFirebasePushNotification(ChattingHistoryModel message)
        //{
        //    try
        //    {
        //        if (message.SenderDeviceToken != null && message.SenderDeviceToken != string.Empty && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
        //        {
        //            var data = new Object();
        //            data = new
        //            {
        //                to = message.SenderDeviceToken,
        //                data = new
        //                {
        //                    type = NotificationTypes.SetMessageAsRead.GetHashCode(),
        //                    ChattingHistory = message
        //                }
        //            };
        //            var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
        //            var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);
        //            var jsonBody = JsonConvert.SerializeObject(data);
        //            using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
        //            {
        //                httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
        //                httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
        //                httpRequest.Content = new StringContent(jsonBody, Encoding.UTF8, "application/json");
        //                ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
        //                using (var httpClient = new HttpClient())
        //                {
        //                    if(CheckForInternetConnection())
        //                    {
        //                        try
        //                        {
        //                            var res = await httpClient.SendAsync(httpRequest);
        //                            if (res.IsSuccessStatusCode)
        //                            {
        //                            }
        //                            else
        //                            {

        //                            }
        //                        }
        //                        catch (Exception ex)
        //                        {
        //                            ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //                        }

        //                    }
        //                }
        //            }
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //    }
        //}
        public async void SendUsingFirebasePushNotification(ChattingHistoryModel message, NotificationTypes notificationType,string deviceToken)
        {
            try
            {
                if (deviceToken != null && deviceToken != string.Empty && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                {
                    var data = new Object();
                    data = new
                    {
                        to = deviceToken,
                        data = new
                        {
                            type = notificationType.GetHashCode(),
                            ChattingHistory = message
                        }
                    };
                    var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
                    var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);
                    var jsonBody = JsonConvert.SerializeObject(data);
                    using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
                    {
                        httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
                        httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
                        httpRequest.Content = new StringContent(jsonBody, Encoding.UTF8, "application/json");
                        ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
                        using (var httpClient = new HttpClient())
                        {
                            if(CheckForInternetConnection())
                            {
                                try
                                {
                                    var res = await httpClient.SendAsync(httpRequest);
                                    if (res.IsSuccessStatusCode)
                                    {
                                    }
                                    else
                                    {

                                    }
                                }
                                catch (Exception ex)
                                {
                                    ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                                }

                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
        }

        public async void SendNotificationWhenUpdatingChattingUsers(ChattingUserModel user, List<string> connectedUserDeviceTokens, NotificationTypes notificationType, bool Await)
        {
            if (connectedUserDeviceTokens != null && connectedUserDeviceTokens.Count > 0 && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
            {
                var data = new Object();
                data = new
                {
                    registration_ids = connectedUserDeviceTokens,
                    data = new
                    {
                        type = notificationType.GetHashCode(),
                        ChattingUser = user
                    }
                };
                var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
                var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);
                var jsonBody = JsonConvert.SerializeObject(data);
                using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
                {
                    httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
                    httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
                    httpRequest.Content = new StringContent(jsonBody, Encoding.UTF8, "application/json");
                    ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
                    using (var httpClient = new HttpClient())
                    {

                        bool connected = CheckForInternetConnection();
                        if(connected)
                        {
                            try
                            {
                                var res = await httpClient.SendAsync(httpRequest);
                                if (res.IsSuccessStatusCode)
                                {
                                }
                                else
                                {

                                }
                            }
                            catch (Exception ex)
                            {
                                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                            }

                        }
                    }
                }
            }

        }
        public static bool CheckForInternetConnection()
        {
            try
            {
                using (var client = new WebClient())
                using (client.OpenRead("https://google.com/generate_204"))
                    return true;
            }
            catch
            {
                return false;
            }
        }

        //test;
        //public bool PingHost()
        //{

        //    string host = "https://fcm.googleapis.com/fcm/send";
        //    Uri uri = new Uri(host);
        //    int timeOut = 5000;
        //    bool result = false;
        //    Ping p = new Ping();
        //    try
        //    {
        //        PingReply reply = p.Send(uri.Host, timeOut);

        //        if (reply.Status == IPStatus.Success)
        //            return true;
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //    }
        //    return result;
        //}
        public GlobalErrors UpdateChattingUsersForBOUsers(ChattingUserModel user)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (_hubContext != null)
                {
                    _hubContext.Clients.All.SendAsync("UpdateChattingUsers", user);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public ChattingMediaTypes GetChattingMediaTypeFromExtension(string fileExtension)
        {
            // Find a better solution than this...
            try
            {
                switch (fileExtension)
                {
                    case ".png":
                    case ".jpeg":
                    case ".jpg":
                    case ".bmp":
                    case ".jpe":
                    case ".gif":
                        return ChattingMediaTypes.Image;

                    case ".mp4":
                    case ".mov":
                    case ".wmv":
                    case ".avi":
                        return ChattingMediaTypes.Video;

                    default:
                        return ChattingMediaTypes.Attachment;

                        //case ".doc":
                        //case ".docx":
                        //case ".pdf":
                        //case ".xlsx":
                        //case ".xlsm":
                        //case ".xlsb":
                        //case ".xltx":
                        //    return ChattingMediaTypes.Attachment;
                        //default:
                        //    return ChattingMediaTypes.None;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return ChattingMediaTypes.None;
            }
        }

        #region[Chatting List Assignment]
        public GlobalErrors SaveAddedEmployeesToChattingList(List<ChattingAssignmentModel> assignmentList, int employeeId, int languageId)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string addedEmployeeIds = string.Empty;

            try
            {
                List<ChattingUserModel> availableUsers = new List<ChattingUserModel>();
                List<ChattingUserModel> BOUsers = new List<ChattingUserModel>();
                List<ChattingUserModel> FOUsers = new List<ChattingUserModel>();
                addedEmployeeIds = string.Join(',', assignmentList.Select(a=> a.AddedUserId));
                result = _chattingManager.SaveAddedEmployeesToChattingList(assignmentList);
                if (result == GlobalErrors.Success)
                {
                    if ( assignmentList[0].SelectedUserId >= 0 || assignmentList[0].SelectedSecurityGroupId >= 0)
                    {
                        if(assignmentList[0].SelectedUserId >= 0)
                        {
                            result = _chattingManager.GetUsersToNotifyOfAssignmentChange(assignmentList[0].SelectedUserId, -1, languageId, true, false, false, false, addedEmployeeIds, ref availableUsers);
                        }
                        else if (assignmentList[0].SelectedSecurityGroupId >= 0)
                        {
                            // get all emps available for this S.G.
                            result = _chattingManager.GetUsersToNotifyOfAssignmentChange(-1, assignmentList[0].SelectedSecurityGroupId, languageId, false, true, false, false, addedEmployeeIds, ref availableUsers);
                        }
                        if (result == GlobalErrors.Success && availableUsers != null && availableUsers.Count > 0)
                        {
                            BOUsers = availableUsers.Where(user => user.IsBOUser == true ).ToList();
                            FOUsers = availableUsers.Where(user => user.IsBOUser == false  && user.DeviceToken != null && user.DeviceToken != string.Empty).ToList();

                            ChattingUserModel user = new ChattingUserModel();
                            user.UserId = employeeId;
                            user.AddRemoveAssignment = true;
                            user.AssignmentStatus = ChattingAssignmentStatus.Add;
                            user.ChattingAssignmentStatusValue = ChattingAssignmentStatus.Add.GetHashCode();
                            user.IsBOUser = true;
                            if (CheckForInternetConnection())
                            {
                                if (FOUsers != null && FOUsers.Count > 0)
                                {
                                    string deviceToken = string.Empty;
                                    List<string> devices = new List<string>();

                                    foreach (ChattingUserModel fouser in FOUsers)
                                    {
                                        devices.Add(fouser.DeviceToken);
                                    }
                                    SendNotificationWhenUpdatingChattingUsers(user, devices, NotificationTypes.AddRemoveChattingAssignment, false);
                                }
                            }
                              
                            if (BOUsers != null && BOUsers.Count > 0)
                            {
                                UpdateChattingUsersForBOUsers(user);
                            }
                        }
                    }                    
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveAddedSecurityGroupsToChattingList(List<ChattingAssignmentModel> assignmentList, int employeeId, int languageId)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string addedSecurityGroupIds = string.Empty;

            try
            {
                List<ChattingUserModel> availableUsers = new List<ChattingUserModel>();
                List<ChattingUserModel> BOUsers = new List<ChattingUserModel>();
                List<ChattingUserModel> FOUsers = new List<ChattingUserModel>();
                addedSecurityGroupIds = string.Join(',', assignmentList.Select(a => a.AddedSecurityGroupId));
                result = _chattingManager.SaveAddedSecurityGroupsToChattingList(assignmentList);
                if (result == GlobalErrors.Success)
                {

                    if (assignmentList[0].SelectedUserId >= 0 || assignmentList[0].SelectedSecurityGroupId >= 0)
                    {
                        if(assignmentList[0].SelectedUserId >= 0)
                        {
                            result = _chattingManager.GetUsersToNotifyOfAssignmentChange(assignmentList[0].SelectedUserId, -1, languageId, false, false, true, false, addedSecurityGroupIds, ref availableUsers);
                        }
                        else if (assignmentList[0].SelectedSecurityGroupId >= 0)
                        {
                            result = _chattingManager.GetUsersToNotifyOfAssignmentChange(-1, assignmentList[0].SelectedSecurityGroupId, languageId, false, false, false, true, addedSecurityGroupIds, ref availableUsers);
                        }
                        if (result == GlobalErrors.Success && availableUsers != null && availableUsers.Count > 0)
                        {
                            BOUsers = availableUsers.Where(user => user.IsBOUser == true ).ToList();
                            FOUsers = availableUsers.Where(user => user.IsBOUser == false  && user.DeviceToken != null && user.DeviceToken != string.Empty).ToList();

                            ChattingUserModel user = new ChattingUserModel();
                            user.UserId = employeeId;
                            user.AddRemoveAssignment = true;
                            user.AssignmentStatus = ChattingAssignmentStatus.Add;
                            user.ChattingAssignmentStatusValue = ChattingAssignmentStatus.Add.GetHashCode();
                            user.IsBOUser = true;
                            if(CheckForInternetConnection())
                            {
                                if (FOUsers != null && FOUsers.Count > 0)
                                {
                                    string deviceToken = string.Empty;
                                    List<string> devices = new List<string>();

                                    foreach (ChattingUserModel fouser in FOUsers)
                                    {
                                        devices.Add(fouser.DeviceToken);
                                    }
                                    SendNotificationWhenUpdatingChattingUsers(user, devices, NotificationTypes.AddRemoveChattingAssignment, false);
                                }
                            }
                          
                            if (BOUsers != null && BOUsers.Count > 0)
                            {
                                UpdateChattingUsersForBOUsers(user);
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors DeleteEmployeesFromChattingList(List<ChattingAssignmentModel> assignmentList, int employeeId, int languageId)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string addedEmployeeIds = string.Empty;
            try
            {
                List<ChattingUserModel> availableUsers = new List<ChattingUserModel>();
                List<ChattingUserModel> BOUsers = new List<ChattingUserModel>();
                List<ChattingUserModel> FOUsers = new List<ChattingUserModel>();
                addedEmployeeIds = string.Join(',', assignmentList.Select(a => a.AddedUserId));

                if (assignmentList[0].SelectedUserId >= 0 || assignmentList[0].SelectedSecurityGroupId >= 0)
                {
                    if (assignmentList[0].SelectedUserId >= 0)
                    {
                        result = _chattingManager.GetUsersToNotifyOfAssignmentChange(assignmentList[0].SelectedUserId, -1, languageId, true, false, false, false, addedEmployeeIds, ref availableUsers);
                    }
                    else if (assignmentList[0].SelectedSecurityGroupId >= 0)
                    {
                        result = _chattingManager.GetUsersToNotifyOfAssignmentChange(-1, assignmentList[0].SelectedSecurityGroupId, languageId, false, true, false, false, addedEmployeeIds, ref availableUsers);
                    }
                    if (result == GlobalErrors.Success && availableUsers != null && availableUsers.Count > 0)
                    {
                        result = _chattingManager.DeleteEmployeesFromChattingList(assignmentList);
                        if (result == GlobalErrors.Success)
                        {
                            BOUsers = availableUsers.Where(user => user.IsBOUser == true).ToList();
                            FOUsers = availableUsers.Where(user => user.IsBOUser == false && user.DeviceToken != null && user.DeviceToken != string.Empty).ToList();

                            ChattingUserModel user = new ChattingUserModel();
                            user.UserId = employeeId;
                            user.AddRemoveAssignment = true;
                            user.AssignmentStatus = ChattingAssignmentStatus.Remove;
                            user.ChattingAssignmentStatusValue = ChattingAssignmentStatus.Remove.GetHashCode();
                            user.IsBOUser = true;

                            if(CheckForInternetConnection())
                            {
                                if (FOUsers != null && FOUsers.Count > 0)
                                {
                                    string deviceToken = string.Empty;
                                    List<string> devices = new List<string>();

                                    foreach (ChattingUserModel fouser in FOUsers)
                                    {
                                        devices.Add(fouser.DeviceToken);
                                    }
                                    SendNotificationWhenUpdatingChattingUsers(user, devices, NotificationTypes.AddRemoveChattingAssignment, false);
                                }
                            }
                            
                            if (BOUsers != null && BOUsers.Count > 0)
                            {
                                UpdateChattingUsersForBOUsers(user);
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors DeleteSecurityGroupsFromChattingList(List<ChattingAssignmentModel> assignmentList, int employeeId, int languageId)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string addedSecurityGroupIds = string.Empty;

            try
            {
                List<ChattingUserModel> availableUsers = new List<ChattingUserModel>();
                List<ChattingUserModel> BOUsers = new List<ChattingUserModel>();
                List<ChattingUserModel> FOUsers = new List<ChattingUserModel>();
                addedSecurityGroupIds = string.Join(',', assignmentList.Select(a => a.AddedSecurityGroupId));

                if (assignmentList[0].SelectedUserId >= 0 || assignmentList[0].SelectedSecurityGroupId >= 0)
                {
                    if (assignmentList[0].SelectedUserId >= 0)
                    {
                        result = _chattingManager.GetUsersToNotifyOfAssignmentChange(assignmentList[0].SelectedUserId, -1, languageId, false, false, true, false, addedSecurityGroupIds, ref availableUsers);
                    }
                    else if (assignmentList[0].SelectedSecurityGroupId >= 0)
                    {
                        result = _chattingManager.GetUsersToNotifyOfAssignmentChange(-1, assignmentList[0].SelectedSecurityGroupId, languageId, false, false, false, true, addedSecurityGroupIds, ref availableUsers);
                    }
                    if (result == GlobalErrors.Success && availableUsers != null && availableUsers.Count > 0)
                    {
                        result = _chattingManager.DeleteSecurityGroupsFromChattingList(assignmentList);
                        if (result == GlobalErrors.Success)
                        {
                            BOUsers = availableUsers.Where(user => user.IsBOUser == true ).ToList();
                            FOUsers = availableUsers.Where(user => user.IsBOUser == false && user.DeviceToken != null && user.DeviceToken != string.Empty).ToList();

                            ChattingUserModel user = new ChattingUserModel();
                            user.UserId = employeeId;
                            user.AddRemoveAssignment = true;
                            user.AssignmentStatus = ChattingAssignmentStatus.Remove;
                            user.ChattingAssignmentStatusValue = ChattingAssignmentStatus.Remove.GetHashCode();
                            user.IsBOUser = true;

                            if(CheckForInternetConnection())
                            {
                                if (FOUsers != null && FOUsers.Count > 0)
                                {
                                    string deviceToken = string.Empty;
                                    List<string> devices = new List<string>();

                                    foreach (ChattingUserModel fouser in FOUsers)
                                    {
                                        devices.Add(fouser.DeviceToken);
                                    }
                                    SendNotificationWhenUpdatingChattingUsers(user, devices, NotificationTypes.AddRemoveChattingAssignment, false);
                                }
                            }
                           
                            if (BOUsers != null && BOUsers.Count > 0)
                            {
                                UpdateChattingUsersForBOUsers(user);
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        #endregion
    }
}
