﻿using Microsoft.AspNetCore.Http;
using Sonic.Web.Core;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Models;
using Sonic.Web.Resources;
using Sonic.Web.SecureLibrary;
using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Threading.Tasks;
using System.Linq;

namespace Sonic.Web.Service
{
    public class EmployeeService
    {
        private readonly IRequestRepository _requestRepository;
        private readonly MasterDataService _masterDataService;
        private readonly EmployeeManager _employeeManager;
        private readonly PasswordManager _passwordManager;
        private readonly CustomerManager _customerManager;
        private readonly AccountManager _accountManager;
        private readonly TransactionManager _transactionManager;
        public EmployeeService(IRequestRepository requestRepository)
        {
            _requestRepository = requestRepository;
            _masterDataService = new MasterDataService(requestRepository);
            _employeeManager = new EmployeeManager(requestRepository);
            _passwordManager = new PasswordManager(requestRepository);
            _customerManager = new CustomerManager(requestRepository);
            _accountManager = new AccountManager(requestRepository);
            _transactionManager = new TransactionManager(requestRepository);
        }
        public GlobalErrors GetEmployeeTransactions(GeneralFilter filter, ref SharedTableResult<TransactionModel> transactions)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _employeeManager.GetEmployeeTransactions(filter, ref transactions);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                transactions = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors moveTransactions(List<TransactionModel> transactions , int fromEmployeeId , int toEmployeeId)
        {
            GlobalErrors result = GlobalErrors.Success;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();

                List<TransactionModel> needBalanceUpdateInvoices = transactions.Where(trans =>
                (trans.CustomerTypeId != (int)CustomerTypes.CreditCustomer && trans.SalesMode != SalesModes.CreditSales) ||
                ((trans.SalesMode == SalesModes.CreditSales && _requestRepository.Configurations.UpdateEmployeeBalanceOnCreditSaleForCashCustomerOnly && trans.CustomerTypeId == (int)CustomerTypes.CashCustomer) ||
                (trans.CustomerTypeId == (int)CustomerTypes.CreditCustomer && _requestRepository.Configurations.UpdateEmployeeBalanceOnCreditSale))).ToList();
                if (needBalanceUpdateInvoices != null && needBalanceUpdateInvoices.Count > 0)
                {
                    int fromEmployeeAccountId = -1;
                    int toEmployeeAccountId = -1;
                    result = _accountManager.GetEmployeeAccountID(fromEmployeeId, ref fromEmployeeAccountId, dbHelper);
                    if(result  == GlobalErrors.Success)
                    {
                        result = _accountManager.GetEmployeeAccountID(toEmployeeId, ref toEmployeeAccountId, dbHelper);
                    }
                    if(result == GlobalErrors.Success)
                    {
                        decimal sumOfRemainingAmount = needBalanceUpdateInvoices.Sum(trans => trans.RemainingAmount);
                        //Update Account balances for source employee (Employee account)
                        result = _accountManager.UpdateAccountBalance(fromEmployeeAccountId, sumOfRemainingAmount * -1, dbHelper, 5);
                        if(result == GlobalErrors.Success)
                        {
                            //Update Account balances for distination employee (Employee account)
                            result = _accountManager.UpdateAccountBalance(toEmployeeAccountId, sumOfRemainingAmount, dbHelper, 5);
                        }
                    }
                }
                if(result == GlobalErrors.Success)
                {
                    result = _transactionManager.MoveInvoiceTransaction(transactions, toEmployeeId, fromEmployeeId, dbHelper);
                }                
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dbHelper.CommitTransaction();
                }
                else
                {
                    dbHelper.RollBackTransaction();
                }
                if (dbHelper != null)
                {
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors CheckEmployeeAccount(int employeeId, ref bool isAccountExist)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.CheckEmployeeAccount(employeeId, ref isAccountExist);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetOrganizationEmployees(string organizationId, ref List<EmployeeModel> employeeList, ref List<EmployeeModel> routeEmployeeList, ref List<EmployeeModel> foEmployeeList)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetOrganizationEmployees(organizationId, ref employeeList, ref routeEmployeeList, ref foEmployeeList);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetCustomerEmployees(int customerId, int outletId, ref List<EmployeeModel> employeeList)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetCustomerEmployees(customerId, outletId, ref employeeList);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors CheckEmployeeAccount(int employeeId, int divisionId, ref bool isAccountExist)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _accountManager.CheckEmployeeAccount(employeeId, divisionId, ref isAccountExist);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors FillOperatorConfigurations()
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.FillEmployeeConfigurations();
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors FillOperatorRelatedData(bool fillEmployeeSuppervise = true, bool fillDivisionAccess = true, bool fillOrganizationAccess = true, int chosenOrg = -1 , bool fromLogin = false)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            List<DivisionModel> divisions = new List<DivisionModel>();
            try
            {
                result = _employeeManager.FillOperatorRelatedStaff();
                if (result == GlobalErrors.Success)
                {
                    if (fillEmployeeSuppervise)
                    {
                        _requestRepository.CurrentOperator.SubordinatesEmployeesAsString = _employeeManager.GetAllSuppervisedEmployees(_requestRepository.CurrentOperator.EmployeeId, null);
                    }
                    if (_requestRepository.Configurations.ForceOrganizationSelectionOnLogin && !fromLogin)
                    {
                        fillOrganizationAccess = false;
                    }
                    if (fillOrganizationAccess)
                    {
                        _requestRepository.CurrentOperator.OrganizationId = _employeeManager.GetEmployeeOrganization(_requestRepository.CurrentOperator.EmployeeId);
                        _requestRepository.CurrentOperator.OrganizationAccess = _employeeManager.GetEmployeeOrganizationAccess(_requestRepository.CurrentOperator.EmployeeId);
                        _requestRepository.CurrentOperator.OrganizationAccess += ",";
                        _requestRepository.CurrentOperator.OrganizationAccess += _requestRepository.CurrentOperator.OrganizationId;
                        _requestRepository.CurrentOperator.OriginalOrganizationAccess = _requestRepository.CurrentOperator.OrganizationAccess;                        
                        if (_requestRepository.Configurations.ForceOrganizationSelectionOnLogin && chosenOrg != -1)
                        {                            
                            _requestRepository.CurrentOperator.OrganizationAccess = chosenOrg.ToString();
                        }
                        CoreDataBaseConstants.ForceOrganizationSelectionOnLogin = _requestRepository.Configurations.ForceOrganizationSelectionOnLogin;
                        CoreDataBaseConstants.CacheCustomerOrganozationQueries();
                    }
                    if (fillDivisionAccess)
                    {
                        result = _employeeManager.GetEmployeeDivisionAccessAsString(_requestRepository.CurrentOperator.EmployeeId, ref divisions);

                        if(divisions != null && divisions.Count > 0)
                        {
                            _requestRepository.CurrentOperator.OriginalDivisionAccess = string.Join(",", divisions.Select(d => d.DivisionId.ToString()));
                            if (_requestRepository.Configurations.GetDivisionsUnderOrganizationAccess)
                            {
                                var organizationIds = _requestRepository.CurrentOperator.OrganizationAccess.Split(',').Select(s => s.Trim()).ToList();
                                _requestRepository.CurrentOperator.DivisionAccess = string.Join(",", divisions.Where(d => organizationIds.Contains(d.OrganizationId.ToString())).Select(d => d.DivisionId.ToString()));
                                if (string.IsNullOrEmpty(_requestRepository.CurrentOperator.DivisionAccess))
                                {
                                    _requestRepository.CurrentOperator.DivisionAccess = "-1";
                                }
                            }
                            else
                            {
                                _requestRepository.CurrentOperator.DivisionAccess = _requestRepository.CurrentOperator.OriginalDivisionAccess;
                            }

                        }
                        else
                        {
                            _requestRepository.CurrentOperator.DivisionAccess = "-1";
                            _requestRepository.CurrentOperator.OriginalDivisionAccess = "-1";

                        }
                                         
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
       
        public GlobalErrors FillMenuData(ref List<MenuModel> allAvailableMenus, ref List<MenuModel> addedMenus, int securityGroupId, FormMode mode)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.CashOperatorMenus(null, ref allAvailableMenus, _requestRepository.CurrentOperator, ref addedMenus, securityGroupId, -1, mode);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetAllEmployees(ref List<EmployeeModel> employeeList)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetAllEmployees(ref employeeList);

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public List<EmployeeModel> GetEmployeeLoadingVehicle(int vehicleId, int organizationId)
        {
            try
            {
                return _employeeManager.GetEmployeeLoadingVehicle(vehicleId, organizationId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }
        public List<EmployeeModel> GetWarehouseStoreKeeper(int warehouseId, int organizationId)
        {
            try
            {
                return _employeeManager.GetWarehouseStoreKeeper(warehouseId, organizationId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }


        public List<EmployeeModel> GetDriverLinkedToTruck(int truckId)
        {
            try
            {
                return _employeeManager.GetDriverLinkedToTruck(truckId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }
        public GlobalErrors GetParentMenus(ref List<MenuModel> menus)
        {
            try
            {
                return _employeeManager.GetParentMenus(ref menus);
            }
            catch (Exception ex)
            {
                return  GlobalErrors.Error;
            }
        }
        public GlobalErrors UpdateOperatorDataFromProfile(OperatorProfileModel operatorModel,ref List<string> errors)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.UpdateOperatorDataFromProfile(operatorModel,ref errors);

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        //public static GlobalErrors UpdateUserProfileImage(string rootPath,IFormFile file)
        //{
        //    var result = GlobalErrors.NotInitialized;
        //    try
        //    {
        //        string folderName = "ClientApp\\src\\assets\\img\\users-img";
        //        string newPath = Path.Combine(rootPath, folderName);
        //        if (!Directory.Exists(newPath))
        //        {
        //            Directory.CreateDirectory(newPath);
        //        }
        //        if (file.Length > 0)
        //        {
        //            string fileName = CoreSession.Instance.CurrentOperator.EmployeeId + ".jpg";// ContentDispositionHeaderValue.Parse(file.ContentDisposition).FileName.Trim('"');
        //            string fullPath = Path.Combine(newPath, fileName);
        //            using (var stream = new FileStream(fullPath, FileMode.Create))
        //            {
        //                file.CopyTo(stream);
        //                result = GlobalErrors.Success;
        //            }
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        return GlobalErrors.Error;
        //    }
        //    return result;
        //}
        public GlobalErrors GetLandingMenu(ref MenuModel menuModel)
        {
            try
            {
                return _employeeManager.GetLandingMenu(ref menuModel,_requestRepository.CurrentOperator.OperatorId);
            }
            catch (Exception ex)
            {
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetEmployeeProfileData(ref EmployeeModel employee, string rootPath)
        {
            try
            {
                return _employeeManager.GetEmployeeProfileData(ref employee, rootPath);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetNotConnectedHelpersOrDrivers(int employeeId, int organizationID, EmployeeTypes employeeType, ref List<EmployeeModel> freeEmployeesList)
        {
            try
            {
                return _employeeManager.GetNotConnectedHelpersOrDrivers(employeeId, organizationID, employeeType, ref freeEmployeesList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                freeEmployeesList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContactCountrys(ref List<ContactCountryModel> ContactCountrysList)
        {
            try
            {
                return _employeeManager.GetContactCountrys(ref ContactCountrysList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                ContactCountrysList = null;
                return GlobalErrors.Error;
            }
        } 
        public GlobalErrors GetChannels(int organizationID, ref List<ChannelModel> freeChannelsList)
        {
            try
            {
                return _employeeManager.GetChannels(organizationID, ref freeChannelsList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                freeChannelsList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetEmployeeBranches(int organizationID, ref List<EmployeeBranchModel> EmployeeBranchesList)
        {
            try
            {
                return _employeeManager.GetEmployeeBranches(organizationID, ref EmployeeBranchesList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                EmployeeBranchesList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetRegions(ref List<RegionModel> freeRegionsList)
        {
            try
            {
                return _employeeManager.GetRegions(ref freeRegionsList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                freeRegionsList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetRelatedEmployees(int employeeId, int organizationId, int employeeTypeId, ref List<EmployeeModel> freeEmployeesList)
        {
            try
            {
                return _employeeManager.GetRelatedEmployees(employeeId, organizationId, employeeTypeId, ref freeEmployeesList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                freeEmployeesList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetEmployeeTypes(ref List<EmployeeTypeModel> employeeTypesList)
        {
            try
            {
                return _employeeManager.GetEmployeeTypes(ref employeeTypesList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                employeeTypesList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SaveEmployee(EmployeeModel employee)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.SaveEmployee(employee);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetEmployeesList(EmployeeFilter employeeFilter, ref SharedTableResult<EmployeeModel> employeeList)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetEmployeesList(employeeFilter, ref employeeList);

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors UpdateEmployeeActiveStatus(int employeeId, bool isActive)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.UpdateEmployeeActiveStatus(employeeId, isActive,null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetEmployeeData(int employeeId, ref EmployeeModel selectedEmployee)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetEmployeeData(employeeId, ref selectedEmployee);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors ValidateEmployeeToSave(EmployeeModel employee, ref bool isValid, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            bool isValidPassword = false;
            errorMessage = string.Empty;
            bool uniqueEmployeeCode = true;
            bool uniqueUserName = true;
            isValid = true;
            try
            {
                string excludedString = string.Empty;

                // Validate Code
                if (!employee.IsNewEmployee)
                {
                    excludedString = string.Format(" And Employee.EmployeeId Not In ({0})",
                    employee.EmployeeId);
                }
                result = _masterDataService.CheckIfUniqueValue("Employee", "EmployeeCode", employee.EmployeeCode, excludedString, ref uniqueEmployeeCode);
                if(result != GlobalErrors.Success || !uniqueEmployeeCode)
                {
                    errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Unique_Code,_requestRepository.LanguageId);
                    isValid = false;
                    return result;
                }
                // Validate User Name
                if (result == GlobalErrors.Success  && !string.IsNullOrEmpty(employee.OperatorName) && !string.IsNullOrEmpty(employee.OperatorPassword) && !string.IsNullOrEmpty(employee.ConfirmPassword))
                {
                    if (!employee.IsNewEmployee)
                    {
                        excludedString = string.Format(" And Operator.OperatorId Not In (Select OperatorID From EmployeeOperator Where EmployeeID = {0})",
                        employee.EmployeeId);
                    }
                    result = _masterDataService.CheckIfUniqueValue("Operator", "OperatorName", employee.OperatorName, excludedString, ref uniqueUserName);
                    if(result != GlobalErrors.Success || !uniqueUserName)
                    {
                        errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Unique_User_Name,_requestRepository.LanguageId);
                        isValid = false;
                        return result;
                    }
                    if (employee.AllowToChangePassword && (employee.IsNewEmployee || (!employee.IsNewEmployee && employee.OperatorPassword != employee.OldPassword)))
                    {
                        // Validate Password
                        isValidPassword = _passwordManager.ValidatePassword(employee.OperatorPassword, employee.OperatorId, employee.EmployeeTypeId, _requestRepository.LanguageId, ref errorMessage);
                        if (!isValidPassword)
                        {
                            isValid = false;
                            return result;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetLandmarkEmployees(int areaId, ref List<EmployeeModel> employees)
        {
            try
            {
                return  _employeeManager.GetLandmarkEmployees(areaId, ref employees);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return  GlobalErrors.Error;
            }
        }
        public GlobalErrors GetEmployeesFilterData(EmployeeFilter filter , ref List<EmployeeModel> employeeList)
        {
            try
            {
                return  _employeeManager.GetEmployeesFilterData(filter , ref employeeList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetCustomerRouteEmployees(EmployeeFilter filter, ref List<EmployeeModel> employeeList)
        {
            try
            {
                return _employeeManager.GetCustomerRouteEmployees(filter, ref employeeList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors ChangePassword(OperatorProfileModel operatorModel, int languageId, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            errorMessage = string.Empty;
            try
            {
                int operatorID = -1;
                result = _employeeManager.ChangePassword(operatorID,operatorModel, languageId, ref errorMessage);
            } 
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public bool AllowChangePassword(int operatorID, ref string errorMessage)
        {
            errorMessage = string.Empty;
            bool allowChangePassword = true;
            try
            {
                allowChangePassword = _employeeManager.AllowChangePassword(operatorID, ref errorMessage);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return allowChangePassword;
        }
        public  GlobalErrors FillOperatorDataOld()
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.FillEmployeeConfigurations();
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveEmployeeChannel(ChannelModel channelObj)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.SaveEmployeeChannel(channelObj);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetAllEmployeeChannels(GeneralFilter filter, ref SharedTableResult<ChannelModel> channels)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                List<TextLanguage> descriptions = new List<TextLanguage>();
                List<OrganizationModel> organizations = new List<OrganizationModel>();
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                result = _employeeManager.GetAllEmployeeChannels(filter, ref channels);
                if (result == GlobalErrors.Success)
                {
                    foreach (ChannelModel channel in channels.Data)
                    {
                        descriptions = new List<TextLanguage>();
                        result = _employeeManager.GetChannelDescriptions(channel.ChannelId, ref descriptions);
                        if (result != GlobalErrors.Success) break;
                        channel.ChannelDescriptionLanguages = descriptions;
                    }
                }
                if (result == GlobalErrors.Success)
                {
                    foreach (ChannelModel channel in channels.Data)
                    {
                        organizations = new List<OrganizationModel>();
                        result = _employeeManager.GetChannelOrganizations(channel.ChannelId, ref organizations);
                        if (result != GlobalErrors.Success) break;
                        channel.OrganizationList = organizations;
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                channels = null;
                return GlobalErrors.Error;
            }
            return result;
        }       
        public GlobalErrors EditEmployeeChannel(ChannelModel channelObj)
        {
            try
            {
                return _employeeManager.EditEmployeeChannel(channelObj);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors DeleteEmployeeChannel(int channelId)
        {
            try
            {
                return _employeeManager.DeleteEmployeeChannel(channelId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors AllowToDeleteChannel(int channelId, ref string response)
        {
            try
            {
                DBHelper<int> dBHelper = new DBHelper<int>();
                if (_employeeManager.IsChannelLinkedToEmployees(channelId, dBHelper))
                {
                    response = ResourcesManager.TranslateKey(MessagesConstants.Desc_Channel_Linked_To_Employee, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return GlobalErrors.Success;
        }
        public List<EmployeeModel> GetAvailableDrivers(int driverId)
        {
            try
            {
                return _employeeManager.GetAvailableDrivers(driverId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }
        public GlobalErrors GetAllInspectors(string organizationIds, ref List<EmployeeModel> inspectorList)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetAllInspectors(organizationIds, ref inspectorList);

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetAllSurveyEmployees(string organizationId, ref List<EmployeeModel> employeeList)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetAllSurveyEmployees(organizationId, ref employeeList);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors FillSecurityGroupsList(string organizationIds,bool isSurvey, ref List<SecurityGroupModel> securityGroups)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.getAllSecurityGroups(organizationIds, isSurvey, ref securityGroups);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }

        }
        public GlobalErrors GetItemsList(ref List<ItemListModel> itemsList)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetItemsList(ref itemsList);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }

        }
        public GlobalErrors CheckIfEmployeeIsHold(int employeeId, ref bool isOnhold)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.CheckIfEmployeeIsOnHold(employeeId, ref isOnhold);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetDevicesList(EmployeeFilter filter,ref SharedTableResult<DeviceModel> devices)
        {
            try
            {
                return _employeeManager.GetDevicesList(filter, ref devices);
            }
            catch(Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetDeviceData(string deviceSerial, ref DeviceModel deviceData)
        {
            try
            {
                return _employeeManager.GetDeviceData(deviceSerial, ref deviceData);
            }
            catch(Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors DeleteDevice(string deviceSerial,ref string errorMessage)
        {
            try
            {
                return _employeeManager.DeleteDevice(deviceSerial,ref errorMessage);
            }
            catch(Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }

        }

        public GlobalErrors ActivateDeactivateDevice(string deviceSerial, bool isActivate)
        {
            try
            {
                return _employeeManager.ActivateDeactivateDevice(deviceSerial, isActivate);
            }
            catch(Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }

        }

        public  bool CheckForInternetConnection()
        {
            try
            {
               return _employeeManager.CheckForInternetConnection();
            }
            catch
            {
                return false;
            }
        }
        public GlobalErrors GetCRMMenus(ref List<CRMMenuModel> teleRepMenus)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = EmployeeManager.GetCRMMenus(null, ref teleRepMenus, _requestRepository.CurrentOperator);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetEmployeeContactCountry(int employeeId,ref int contactCountryId)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetEmployeeContactCountry(employeeId,ref contactCountryId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        #region CHATTING
        public GlobalErrors GetEmployeeIdFromOperatorId(int operatorId, ref int employeeId)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetEmployeeIdFromOperatorId(operatorId, ref employeeId);

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }


        public GlobalErrors CheckIfChattingIsAvailableForOperator(int operatorId, ref bool isAvailable)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try 
            {
                result = _employeeManager.CheckIfChattingIsAvailableForOperator(operatorId, ref isAvailable);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetChattingListForAvailableUsersToAdd(EmployeeFilter filter, ref List<EmployeeModel> chattingList)
        {
            try
            {
                return _employeeManager.GetChattingListForAvailableUsersToAdd(filter, ref chattingList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetChattingListForAvailableSecurityGroupsToAdd(EmployeeFilter filter, ref List<SecurityGroupModel> chattingList)
        {
            try
            {
                return _employeeManager.GetChattingListForAvailableSecurityGroupsToAdd(filter, ref chattingList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetChattingListForAddedBeforeUsers(EmployeeFilter filter, ref List<EmployeeModel> addedChattingList)
        {
            try
            {
                return _employeeManager.GetChattingListForAddedBeforeUsers(filter, ref addedChattingList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetChattingListForAddedBeforeSecurityGroups(EmployeeFilter filter, ref List<SecurityGroupModel> addedBeforeSGChattingList)
        {
            try
            {
                return _employeeManager.GetChattingListForAddedBeforeSecurityGroups(filter, ref addedBeforeSGChattingList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        #endregion

        #region[Employee Brands List]
        public GlobalErrors GetEmployeeBrandsListSharedTable(GeneralFilter filter, ref SharedTableResult<EmployeeBrandsModel> employeeBrandsList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _employeeManager.GetEmployeeBrandsListSharedTable(filter, ref employeeBrandsList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                employeeBrandsList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetEmployeeBrandsListData(int brandListId, ref EmployeeBrandsModel data)
        {
            try
            {
                GlobalErrors result = GlobalErrors.NotInitialized;
                List<EmployeeModel> appliedEmployees = new List<EmployeeModel>();
                List<SecurityGroupModel> appliedSecurityGroups = new List<SecurityGroupModel>();
                List<TextLanguage> descriptions = new List<TextLanguage>();
                List<BrandModel> brands = new List<BrandModel>();
                List<OrganizationModel> orginazations = new List<OrganizationModel>();
                var ItemsId = new List<int>();
                bool allEmployees = false;

                result = _employeeManager.GetBrandsForEmployeeBrandsList(brandListId, ref brands);
                if (result == GlobalErrors.Success)
                    result = _employeeManager.GetBrandsListDescriptions(brandListId, ref descriptions);
                if (result == GlobalErrors.Success)
                    result = _employeeManager.GetBrandsListOrganizations(brandListId, ref orginazations);
                if (result == GlobalErrors.Success)
                   result = _employeeManager.GetBrandListAppliedEmployees(brandListId, ref appliedEmployees);
                 if (result == GlobalErrors.Success)
                   result = _employeeManager.GetBrandListAppliedToAllEmployees(brandListId, ref allEmployees);
                if (result == GlobalErrors.Success)
                    result = _employeeManager.GetBrandListAppliedSecurityGroups(brandListId, ref appliedSecurityGroups);

                if (result == GlobalErrors.Success)
                {
                    data.Brands = brands;
                    data.Employees = appliedEmployees;
                    data.SecurityGroups = appliedSecurityGroups;
                    data.LanguageDescriptions = descriptions;
                    data.AllEmployee = allEmployees;
                    data.OrganizationList = orginazations;
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SaveEmployeeBrandsList(EmployeeBrandsModel brandList)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (brandList.BrandsListId <= 0)
                {
                    result = _employeeManager.InsertEmployeeBrandsList(brandList);
                }
                else
                {
                    result = _employeeManager.UpdateEmployeeBrandsList(brandList);
                }
                if (result == GlobalErrors.SuccessWithZeroRowAffected) result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors DeleteEmployeeBrandsList(int brandListId)
        {
            try
            {
                return _employeeManager.DeleteEmployeeBrandsList(brandListId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors IsBrandValidToAdd(EmployeeBrandsModel brandsModel, ref bool isUsed)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                DBHelper<int> dBHelper = new DBHelper<int>();
                result = _employeeManager.IsBrandValidToAdd(brandsModel, dBHelper, ref isUsed);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        #endregion

        #region [Employee Calendar]
        public GlobalErrors SaveEmployeeCalendarEvents(EmployeeCalendarModel employeeCalendar)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                DateTime selectedDate = new DateTime(employeeCalendar.Year, employeeCalendar.Month, employeeCalendar.Day);
                int currentYear = selectedDate.Year;
                switch (employeeCalendar.RepeatPeriod)
                {
                    case RepeatitionPeriods.Daily:
                        while (selectedDate.Year == currentYear)
                        {
                            employeeCalendar.Year = selectedDate.Year;
                            employeeCalendar.Month = selectedDate.Month;
                            employeeCalendar.Day = selectedDate.Day;
                            result = _employeeManager.InsertEmployeeCalendarEvents(employeeCalendar, null);
                            selectedDate = selectedDate.AddDays(1);
                        }
                        break;
                    case RepeatitionPeriods.Monthly:
                        for (int monthIndex = selectedDate.Month; monthIndex <= 12; monthIndex++)
                        {
                            if (DateTime.DaysInMonth(selectedDate.Year, selectedDate.Month) >= selectedDate.Day)
                            {
                                employeeCalendar.Year = selectedDate.Year;
                                employeeCalendar.Month = monthIndex;
                                employeeCalendar.Day = selectedDate.Day;
                                result = _employeeManager.InsertEmployeeCalendarEvents(employeeCalendar, null);
                            }
                        }
                        break;
                    case RepeatitionPeriods.Weekly:
                        while (selectedDate.Year == currentYear)
                        {
                            employeeCalendar.Year = selectedDate.Year;
                            employeeCalendar.Month = selectedDate.Month;
                            employeeCalendar.Day = selectedDate.Day;
                            result = _employeeManager.InsertEmployeeCalendarEvents(employeeCalendar, null);
                            selectedDate = selectedDate.AddDays(7);
                        }
                        break;
                    case RepeatitionPeriods.Never:
                        employeeCalendar.Year = selectedDate.Year;
                        employeeCalendar.Month = selectedDate.Month;
                        employeeCalendar.Day = selectedDate.Day;
                        result = _employeeManager.InsertEmployeeCalendarEvents(employeeCalendar, null);
                        break;
                }
                result = _employeeManager.InsertEmployeeCalendarEvents(employeeCalendar,null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetEmployeeCalendar(EmployeeCalendarModel calendarDate, ref List<EmployeeCalendarModel> monthEvents)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetEmployeeCalendar(calendarDate);
                if(result == GlobalErrors.Success)
                {
                    result = _employeeManager.GetMonthEvents(calendarDate,ref monthEvents);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors UpdateEmployeeCalendarEvents(EmployeeCalendarModel employeeCalendar)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                 result = _employeeManager.UpdateEmployeeCalendarEvents(employeeCalendar, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors DeleteCalendarEvents(EmployeeCalendarModel calendarEvent)
        {
            try
            {
                return _employeeManager.DeleteCalendarEvents(calendarEvent);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        #endregion

        #region[TeleRep Plan - CRM]

        public GlobalErrors GetTeleRepCustomers(int employeeId, ref List<SimpleCustomerModel> customers)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _employeeManager.GetTeleRepCustomers(employeeId, ref customers);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveTeleRepCustomers(int employeeId, List<SimpleCustomerModel> customers) 
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dBHelper = null;
            try
            {
                dBHelper = new DBHelper<int>();
                dBHelper.BeginTransaction();

                // Temporary ==>
                result = _employeeManager.PrepareTablesForTeleRep(employeeId, dBHelper);
                // Temporary ==> 

                result = _employeeManager.DeleteTeleRepCustomers(employeeId, dBHelper);
                if (result == GlobalErrors.Success && customers != null && customers.Count > 0) 
                {
                    result = _employeeManager.InsertTeleRepCustomers(employeeId,customers, dBHelper);
                }
                if (result == GlobalErrors.Success) 
                {
                    result = _employeeManager.FixFutureCallPlansCustomers(employeeId, dBHelper);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dBHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dBHelper.CommitTransaction();
                    }
                    else
                    {
                        dBHelper.RollBackTransaction();
                    }
                    dBHelper.Dispose();
                    dBHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors GetTeleRepPlanData(int employeeId, DateModel dateModel, ref TeleRepPlanModel plan)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                plan.EmployeeId = employeeId;
                plan.DateModel = dateModel;

                // Temporary ==>
                result = _employeeManager.PrepareTablesForTeleRep(employeeId, null);
                // Temporary ==> 
                
                int callPlanId = -1;
                result = _employeeManager.GetCallPlanId(ref callPlanId, employeeId, dateModel, null);
                if (result == GlobalErrors.Success) 
                {
                    plan.CallPlanId = callPlanId;
                    List<SimpleCustomerModel> callPlanCustomers = new List<SimpleCustomerModel>();
                    if (callPlanId > 0)
                    {
                        result = _employeeManager.GetTeleRepPlanDetails(employeeId, dateModel, callPlanId, ref callPlanCustomers);
                        if(result == GlobalErrors.Success && callPlanCustomers != null && callPlanCustomers.Count > 0) 
                        {
                            plan.Customers = callPlanCustomers;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors SaveTeleRepCallPlan(TeleRepPlanModel plan)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dBHelper = null;
            try
            {
                bool isNewplan = false;
                dBHelper = new DBHelper<int>();
                dBHelper.BeginTransaction();
                // Temporary ==>
                result = _employeeManager.PrepareTablesForTeleRep(plan.EmployeeId, dBHelper);
                // Temporary ==> 
                if (plan.CallPlanId == -1)
                {
                    isNewplan = true;
                    // save header if plan is new
                    result = _employeeManager.SaveTeleRepCallPlanHeader(plan.EmployeeId, plan.DateModel, dBHelper);
                    if (result == GlobalErrors.Success)
                    {
                        int callPlanId = -1;
                        result = _employeeManager.GetCallPlanId(ref callPlanId, plan.EmployeeId, plan.DateModel, dBHelper);
                        if (result != GlobalErrors.Success || callPlanId == -1)
                        {
                            return GlobalErrors.Error;
                        }
                        plan.CallPlanId = callPlanId;
                    }
                }
                else 
                {
                    result = GlobalErrors.Success;
                }
                if (result == GlobalErrors.Success) 
                {
                    // save details
                    result = _employeeManager.DeleteTeleRepPlansDetailsInEdit(plan.EmployeeId, plan.CallPlanId, plan.DateModel, dBHelper);
                    if (result == GlobalErrors.Success && plan.Customers != null && plan.Customers.Count > 0) 
                    {
                        result = _employeeManager.SaveTeleRepCallPlanDetails(plan.EmployeeId, plan.CallPlanId, plan.Customers, dBHelper);
                    }
                    else if (result == GlobalErrors.Success && (plan.Customers == null || plan.Customers.Count == 0) && !isNewplan) 
                    {
                        result = _employeeManager.DeleteExistingCallPlanWithNoCustomers(plan.EmployeeId, plan.CallPlanId,dBHelper);
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dBHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dBHelper.CommitTransaction();
                    }
                    else
                    {
                        dBHelper.RollBackTransaction();
                    }
                    dBHelper.Dispose();
                    dBHelper = null;
                }
            }
            return result;
        }

        public GlobalErrors GetTeleRepVisitStatuses(ref List<TeleRepVisitStatusModel> teleRepVisitStatuses)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try 
            {
                result = _employeeManager.GetTeleRepVisitStatuses(ref teleRepVisitStatuses);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        #endregion
        public GlobalErrors SaveNewGenesysToken(GenesysTokenModel tokenModel)
        {
            try
            {
                return _employeeManager.SaveNewGenesysToken(tokenModel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetEmployeeGenesysToken(ref GenesysTokenModel tokenModel)
        {
            try
            {
                return _employeeManager.GetEmployeeGenesysToken(ref tokenModel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors PrepareTablesForTeleRep(int employeeId, DBHelper<int> dBHelper)
        {
            try
            {
                return _employeeManager.PrepareTablesForTeleRep(employeeId, dBHelper);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
    }
}