﻿using Sonic.Web.Core;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using System;
using System.Collections.Generic;
using System.Reflection;
using System.Linq;
using Sonic.Web.Model.Location;


namespace Sonic.Web.Service
{
    public class LocationService
    {
        private readonly IRequestRepository _requestRepository;

        private readonly LocationManager _locationManager;
        public LocationService(IRequestRepository requestRepository)
        {
            _requestRepository = requestRepository;
            _locationManager = new LocationManager(requestRepository);
        }
        public GlobalErrors GetCountries(ref List<Country> countries)
        {
            try
            {
                return _locationManager.GetCountries(ref countries, false,null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetStates(int countryId, ref List<State> State)
        {
            try
            {
                return _locationManager.GetStates(countryId, ref State, false,null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetCities(int countryId, int stateId, ref List<City> city)
        {
            try
            {
                return _locationManager.GetCities(countryId, stateId, ref city, false,null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetAreas(int cityId, ref List<Area> area)
        {
            return GetAreas(-1, -1, cityId, ref area);
        }

        public GlobalErrors GetAreas(int countryID, int stateID, int cityId, ref List<Area> area)
        {
            try
            {
                return _locationManager.GetAreas(countryID, stateID, cityId, ref area, false,null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetStreets(int countryID, int stateID, int cityId, int areaId, ref List<Street> street)
        {
            try
            {
                return _locationManager.GetStreets(countryID, stateID, cityId, areaId, ref street, false);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetLocatoinLists(ref List<LocationModel> LocationsList)
        {
            try
            {
                LocationsList = new List<LocationModel>();
                return _locationManager.GetLocatoinLists(ref LocationsList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContactCountry(ref List<Country> contactCountryList)
        {
            try
            {
                return _locationManager.GetContactCountry(ref contactCountryList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        #region Locations Hierarchy 
        public GlobalErrors GetGeoLocationHierarchyData(ref dynamic locationsHierarchyData, LocationHierarchyFilter locationHierarchyFilter)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = prepareLocationsData(ref locationsHierarchyData, locationHierarchyFilter);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }

            return result;
        }


        public GlobalErrors prepareLocationsData(ref dynamic locationsHierarchyData, LocationHierarchyFilter locationHierarchyFilter)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                var locationLevels = _requestRepository.Configurations.GeographicalLocationHierarchyLevel;
                var countriesList = new List<Country>();
                var statesList = new List<State>();
                var citiesList = new List<City>();
                var areasList = new List<Area>();
                var streetsList = new List<Street>();


                result = _locationManager.GetCountries(ref countriesList,true,null);
                if (result == GlobalErrors.Success)
                    result = _locationManager.GetStates(-1, ref statesList, true,null);
                if (result == GlobalErrors.Success)
                    result = _locationManager.GetCities(-1, -1, ref citiesList, true,null);
                if (result == GlobalErrors.Success)
                    result = _locationManager.GetAreas(-1, -1, -1, ref areasList, true,null);
                if (result == GlobalErrors.Success)
                    result = _locationManager.GetStreets(-1, -1, -1, -1, ref streetsList, true);

                var countriesIds = countriesList.Select(x => x.CountryId).ToList();
                var statesIds = statesList.Select(x => x.StateId).ToList();
                var citiesIds = citiesList.Select(x => x.CityId).ToList();
                var areasIds = areasList.Select(x => x.AreaId).ToList();

                var hirarchyLevel = locationHierarchyFilter.ParentId.HasValue ? locationHierarchyFilter.locationHierarchyLevel : (GeographicalLocationHierarchyLevel)_requestRepository.Configurations.GeographicalLocationHierarchyLevel;
                switch (hirarchyLevel)
                {
                    case GeographicalLocationHierarchyLevel.Country:
                        if (!string.IsNullOrEmpty(locationHierarchyFilter.SearchFilter))
                        {
                            var streets = PrepareStreetsHirarchyLevel(streetsList, areasIds, citiesIds, statesIds, countriesIds, locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var areas = PrepareAreasHirarchyLevel(areasList, citiesIds, statesIds, countriesIds, streets.Select(v => v.AreaId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var cities = PrepareCitiesHirarchyLevel(citiesList, statesIds, countriesIds, areas.Select(x => x.CityId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var statess = PrepareStatesHirarchyLevel(statesList, countriesIds, cities.Select(x => x.StateId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var countries = PrepareCountriesHirarchyLevel(countriesList, statess.Select(x => x.CountryId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();

                            countries.Select(x => { x.Children = statess.Where(v => v.CountryId == x.CountryId).ToList(); return x; }).ToList();
                            countries.Select(x => x.Children.Select(z => { z.Children = cities.Where(v => v.CountryId == x.CountryId && v.StateId == z.StateId).ToList(); return z; }).ToList()).ToList();
                            countries.Select(q => q.Children.Select(y => y.Children.Select(a => { a.Children = areas.Where(v => v.CountryId == q.CountryId && v.StateId == y.StateId && v.CityId == a.CityId).ToList(); return a; }).ToList()).ToList()).ToList();
                            countries.Select(q => q.Children.Select(y => y.Children.Select(a => a.Children.Select(h => { h.Children = streets.Where(v => v.CountryId == q.CountryId && v.StateId == y.StateId && v.CityId == a.CityId).ToList(); return h; }).ToList()).ToList()).ToList()).ToList();
                            locationsHierarchyData = countries;
                        }
                        else
                        {
                            locationsHierarchyData = PrepareCountriesHirarchyLevel(countriesList, null, string.Empty).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                        }
                        break;
                    case GeographicalLocationHierarchyLevel.State:

                        if (locationHierarchyFilter.ParentId.HasValue)
                        {
                            countriesIds = new List<int>(); countriesIds.Add(locationHierarchyFilter.ParentId.Value);
                        }

                        if (!string.IsNullOrEmpty(locationHierarchyFilter.SearchFilter))
                        {
                            var streets = PrepareStreetsHirarchyLevel(streetsList, areasIds, citiesIds, statesIds, countriesIds, locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var areas = PrepareAreasHirarchyLevel(areasList, citiesIds, statesIds, countriesIds, streets.Select(v => v.AreaId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var cities = PrepareCitiesHirarchyLevel(citiesList, statesIds, countriesIds, areas.Select(x => x.CityId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var statess = PrepareStatesHirarchyLevel(statesList, countriesIds, cities.Select(x => x.StateId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();

                            statess.Select(x => { x.Children = cities.Where(v => v.StateId == x.StateId).ToList(); return x; }).ToList();
                            statess.Select(x => x.Children.Select(z => { z.Children = areas.Where(v => v.StateId == x.StateId && v.CityId == z.CityId).ToList(); return z; }).ToList()).ToList();
                            statess.Select(q => q.Children.Select(y => y.Children.Select(a => { a.Children = streets.Where(v => v.StateId == q.StateId && v.CityId == y.CityId && v.AreaId == a.AreaId).ToList(); return a; }).ToList()).ToList()).ToList();
                            locationsHierarchyData = statess;
                        }
                        else
                        {
                            locationsHierarchyData = PrepareStatesHirarchyLevel(statesList, countriesIds, null, string.Empty).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode);
                        }

                        break;

                    case GeographicalLocationHierarchyLevel.City:
                        if (locationHierarchyFilter.ParentId.HasValue)
                        {
                            statesIds = new List<int>(); statesIds.Add(locationHierarchyFilter.ParentId.Value);
                        }

                        if (!string.IsNullOrEmpty(locationHierarchyFilter.SearchFilter))
                        {
                            var streets = PrepareStreetsHirarchyLevel(streetsList, areasIds, citiesIds, statesIds, countriesIds, locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var areas = PrepareAreasHirarchyLevel(areasList, citiesIds, statesIds, countriesIds, streets.Select(v => v.AreaId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var cities = PrepareCitiesHirarchyLevel(citiesList, statesIds, countriesIds, areas.Select(x => x.CityId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();

                            cities.Select(x => { x.Children = areas.Where(v => v.CityId == x.CityId).ToList(); return x; }).ToList();
                            cities.Select(x => x.Children.Select(s => { s.Children = streets.Where(v => v.CityId == x.CityId && v.AreaId == s.AreaId).ToList(); return s; }).ToList()).ToList();
                            locationsHierarchyData = cities;
                        }
                        else
                        {
                            locationsHierarchyData = PrepareCitiesHirarchyLevel(citiesList, statesIds, countriesIds, null, string.Empty).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode);
                        }
                        break;
                    case GeographicalLocationHierarchyLevel.Area:
                        if (locationHierarchyFilter.ParentId.HasValue)
                        {
                            citiesIds = new List<int>(); citiesIds.Add(locationHierarchyFilter.ParentId.Value);
                        }

                        if (!string.IsNullOrEmpty(locationHierarchyFilter.SearchFilter))
                        {
                            var streets = PrepareStreetsHirarchyLevel(streetsList, areasIds, citiesIds, statesIds, countriesIds, locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();
                            var areas = PrepareAreasHirarchyLevel(areasList, citiesIds, statesIds, countriesIds, streets.Select(v => v.AreaId).ToList(), locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode).ToList();

                            locationsHierarchyData = areas.Select(x => { x.Children = streets.Where(b => b.AreaId == x.AreaId).ToList(); return x; }).ToList();
                        }
                        else
                        {
                            locationsHierarchyData = PrepareAreasHirarchyLevel(areasList, citiesIds, statesIds, countriesIds, null, string.Empty).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode);
                        }
                        break;
                    case GeographicalLocationHierarchyLevel.Street:
                        if (locationHierarchyFilter.ParentId.HasValue)
                        {
                            areasIds = new List<int>(); areasIds.Add(locationHierarchyFilter.ParentId.Value);
                        }
                        locationsHierarchyData = PrepareStreetsHirarchyLevel(streetsList, areasIds, citiesIds, statesIds, countriesIds, locationHierarchyFilter.SearchFilter).OrderBy(x => x.NodeName).ThenBy(x => x.NodeCode);

                        break;
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }

            return result;
        }

        public List<LocationHierarchyModel> PrepareStatesHirarchyLevel(List<State> states, List<int> countryIds, List<int> stateIds, string searchFilter)
        {
            try
            {
                var statesData = (from state in states
                                  where
                                  (countryIds != null && countryIds.Count > 0 ? countryIds.Contains(state.CountryId) : true) &&
                                  ((!string.IsNullOrEmpty(searchFilter) ? ((!string.IsNullOrEmpty(state.StatessName) ? state.StatessName.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)
                                  || (!string.IsNullOrEmpty(state.StateCode) ? state.StateCode.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)) : true)
                                  || (stateIds != null && stateIds.Count > 0 ? stateIds.Contains(state.StateId) : false))
                                  select new LocationHierarchyModel
                                  {
                                      NodeName = state.StateCode + " - " + state.StatessName,
                                      NodeCode = state.StateCode,
                                      NodeId = state.StateId,
                                      CountryId = state.CountryId,
                                      StateId = state.StateId,
                                      IsParent = true,
                                      ParentId = state.CountryId,
                                      GeographicalLocationHierarchyLevel = GeographicalLocationHierarchyLevel.State,
                                      Descriptions = state.Descriptions,
                                      DummyCode = GeographicalLocationHierarchyLevel.State.GetHashCode() + "-" + state.StateId,
                                      Children = new List<LocationHierarchyModel>()
                                  }).ToList();

                return statesData;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }

        public List<LocationHierarchyModel> PrepareCitiesHirarchyLevel(List<City> citiesList, List<int> stateIds, List<int> countryIds, List<int> cityIds, string searchFilter)
        {
            try
            {

                var cities = (from city in citiesList
                              where
                              (countryIds != null && countryIds.Count > 0 ? countryIds.Contains(city.CountryId) : true) &&
                              (stateIds != null && stateIds.Count > 0 ? stateIds.Contains(city.StateId) : true) &&
                              ((!string.IsNullOrEmpty(searchFilter) ? ((!string.IsNullOrEmpty(city.CityName) ? city.CityName.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)
                              || (!string.IsNullOrEmpty(city.CityCode) ? city.CityCode.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)) : true)
                              || (cityIds != null && cityIds.Count > 0 ? cityIds.Contains(city.CityId) : false))
                              select new LocationHierarchyModel
                              {

                                  NodeName = city.CityCode + " - " + city.CityName,
                                  NodeCode = city.CityCode,
                                  NodeId = city.CityId,
                                  CountryId = city.CountryId,
                                  StateId = city.StateId,
                                  CityId = city.CityId,
                                  IsParent = true,
                                  ParentId = city.StateId,
                                  Descriptions = city.Descriptions,
                                  GeographicalLocationHierarchyLevel = GeographicalLocationHierarchyLevel.City,
                                  DummyCode = GeographicalLocationHierarchyLevel.City.GetHashCode() + "-" + city.CityId,
                                  Children = new List<LocationHierarchyModel>()
                              }).ToList();

                return cities;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }
        public List<LocationHierarchyModel> PrepareAreasHirarchyLevel(List<Area> areasList, List<int> cityIds, List<int> stateIds, List<int> countryIds, List<int> areasIds, string searchFilter)
        {
            try
            {

                var areas = (from area in areasList
                             where
                             (stateIds != null && stateIds.Count > 0 ? stateIds.Contains(area.StateId) : true) &&
                             (countryIds != null && countryIds.Count > 0 ? countryIds.Contains(area.CountryId) : true) &&
                             (cityIds != null && cityIds.Count > 0 ? cityIds.Contains(area.CityId) : true) &&
                             ((!string.IsNullOrEmpty(searchFilter) ? ((!string.IsNullOrEmpty(area.AreaName) ? area.AreaName.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)
                             || (!string.IsNullOrEmpty(area.AreaCode) ? area.AreaCode.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)) : true)
                             || (areasIds != null && areasIds.Count > 0 ? areasIds.Contains(area.AreaId) : false))

                             select new LocationHierarchyModel
                             {
                                 NodeName = area.AreaCode + " - " + area.AreaName,
                                 NodeCode = area.AreaCode,
                                 NodeId = area.AreaId,
                                 CountryId = area.CountryId,
                                 StateId = area.StateId,
                                 CityId = area.CityId,
                                 AreaId = area.AreaId,
                                 IsParent = true,
                                 ParentId = area.CityId,
                                 GeographicalLocationHierarchyLevel = GeographicalLocationHierarchyLevel.Area,
                                 Descriptions = area.Descriptions,
                                 DummyCode = GeographicalLocationHierarchyLevel.Area.GetHashCode() + "-" + area.AreaId,
                                 Children = new List<LocationHierarchyModel>()
                             }).ToList();



                return areas;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }

        public List<LocationHierarchyModel> PrepareStreetsHirarchyLevel(List<Street> streetsList, List<int> areasIds, List<int> citiesIds, List<int> statesIds, List<int> countriesIds, string searchFilter)
        {
            try
            {

                var streets = (from street in streetsList
                               where areasIds.Contains(street.AreaId) && statesIds.Contains(street.StateId) && countriesIds.Contains(street.CountryId) && citiesIds.Contains(street.CityId)
                               && (!string.IsNullOrEmpty(searchFilter) ? ((!string.IsNullOrEmpty(street.StreetName) ? street.StreetName.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)
                               || (!string.IsNullOrEmpty(street.StreetCode) ? street.StreetCode.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)) : true)
                               select new LocationHierarchyModel
                               {
                                   NodeName = street.StreetCode + " - " + street.StreetName,
                                   NodeCode = street.StreetCode,
                                   NodeId = street.StreetId,
                                   CountryId = street.CountryId,
                                   StateId = street.StateId,
                                   CityId = street.CityId,
                                   AreaId = street.AreaId,
                                   IsParent = false,
                                   ParentId = street.AreaId,
                                   GeographicalLocationHierarchyLevel = GeographicalLocationHierarchyLevel.Street,
                                   Descriptions = street.Descriptions,
                                   DummyCode = GeographicalLocationHierarchyLevel.Street.GetHashCode() + "-" + street.StreetId
                               }).ToList();


                return streets;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }

        public List<LocationHierarchyModel> PrepareCountriesHirarchyLevel(List<Country> countriesList, List<int> countriesIds, string searchFilter)
        {
            try
            {
                var countries = (from country in countriesList
                                 where ((!string.IsNullOrEmpty(searchFilter) ? ((!string.IsNullOrEmpty(country.CountryName) ? country.CountryName.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)
                                 || (!string.IsNullOrEmpty(country.CountryCode) ? country.CountryCode.ToLowerInvariant().Contains(searchFilter.ToLowerInvariant()) : false)) : true)
                                 || (countriesIds != null && countriesIds.Count > 0 ? countriesIds.Contains(country.CountryId) : false))
                                 select new LocationHierarchyModel
                                 {
                                     NodeName = country.CountryCode + " - " + country.CountryName,
                                     NodeCode = country.CountryCode,
                                     NodeId = country.CountryId,
                                     IsParent = true,
                                     ParentId = -1,
                                     CountryId = country.CountryId,
                                     GeographicalLocationHierarchyLevel = GeographicalLocationHierarchyLevel.Country,
                                     Descriptions = country.Descriptions,
                                     DummyCode = GeographicalLocationHierarchyLevel.Country.GetHashCode() + "-" + country.CountryId,
                                     Children = new List<LocationHierarchyModel>()
                                 }).ToList();

                return countries;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }

        public GlobalErrors SaveUpdateLocationHierarchyData(LocationHierarchyModel locationHierarchyModel, ref LocationHierarchyModel parentNode, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();
                string query = string.Empty;
                int insertedId = -1;
                prepareDummyData(ref locationHierarchyModel, dbHelper);

                switch (locationHierarchyModel.GeographicalLocationHierarchyLevel)
                {
                    case GeographicalLocationHierarchyLevel.Country:
                        result = _locationManager.InsertUpdateCountry(locationHierarchyModel.IsEditMode, locationHierarchyModel.CountryId, locationHierarchyModel.Code, locationHierarchyModel.Descriptions, dbHelper, ref errorMessage, ref insertedId);
                        break;
                    case GeographicalLocationHierarchyLevel.State:
                        result = _locationManager.InsertUpdateState(locationHierarchyModel.IsEditMode, locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.Code, locationHierarchyModel.Descriptions, dbHelper, ref errorMessage, ref insertedId);
                        break;
                    case GeographicalLocationHierarchyLevel.City:
                        result = _locationManager.InsertUpdateCity(locationHierarchyModel.IsEditMode, locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.Code, locationHierarchyModel.Descriptions, dbHelper, ref errorMessage, ref insertedId);
                        break;
                    case GeographicalLocationHierarchyLevel.Area:
                        result = _locationManager.InsertUpdateArea(locationHierarchyModel.IsEditMode, locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, locationHierarchyModel.Code, locationHierarchyModel.Descriptions, dbHelper, ref errorMessage, ref insertedId);
                        break;
                    case GeographicalLocationHierarchyLevel.Street:
                        result = _locationManager.InsertUpdateStreet(locationHierarchyModel.IsEditMode, locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, locationHierarchyModel.StreetId, locationHierarchyModel.Code, locationHierarchyModel.Descriptions, dbHelper, ref errorMessage, ref insertedId);
                        break;
                }
                if (result == GlobalErrors.Success && locationHierarchyModel.IsEditMode)
                {
                    //return parent Node to reopen tree in ui
                   var parentLevel = (locationHierarchyModel.GeographicalLocationHierarchyLevel.GetHashCode() + 1) > GeographicalLocationHierarchyLevel.Country.GetHashCode() ? locationHierarchyModel.GeographicalLocationHierarchyLevel : locationHierarchyModel.GeographicalLocationHierarchyLevel + 1;
                        result = GetParentNode(parentLevel, locationHierarchyModel, ref parentNode, dbHelper);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }

            return result;

        }
        public GlobalErrors DeletLocationHierarchyData(LocationHierarchyModel locationHierarchyModel, ref LocationHierarchyModel parentNode)
        {
            var result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                string query = string.Empty;
                dbHelper.BeginTransaction();
                switch (locationHierarchyModel.GeographicalLocationHierarchyLevel)
                {
                    case GeographicalLocationHierarchyLevel.Country:
                        result = _locationManager.DeleteStreet(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, locationHierarchyModel.StreetId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteArea(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteCity(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteState(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteCountry(locationHierarchyModel.CountryId, dbHelper);
                        break;
                    case GeographicalLocationHierarchyLevel.State:
                        result = _locationManager.DeleteStreet(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, locationHierarchyModel.StreetId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteArea(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteCity(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteState(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, dbHelper, false);
                        break;
                    case GeographicalLocationHierarchyLevel.City:
                        result = _locationManager.DeleteStreet(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, locationHierarchyModel.StreetId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteArea(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteCity(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, dbHelper, false);
                        break;
                    case GeographicalLocationHierarchyLevel.Area:
                        result = _locationManager.DeleteStreet(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, locationHierarchyModel.StreetId, dbHelper, true);
                        if (result == GlobalErrors.Success)
                            result = _locationManager.DeleteArea(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, dbHelper, false);
                        break;
                    case GeographicalLocationHierarchyLevel.Street:
                        result = _locationManager.DeleteStreet(locationHierarchyModel.CountryId, locationHierarchyModel.StateId, locationHierarchyModel.CityId, locationHierarchyModel.AreaId, locationHierarchyModel.StreetId, dbHelper, false);
                        break;
                }

                if (result == GlobalErrors.Success)
                {
                    //return parent Node to reopen tree in ui
                    var parentLevel = locationHierarchyModel.GeographicalLocationHierarchyLevel + 1;
                        result = GetParentNode(parentLevel, locationHierarchyModel, ref parentNode, dbHelper);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors prepareDummyData(ref LocationHierarchyModel locationHierarchyModel, DBHelper<int> dBHelper)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                string errorMessage = string.Empty;
                int countryId = -1;
                int stateId = -1;
                int cityId = -1;
                int areaId = -1;
                switch ((GeographicalLocationHierarchyLevel)_requestRepository.Configurations.GeographicalLocationHierarchyLevel)
                {
                    case GeographicalLocationHierarchyLevel.State:
                        result = fillDummyData("CountryID", "Country", -1, -1, -1, ref countryId, dBHelper, GeographicalLocationHierarchyLevel.Country, ref errorMessage);
                        locationHierarchyModel.CountryId = countryId;
                        break;
                    case GeographicalLocationHierarchyLevel.City:
                        result = fillDummyData("CountryID", "Country", -1, -1, -1, ref countryId, dBHelper, GeographicalLocationHierarchyLevel.Country, ref errorMessage);
                        if (result == GlobalErrors.Success)
                        {
                            locationHierarchyModel.CountryId = countryId;
                            result = fillDummyData("StateID", "State", countryId, -1, -1, ref stateId, dBHelper, GeographicalLocationHierarchyLevel.State, ref errorMessage);
                            if (result == GlobalErrors.Success)
                            {
                                locationHierarchyModel.StateId = stateId;
                            }
                        }

                        break;
                    case GeographicalLocationHierarchyLevel.Area:
                        result = fillDummyData("CountryID", "Country", -1, -1, -1, ref countryId, dBHelper, GeographicalLocationHierarchyLevel.Country, ref errorMessage);
                        if (result == GlobalErrors.Success)
                        {
                            locationHierarchyModel.CountryId = countryId;
                            result = fillDummyData("StateID", "State", countryId, -1, -1, ref stateId, dBHelper, GeographicalLocationHierarchyLevel.State, ref errorMessage);
                            if (result == GlobalErrors.Success)
                            {
                                locationHierarchyModel.StateId = stateId;
                                if (result == GlobalErrors.Success)
                                {
                                    result = fillDummyData("CityID", "City", countryId, stateId, -1, ref cityId, dBHelper, GeographicalLocationHierarchyLevel.City, ref errorMessage);
                                    if (result == GlobalErrors.Success)
                                    {
                                        locationHierarchyModel.CityId = cityId;
                                    }
                                }
                            }
                        }
                        break;
                    case GeographicalLocationHierarchyLevel.Street:
                        result = fillDummyData("CountryID", "Country", -1, -1, -1, ref countryId, dBHelper, GeographicalLocationHierarchyLevel.Country, ref errorMessage);
                        if (result == GlobalErrors.Success)
                        {
                            locationHierarchyModel.CountryId = countryId;
                            result = fillDummyData("StateID", "State", countryId, -1, -1, ref stateId, dBHelper, GeographicalLocationHierarchyLevel.State, ref errorMessage);
                            if (result == GlobalErrors.Success)
                            {
                                locationHierarchyModel.StateId = stateId;
                                if (result == GlobalErrors.Success)
                                {
                                    result = fillDummyData("CityID", "City", countryId, stateId, -1, ref cityId, dBHelper, GeographicalLocationHierarchyLevel.City, ref errorMessage);
                                    if (result == GlobalErrors.Success)
                                    {
                                        locationHierarchyModel.CityId = cityId;
                                        if (result == GlobalErrors.Success)
                                        {
                                            result = fillDummyData("AreaID", "Area", countryId, stateId, cityId, ref areaId, dBHelper, GeographicalLocationHierarchyLevel.Area, ref errorMessage);

                                            if (result == GlobalErrors.Success)
                                            {
                                                locationHierarchyModel.AreaId = areaId;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        break;
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }

            return result;

        }
        public GlobalErrors fillDummyData(string fieldName, string tableName, int countryId, int stateId, int cityId, ref int fieldValue, DBHelper<int> dBHelper, GeographicalLocationHierarchyLevel geographicalLocationHierarchyLevel, ref string errorMessage)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _locationManager.GetDummyLevelData(fieldName, tableName, ref fieldValue, dBHelper);
                if (result == GlobalErrors.Success && fieldValue <= 0)
                {

                    var descriptionList = new List<TextLanguage>();
                    var description = new TextLanguage();
                    description.Description = "Dummy-" + tableName;
                    description.LanguageId = _requestRepository.LanguageId;
                    descriptionList.Add(description);
                    switch (geographicalLocationHierarchyLevel)
                    {
                        case GeographicalLocationHierarchyLevel.Country:
                            result = _locationManager.InsertUpdateCountry(false, -1, "Dummy-Code", descriptionList, dBHelper, ref errorMessage, ref fieldValue);
                            break;
                        case GeographicalLocationHierarchyLevel.State:
                            result = _locationManager.InsertUpdateState(false, countryId, -1, "Dummy-Code", descriptionList, dBHelper, ref errorMessage, ref fieldValue);
                            break;
                        case GeographicalLocationHierarchyLevel.City:
                            result = _locationManager.InsertUpdateCity(false, countryId, stateId, -1, "Dummy-Code", descriptionList, dBHelper, ref errorMessage, ref fieldValue);
                            break;
                        case GeographicalLocationHierarchyLevel.Area:
                            result = _locationManager.InsertUpdateArea(false, countryId, stateId, cityId, -1, "Dummy-Code", descriptionList, dBHelper, ref errorMessage, ref fieldValue);
                            break;
                    }
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetParentNode(GeographicalLocationHierarchyLevel level, LocationHierarchyModel nodeData, ref LocationHierarchyModel parentNode,DBHelper<int> dBHelper)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                var parentId = -1;
                switch (level)
                {
                    case GeographicalLocationHierarchyLevel.Country:
                        var countries = new List<Country>();
                        DBHelper<Country> dbhelperCountry = null;
                        dbhelperCountry = new DBHelper<Country>(dBHelper.GetConnection(), dBHelper.GetDBTransaction());
                        parentId = nodeData.IsEditMode ? nodeData.CountryId : nodeData.ParentId;
                        result = _locationManager.GetCountries(ref countries, true, dbhelperCountry, parentId);
                        if (result == GlobalErrors.Success)
                            parentNode = PrepareCountriesHirarchyLevel(countries, null, string.Empty).FirstOrDefault();
                        break;
                    case GeographicalLocationHierarchyLevel.State:
                        var Statess = new List<State>();
                        DBHelper<State> dbhelperState = null;
                        dbhelperState = new DBHelper<State>(dBHelper.GetConnection(), dBHelper.GetDBTransaction());
                        parentId = nodeData.IsEditMode ? nodeData.StateId : nodeData.ParentId;
                        result = _locationManager.GetStates(-1, ref Statess, true, dbhelperState, parentId);
                        if (result == GlobalErrors.Success)
                            parentNode = PrepareStatesHirarchyLevel(Statess, null, null, string.Empty).FirstOrDefault();
                        break;
                    case GeographicalLocationHierarchyLevel.City:
                        var Cities = new List<City>();
                        DBHelper<City> dbhelperCity = null;
                        dbhelperCity = new DBHelper<City>(dBHelper.GetConnection(), dBHelper.GetDBTransaction());
                        parentId = nodeData.IsEditMode ? nodeData.CityId : nodeData.ParentId;
                        result = _locationManager.GetCities(-1, -1, ref Cities, true, dbhelperCity, parentId);
                        if (result == GlobalErrors.Success)
                            parentNode = PrepareCitiesHirarchyLevel(Cities, null, null, null, string.Empty).FirstOrDefault();
                        break;
                    case GeographicalLocationHierarchyLevel.Area:
                        var areaies = new List<Area>();
                        DBHelper<Area> dbhelperArea = null;
                        dbhelperArea = new DBHelper<Area>(dBHelper.GetConnection(), dBHelper.GetDBTransaction());
                        parentId = nodeData.IsEditMode ? nodeData.AreaId : nodeData.ParentId;
                        result = _locationManager.GetAreas(-1, -1, -1, ref areaies, true, dbhelperArea, parentId);
                        if (result == GlobalErrors.Success)
                            parentNode = PrepareAreasHirarchyLevel(areaies, null, null, null, null, string.Empty).FirstOrDefault();
                        break;
                    default:
                        result = GlobalErrors.Success;
                        break;
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        #endregion

    }


}
