﻿using Microsoft.AspNetCore.SignalR;
using Newtonsoft.Json;
using Sonic.Web.Core;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Resources;
using System;
using System.Linq;
using System.Collections.Generic;
using System.Net;
using System.Net.Http;
using System.Reflection;
using System.Text;

namespace Sonic.Web.Service
{
    public class NotificationService
    {
        private readonly IRequestRepository _requestRepository;
        private readonly NotificationManager _notificationManager;
        private IHubContext<NotifyHub, ITypedHubClient> _hubContext;
        public NotificationService(IHubContext<NotifyHub, ITypedHubClient> hubContext, IRequestRepository requestRepository)
        {
            _hubContext = hubContext;  // hub is null if NotificationService called from echo (Sonic Services Solution)
            _requestRepository = requestRepository;
            _notificationManager = new NotificationManager(requestRepository);
        }
        public GlobalErrors GetEchoNotificationList(int pageNumber, int pageSize, ref List<NotificaitonsModel> notifications)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (pageNumber < 0)
                {
                    pageNumber = 0;
                }
                if (pageSize < 0)
                {
                    pageNumber = 25;
                }
                result = _notificationManager.GetEchoNotificationList(pageNumber, pageSize, ref notifications);
                if(result == GlobalErrors.Success)
                {
                    result = _notificationManager.SetNotificationAsRead();
                }
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }        
        public GlobalErrors GetBadgeCount(ref int badge)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                return _notificationManager.GetBadgeCount(_requestRepository.CurrentEchoOperator.CustomerId, _requestRepository.CurrentEchoOperator.OutletId, ref badge);
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
        public GlobalErrors UpdateCustomerDeviceToken(string token)
        {
            try
            {
                return _notificationManager.UpdateCustomerDeviceToken(token);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }


        #region[Sonic Notifications]
        public GlobalErrors SendNotification(NotificationModel notification, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            bool commitNow = false;
            EmployeeKeyModel keyModel = new EmployeeKeyModel();
            EmployeeKeyFilter employeeKeyFilter = new EmployeeKeyFilter();
            employeeKeyFilter.CustomerId = notification.CustomerId;
            employeeKeyFilter.KeyTypeId = notification.KeyTypeId;
            employeeKeyFilter.EmployeeId = notification.EmployeeId;

            try
            {
                if(dbHelper == null)
                {
                    dbHelper = new DBHelper<int>();
                    dbHelper.BeginTransaction();
                    commitNow = true;
                }
                result = _notificationManager.SaveNotification(notification, dbHelper);
                if (result == GlobalErrors.Success) 
                {
                    result = _notificationManager.SaveNotificationDescriptions(notification, dbHelper);
                }
                if (result == GlobalErrors.Success && notification.NotificationTypeId == NotificationTypes.EmployeeKeys)
                {
                    result = _notificationManager.SaveEmployeeKeysNotification(notification, dbHelper);
                }
                if (result == GlobalErrors.Success && notification.NotificationTypeId == NotificationTypes.DiscountKeys)
                {
                    result = _notificationManager.SaveApprovalRangeHistory(notification, dbHelper);
                }
                if (result == GlobalErrors.Success)
                {
                    if (notification.ReceiversList != null && notification.ReceiversList.Count > 0)
                    {
                        //if (notification.IsBOUser && !notification.ReceiversList.Contains(0))// just for test
                        //{
                        //    notification.ReceiversList.Add(0); 
                        //}
                        result = _notificationManager.SaveNotificationAssignment(notification, dbHelper);
                    }
                    else
                    {
                        return result = GlobalErrors.NoReceiverSelected; // No receivers selected
                    }
                }
                if(result == GlobalErrors.Success)
                {
                    result = _notificationManager.GetFirstKeyFromEmployeeKeysNotification(employeeKeyFilter, ref keyModel);

                    if (keyModel == null) return GlobalErrors.NoMoreRows;
                }
                if (result == GlobalErrors.Success)
                {
                    if (notification.IsBOUser)
                    {
                        if (_hubContext != null)
                        {
                            // Send notification to BO user/users
                            _hubContext.Clients.All.BroadcastMessage(notification);
                        }
                    }
                    else
                    {
                        // Send notification to FO user/users
                        SendNotifiactionToFO(notification);
                    }
                }
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            finally
            {
                if (dbHelper != null && commitNow)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                }
            }
            return result;
        }
        public async void SendNotifiactionToFO(NotificationModel notification)
        {
            try
            {
                if (notification.DeviceTokenList != null && notification.DeviceTokenList.Count > 0 && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                {
                    var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
                    var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);
                    using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
                    {
                        httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
                        httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
                        httpRequest.Content = new StringContent(notification.jsonBody, Encoding.UTF8, "application/json");
                        ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
                        using (var httpClient = new HttpClient())
                        {
                            var res = await httpClient.SendAsync(httpRequest);
                            if (res.IsSuccessStatusCode)
                            {
                            }
                            else
                            {

                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
        }

        public async void SendMessageUsingFirebasePushNotification(EmployeeKeysOnline message)
        {
            try
            {
                if (message.SenderDeviceToken != null && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                {

                    var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
                    var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);

                    // Data to be sent
                    var data = new Object();
                    data = new
                    {
                        to = message.SenderDeviceToken,
                        data = new
                        {
                            type = NotificationTypes.EmployeeKeys.GetHashCode(),
                            EmployeeKeysOnline = message
                        }
                    };
                    var jsonBody = JsonConvert.SerializeObject(data);

                    using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
                    {
                        httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
                        httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
                        httpRequest.Content = new StringContent(jsonBody, Encoding.UTF8, "application/json");
                        ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
                        using (var httpClient = new HttpClient())
                        {
                            var res = await httpClient.SendAsync(httpRequest);
                            if (res.IsSuccessStatusCode)
                            {
                            }
                            else
                            {

                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
        }

        public GlobalErrors SetNotificationReadFlag(SetNotificationAsReadModel notification)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;

            try
            {
                result = _notificationManager.SetNotificationReadFlag(notification);
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
        public GlobalErrors GetEmployeeNotificationsLazyLoading(int currentNotificationCount, ref List<NotificationModel> notificationsList)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _notificationManager.GetEmployeeNotificationsLazyLoading(currentNotificationCount, ref notificationsList);
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
        public GlobalErrors GetEmployeeUnreadNotificationsCount(ref int unreadNotificationsCoun)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _notificationManager.GetEmployeeUnreadNotificationsCount(ref unreadNotificationsCoun);
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
        public GlobalErrors GetNotificationTemplateListSharedTable(NotificationTemplateFilterModel filter, ref SharedTableResult<NotificationTemplateModel> templatesListShared)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _notificationManager.GetNotificationTemplateListSharedTable(filter, ref templatesListShared);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                templatesListShared = null;
                return GlobalErrors.Error;
            }
        } 
        public GlobalErrors IsValidToGetCommCalcNotificationType(ref bool isValid)
        {
            try
            {
                 
                return _notificationManager.IsValidToGetCommCalcNotificationType(ref isValid);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetNotificationTemplateList(int notificationTypeId, ref List<NotificationTemplateModel> templatesList)
        {
            try
            {
                return _notificationManager.GetNotificationTemplateList(notificationTypeId, ref templatesList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                templatesList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetNotificationTemplateData(int templateId, ref List<TextLanguage> descriptionLanguages, ref List<TextLanguage> headerDescriptionLanguages, ref List<TextLanguage> bodyDescriptionLanguages)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _notificationManager.GetNotificationTemplateDescriptions(templateId, CoreDataBaseConstants.QueryColumnsNames.Description, ref descriptionLanguages);
                if (result == GlobalErrors.Success)
                {
                    result = _notificationManager.GetNotificationTemplateDescriptions(templateId, CoreDataBaseConstants.QueryColumnsNames.HeaderDescription, ref headerDescriptionLanguages);
                }
                if (result == GlobalErrors.Success)
                {
                    result = _notificationManager.GetNotificationTemplateDescriptions(templateId, CoreDataBaseConstants.QueryColumnsNames.BodyDescription, ref bodyDescriptionLanguages);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveNotificationTemplate(NotificationTemplateModel template)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();
                if(template.TemplateId == -1)
                {
                    result = _notificationManager.InsertNewNotificationTemplate(template, dbHelper);
                }
                else
                {
                    result = _notificationManager.UpdateNotificationTemplate(template, dbHelper);
                    if (result == GlobalErrors.Success)
                    {
                        result = _notificationManager.DeleteNotificationTemplateDescriptions(template.TemplateId, dbHelper);
                    }
                }
                if(result == GlobalErrors.Success)
                {
                    result = _notificationManager.SaveNotificationTemplateDescriptions(template, dbHelper);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors UpdateNotificationTemplateActiveStatus(int templateId, bool isActive)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _notificationManager.UpdateNotificationTemplateActiveStatus(templateId, isActive);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors ReceiveEmployeeKey(EmployeeKeysOnline message)
        {
            NotificationModel notification = new NotificationModel();
            NotificationTextLanguage descriptionObject = new NotificationTextLanguage();
            List<int> ReceiversList = new List<int>();
            GlobalErrors result = GlobalErrors.NotInitialized;
            string configName = string.Empty;
            try
            {
                result = GetMessageBodyForRequestKey(ref notification, message);
                
                if (result == GlobalErrors.Success || result == GlobalErrors.NotInitialized)
                {
                    //  result = GetEmployeeHasPrivilegeKey(ref ReceiversList, message.KeyTypeID);
                    switch ((KeyTypes)message.KeyTypeID)
                    {
                        case KeyTypes.SellOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForSaleOnHold;
                            break;
                        case KeyTypes.ProcessExchangeOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForProcessExchangeForOnHoldCustomer;
                            break;
                        case KeyTypes.ProcessExchangeCustomerVoilatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForProcessExchangeViolatedPaymentTerm;
                            break;
                        case KeyTypes.AllowAddNewCustomerKey:
                           configName = ConfigurationConstants.AllowAddNewCustomersWithKey;
                            break;
                        case KeyTypes.ProcessExchangeCustomerExceedsLimitKey:
                            configName = ConfigurationConstants.AllowProcessExchangeCustomerExceedsLimitUsingKey;
                            break;
                        case KeyTypes.SellCustomerVoilatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForSaleViolatedPaymentTerm;
                            break;
                        case KeyTypes.AllowSellCustomerExceedsInvoicesNumberKey:
                            configName = ConfigurationConstants.AllowSellCustomerExceedsInvoicesNumberUsingKey;
                            break;
                        case KeyTypes.AllowRejectReturnOrdersKey:
                            configName = ConfigurationConstants.AllowRejectReturnOrdersUsingKey;
                            break;
                        case KeyTypes.AllowRescheduleReturnOrdersKey:
                            configName = ConfigurationConstants.AllowRescheduleReturnOrdersUsingKey;
                            break;
                        case KeyTypes.AllowExchangeForCustomerExceedsExchangeInvoicesNumberKey:
                            configName = ConfigurationConstants.AllowExchangeForCustomerExceedsExchangeInvoicesNumberUsingKey;
                            break;
                        case KeyTypes.SellCustomerExceedsLimitKey:
                            configName = ConfigurationConstants.AllowSellCustomerExceedsLimitUsingKey;
                            break;
                        case KeyTypes.SellCustomerExceedsLimitKeyFromBO:
                            configName = ConfigurationConstants.AllowSellCustomerExceedsLimitUsingKeyFromBO;
                            break;
                        case KeyTypes.AllowEditDiscountWithPasswordOnReturns:
                            configName = ConfigurationConstants.AllowEditDiscountWithPasswordOnReturns;
                            break;
                        case KeyTypes.ApplyKeyToEditHeaderDiscount:
                            configName = ConfigurationConstants.ApplyKeyToEditHeaderDiscount;
                            break;
                        case KeyTypes.EditPriceKey:
                            if (message.EditPriceKeyMode == EditPriceKeyMode.Sales)
                            {
                                configName = ConfigurationConstants.AllowEditPriceWithPasswordOnSales;
                            }
                            else if (message.EditPriceKeyMode == EditPriceKeyMode.Orders)
                            {
                                configName = ConfigurationConstants.AllowEditPriceWithPasswordOnOrders;
                            }
                            break;
                        case KeyTypes.AllowEditPriceToBeOutOfRangeKey:
                            configName = ConfigurationConstants.ApplyKeyToAllowEditPriceToBeOutOfRange;
                            break;
                        case KeyTypes.EditDiscountKey:
                            if (message.EditPriceKeyMode == EditPriceKeyMode.Sales)
                            {
                                configName = ConfigurationConstants.AllowEditDiscountWithPasswordOnSales;
                            }
                            else if (message.EditPriceKeyMode == EditPriceKeyMode.Orders)
                            {
                                configName = ConfigurationConstants.AllowEditDiscountWithPasswordOnOrders;
                            }
                            break;
                        case KeyTypes.AllowReprintinginFoforCustomerTransaction:
                            configName = ConfigurationConstants.AllowReprintinginFoforCustomerTransaction;
                            break;
                        case KeyTypes.AllowReprintinginFoforWareHouseTransaction:
                            configName = ConfigurationConstants.AllowReprintinginFoforWareHouseTransaction;
                            break;
                        case KeyTypes.EditFOCKey:
                            configName = ConfigurationConstants.AllowEditFOCWithPasswordOnSales;
                            break;
                        case KeyTypes.VoidPaymentKey:
                            configName = ConfigurationConstants.ApplyKeyToVoidPayment;
                            break;
                        case KeyTypes.AcceptSalesLessThanReturn:
                            configName = ConfigurationConstants.ApplyKeyToSkipSalesLessThanReturn;
                            break;
                        case KeyTypes.SkipPromotionKey:
                            configName = ConfigurationConstants.ApplyKeyToSkipPromotion;
                            break;
                        case KeyTypes.SkipPromotionIndividuallyKey:
                            configName = ConfigurationConstants.ApplyKeyToSkipPromotionIndividually;
                            break;
                        case KeyTypes.ReturnGoodItemsKey:
                            configName = ConfigurationConstants.ApplyKeyToAllowReturnGoodItems;
                            break;
                        case KeyTypes.VoidInvoiceKey:
                            configName = ConfigurationConstants.AllowVoidInvoiceWithKey;
                            break;
                        case KeyTypes.ExchangeItemsBetweenVansKey:
                            configName = ConfigurationConstants.ApplyKeyToExchangeItemsBetweenVans;
                            break;
                        case KeyTypes.VisitCustomerOutOffRouteKey:
                            configName = ConfigurationConstants.ApplyKeyToVisitTerritoryCustomer;
                            break;
                        case KeyTypes.VisitCustomerWithoutBarcodeScanningKey:
                            configName = ConfigurationConstants.ApplyKeyToVisitCustomerWithoutBarcode;
                            break;
                        case KeyTypes.EndCustomerVisitWithoutBarcodeScanningKey:
                            configName = ConfigurationConstants.ApplyKeyToEndCustomerVisitWithoutBarcode;
                            break;
                        case KeyTypes.ChangeCurrencyExchangeRateForCustomerKey:
                            configName = ConfigurationConstants.SupportMultiCurrency;
                            break;
                        case KeyTypes.ChangeCurrencyExchangeRatePermanentlyKey:
                            configName = ConfigurationConstants.SupportMultiCurrency;
                            break;
                        case KeyTypes.AllowUnlimitedFOCKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowUnlimitedFOC;
                            break;
                        case KeyTypes.ChangeSystemDate:
                            configName = ConfigurationConstants.ApplyKeyToChangeSystemDate;
                            break;
                        case KeyTypes.AllowOrderCustomerExceedsLimitKey:
                            configName = ConfigurationConstants.AllowOrderCustomerExceedsLimitUsingKey;
                            break;
                        case KeyTypes.SkipCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyToSkipCustomer;
                            break;
                        case KeyTypes.AllowExchangeKey:
                            configName = ConfigurationConstants.ApplyKeyToAllowExchange;
                            break;
                        case KeyTypes.OrderToCustomerViolatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForOrderViolatedPaymentTerm;
                            break;
                        case KeyTypes.AllowPartialDeliveryInOrderKey:
                            configName = ConfigurationConstants.AllowPartialDeliveryUsingKeyInOrder;
                            break;
                        case KeyTypes.CreateOrderForOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForOrderOnHoldCustomer;
                            break;
                        case KeyTypes.AllowCustomerExceedContainersLimitKey:
                            configName = ConfigurationConstants.AllowExceedContainersLimitUsingKey;
                            break;
                        case KeyTypes.CreateReturnOrderForCustomerViolatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForReturnOrderViolatedPaymentTerm;
                            break;
                        case KeyTypes.CreateReturnOrderForOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForReturnOrderOnHoldCustomer;
                            break;
                        case KeyTypes.ApplayKeyToEditPriceOnReturn:
                            configName = ConfigurationConstants.ApplyKeyToEditPriceOnReturn;
                            break;
                        case KeyTypes.CreateExchangeOrderForOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForExchangeOrderForOnHoldCustomer;
                            break;
                        case KeyTypes.CreateExchangeOrderForCustomerExceededCreditLimitKey:
                            configName = ConfigurationConstants.ApplyKeysForExchangeOrderOverCreditLimit;
                            break;
                        case KeyTypes.CreateExchangeOrderForCustomerViolatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForExchangeOrderViolatedPaymentTerm;
                            break;
                        case KeyTypes.AllowRejectDeliveryKey:
                            configName = ConfigurationConstants.AllowRejectDeliveryUsingKey;
                            break;
                        case KeyTypes.AllowRescheduleDeliveryKey:
                            configName = ConfigurationConstants.AllowRescheduleDeliveryUsingKey;
                            break;
                        case KeyTypes.ApplyCashDiscountKeyWithPassword:
                            configName = ConfigurationConstants.ApplyCashDiscountWithPassword;
                            break;
                        case KeyTypes.ApplySupervisorDiscountKeyWithPassword:
                            configName = ConfigurationConstants.ApplySupervisorDiscountWithPassword;
                            break;
                        case KeyTypes.AllowSalesModeSelectionForCashCustomers:
                            configName = ConfigurationConstants.AllowSalesModeSelectionForCashCustomers;
                            break;
                        case KeyTypes.ApplyKeyToPrintOriginalInvoice:
                            configName = ConfigurationConstants.ApplyKeyToPrintOriginalInvoice;
                            break;
                        case KeyTypes.ApplyKeyToSkipMustSellItem:
                            configName = ConfigurationConstants.ApplyKeyToSkipMustSellItem;
                            break;
                        case KeyTypes.IncreaseNumOfInvoiceDueDatesKey:
                            configName = ConfigurationConstants.ApplyKeyToIncreaseNumOfInvoiceDueDates;
                            break;
                        case KeyTypes.AllowVisitCustomerOutOfGeoFencingKey:
                            configName = ConfigurationConstants.AllowVisitCustomerOutOFGeoFencingWithKey;
                            break;
                        case KeyTypes.AllowSellingCustomersExceedingBillNumberKey:
                            configName = ConfigurationConstants.AllowSellingCustomersExceedingBillNumberWithKey;
                            break;
                        case KeyTypes.AllowCOSKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowCOS;
                            break;
                        case KeyTypes.AllowFixedIncentiveKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowFixedIncentive;
                            break;
                        case KeyTypes.AllowVariableIncentiveKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowVariableIncentive;
                            break;
                        case KeyTypes.AllowPrisonExpensesFOCKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowPrisonExpenses;
                            break;
                        case KeyTypes.AllowGuaranteesExpensesFOCKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowGuaranteesExpenses;
                            break;
                        case KeyTypes.SellCustomerExceedsCommitmentLimitKey:
                            configName = ConfigurationConstants.AllowSellCustomerExceedsCommitmentLimitUsingKey;
                            break;
                        case KeyTypes.AllowReturnsKey:
                            configName = ConfigurationConstants.AllowReturnsKey;
                            break;
                        case KeyTypes.UnlimitedFOCForOnHoldCustomerKey:
                            configName = ConfigurationConstants.AllowUnlimitedFOCForOnHoldCustomerWithKey;
                            break;
                        case KeyTypes.UnlimitedFOCForCustomerViolatedPaymentTermKey:
                            configName = ConfigurationConstants.CheckPaymentTermViolationForUnlimitedFOCWithKey;
                            break;
                        case KeyTypes.UnlimitedFOCForCustomerExceedsCreditLimitKey:
                            configName = ConfigurationConstants.CheckCreditLimitForUnlimitedFOCWithKey;
                            break;
                        case KeyTypes.UnlimitedFOCForCustomersExceedingBillNumberKey:
                            configName = ConfigurationConstants.AllowUnlimitedFOCWhenCustomerExceedsBillNumberWithKey;
                            break;
                        case KeyTypes.ApplyKeyToEditCollectionDiscountPercentage:
                            configName = ConfigurationConstants.ApplyKeyToEditCollectionDiscountPercentage;
                            break;
                        case KeyTypes.AllowSkipStepKey:
                            configName = ConfigurationConstants.AllowSkipStepWithKey;
                            break;
                        case KeyTypes.AllowSkipCustomerVisitKey:
                            configName = ConfigurationConstants.AllowSkipCustomerVisitKey;
                            break;
                        case KeyTypes.AllowVoidExchangeWithKey:
                            configName = ConfigurationConstants.AllowVoidExchangeWithKey;
                            break;
                        case KeyTypes.AllowSkipSalesRestrictionItem:
                            configName = ConfigurationConstants.ApplyKeySalesRestriction;
                            break;
                        case KeyTypes.AllowEndOfDayBeforeVisitAllCustomers:
                            configName = ConfigurationConstants.ApplyKeyToSkipVisitALLCustomersBeforeEOD;
                            break;
                        case KeyTypes.AllowExchangeWhenReturnIsGreaterThanSalesKey:
                            configName = ConfigurationConstants.ApplyKeyToAllowExchangeWhenReturnIsGreaterThanSales;
                            break;
                        case KeyTypes.AllowExceedsOperatorCreditLimitKey:
                            configName = ConfigurationConstants.AllowExceedsOperatorCreditLimitUsingKey;
                            break;
                        case KeyTypes.AllowCashPaymentsKey:
                            configName = ConfigurationConstants.AllowCashPaymentsUsingKey;
                            break;
                        case KeyTypes.AllowExceedPDCLimitKey:
                            configName = ConfigurationConstants.AllowExceedPDCLimitWithKey;
                            break;
                        case KeyTypes.ApplyKeyForCreditSales:
                            configName = ConfigurationConstants.ApplyKeyForCreditSales;
                            break;
                        case KeyTypes.AllowEditNearExpiryPriceUsingKey:
                            configName = ConfigurationConstants.AllowEditNearExpiryPriceUsingKey;
                            break;
                        case KeyTypes.AllowEditNearExpiryPriceToBeOutOfRangeUsingKey:
                            configName = ConfigurationConstants.AllowEditNearExpiryPriceToBeOutOfRangeUsingKey;
                            break;
                        case KeyTypes.AllowSkipNearExpiryPromotionUsingKey:
                            configName = ConfigurationConstants.AllowSkipNearExpiryPromotionUsingKey;
                            break;
                        case KeyTypes.AllowNearExpiryFOCUsingKey:
                            configName = ConfigurationConstants.AllowNearExpiryFOCUsingKey;
                            break;
                        case KeyTypes.AllowEditNearExpiryDiscountUsingKey:
                            configName = ConfigurationConstants.AllowEditNearExpiryDiscountUsingKey;
                            break;
                        case KeyTypes.AllowUpdateNFCIDUsingKey:
                            configName = ConfigurationConstants.AllowUpdateNFCWithKey;
                            break;
                        case KeyTypes.AllowNFCScanUsingKey:
                            configName = ConfigurationConstants.ForceNFCScan;
                            break;
                    }
                    result = GetEmployeeHasPrivilegeKey(ref ReceiversList, configName , (int)message.KeyTypeID, message.EmployeeID);
                }
                if (result == GlobalErrors.Success)
                {
                    notification.ReceiversList = ReceiversList;
                    var data = new Object();
                    data = new
                    {
                        registration_ids = notification.DeviceTokenList,
                        data = new
                        {
                            type = notification.NotificationTypeId,
                            notification
                        }
                    };

                    string jsonBody = JsonConvert.SerializeObject(data);
                    notification.jsonBody = jsonBody;
                    notification.CustomerId = message.CustomerID;
                    notification.KeyTypeId = (KeyTypes)message.KeyTypeID;
                    notification.EmployeeId = message.EmployeeID;
                    result = SendNotification(notification, null);
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetEmployeeHasPrivilegeKey(ref List<int> ReceiversList, string keyName , int keyTypeID = -1,int EmployeeID = -1)
        {
            try
            {
                return _notificationManager.GetEmployeeHasPrivilegeKey(ref ReceiversList, keyName , keyTypeID, EmployeeID);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetMessageBodyForRequestKey(ref NotificationModel notification, EmployeeKeysOnline message)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            List<NotificationTextLanguage> descriptionList;
            try
            {
                descriptionList = new List<NotificationTextLanguage>();
                notification.NotificationId = message.MessageID;
                notification.IsBOUser = true;
                notification.NotificationTypeId = NotificationTypes.EmployeeKeys;
                notification.EmployeeKeyModel = message;
                notification.BodyDescription = message.MessageBody;
                notification.SenderId = message.EmployeeID;
                notification.DeviceToken = message.SenderDeviceToken;
                if (LocalUtilities.MessageBodyForRequestKeyList == null)
                {
                    LocalUtilities.MessageBodyForRequestKeyList = new List<MessageBodyModel>();
                }
                var keys = LocalUtilities.MessageBodyForRequestKeyList.Where(e => e.KeyTypeID == message.KeyTypeID).ToList();
                if (keys != null && keys.Count > 0)
                {
                    descriptionList = keys.Select(v => new NotificationTextLanguage()
                    {
                        LanguageId = v.LanguageId,
                        BodyDescription = v.BodyDescription,
                        HeaderDescription = "Header"
                    }).ToList();
                }
                if (descriptionList.Count == 0)
                {
                    result = _notificationManager.GetMessageBodyForRequestKey(ref descriptionList, message.KeyTypeID);
                    if (result != GlobalErrors.Success)
                    {
                        return result;
                    }

                    if (descriptionList.Count != 0)
                    {
                        LocalUtilities.MessageBodyForRequestKeyList.AddRange(descriptionList.Select(v => new MessageBodyModel()
                        {
                            LanguageId = v.LanguageId,
                            BodyDescription = v.BodyDescription,
                            KeyTypeID = message.KeyTypeID, 
                        }).ToList());
                    }
                }
                descriptionList.ForEach(a =>
                {
                    a.HeaderDescription = string.Empty;
                    a.BodyDescription = string.Format(ResourcesManager.TranslateKey(MessagesConstants.Desc_Key_Message_Online, a.LanguageId), a.BodyDescription, message.CustomerName);
                });
                notification.DescriptionsList = descriptionList;
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors ReceiveDiscountKey(KeyLimitModel message)
        {
            NotificationModel notification = new NotificationModel();
            NotificationTextLanguage descriptionObject = new NotificationTextLanguage();
            List<KeyLimitModel> customerList = new List<KeyLimitModel>();
            List<TextLanguage> customerLanguage = new List<TextLanguage>();
            TextLanguage CustomerNameLanguage = new TextLanguage();
            List<int> ReceiversList = new List<int>();
            GlobalErrors result = GlobalErrors.NotInitialized;
            string configName = string.Empty;
            try
            {
                result = _notificationManager.GetCustomerDescription(message, ref customerList);
                if (result == GlobalErrors.Success && customerList != null && customerList.Count > 0)
                {
                    foreach (KeyLimitModel keyLimit in customerList)
                    {
                        CustomerNameLanguage = new TextLanguage();
                        CustomerNameLanguage.Description = keyLimit.CustomerDescription;
                        CustomerNameLanguage.LanguageId = keyLimit.LanguageId;
                        customerLanguage.Add(CustomerNameLanguage);
                    }
                    message.CustomerName = customerLanguage;
                }

                if (result == GlobalErrors.Success)
                {
                    result = GetMessageBodyForDiscountKey(ref notification, message);
                }
                if (result == GlobalErrors.Success || result == GlobalErrors.NotInitialized)
                {
                    //  result = GetEmployeeHasPrivilegeKey(ref ReceiversList, message.KeyTypeID);
                    switch ((KeyTypes)message.KeyTypeID)
                    {
                        case KeyTypes.SellOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForSaleOnHold;
                            break;
                        case KeyTypes.ProcessExchangeOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForProcessExchangeForOnHoldCustomer;
                            break;
                        case KeyTypes.ProcessExchangeCustomerVoilatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForProcessExchangeViolatedPaymentTerm;
                            break;
                        case KeyTypes.AllowAddNewCustomerKey:
                            configName = ConfigurationConstants.AllowAddNewCustomersWithKey;
                            break;
                        case KeyTypes.ProcessExchangeCustomerExceedsLimitKey:
                            configName = ConfigurationConstants.AllowProcessExchangeCustomerExceedsLimitUsingKey;
                            break;
                        case KeyTypes.SellCustomerVoilatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForSaleViolatedPaymentTerm;
                            break;
                        case KeyTypes.AllowSellCustomerExceedsInvoicesNumberKey:
                            configName = ConfigurationConstants.AllowSellCustomerExceedsInvoicesNumberUsingKey;
                            break;
                        case KeyTypes.AllowRejectReturnOrdersKey:
                            configName = ConfigurationConstants.AllowRejectReturnOrdersUsingKey;
                            break;
                        case KeyTypes.AllowRescheduleReturnOrdersKey:
                            configName = ConfigurationConstants.AllowRescheduleReturnOrdersUsingKey;
                            break;
                        case KeyTypes.AllowExchangeForCustomerExceedsExchangeInvoicesNumberKey:
                            configName = ConfigurationConstants.AllowExchangeForCustomerExceedsExchangeInvoicesNumberUsingKey;
                            break;
                        case KeyTypes.SellCustomerExceedsLimitKey:
                            configName = ConfigurationConstants.AllowSellCustomerExceedsLimitUsingKey;
                            break;
                        case KeyTypes.SellCustomerExceedsLimitKeyFromBO:
                            configName = ConfigurationConstants.AllowSellCustomerExceedsLimitUsingKeyFromBO;
                            break;
                        case KeyTypes.AllowEditDiscountWithPasswordOnReturns:
                            configName = ConfigurationConstants.AllowEditDiscountWithPasswordOnReturns;
                            break;
                        case KeyTypes.ApplyKeyToEditHeaderDiscount:
                            configName = ConfigurationConstants.ApplyKeyToEditHeaderDiscount;
                            break;
                        case KeyTypes.EditPriceKey:
                            if (message.EditPriceKeyMode == EditPriceKeyMode.Sales)
                            {
                                configName = ConfigurationConstants.AllowEditPriceWithPasswordOnSales;
                            }
                            else if(message.EditPriceKeyMode == EditPriceKeyMode.Orders)
                            {
                                configName = ConfigurationConstants.AllowEditPriceWithPasswordOnOrders;
                            }
                            break;
                        case KeyTypes.AllowEditPriceToBeOutOfRangeKey:
                            configName = ConfigurationConstants.ApplyKeyToAllowEditPriceToBeOutOfRange;
                            break;
                        case KeyTypes.EditDiscountKey:
                            if (message.EditDiscountKeyMode == EditDiscountKeyMode.Sales)
                            {
                                configName = ConfigurationConstants.AllowEditDiscountWithPasswordOnSales;
                            }
                            else if (message.EditDiscountKeyMode == EditDiscountKeyMode.Orders)
                            {
                                configName = ConfigurationConstants.AllowEditDiscountWithPasswordOnOrders;
                            }
                            break;
                        case KeyTypes.AllowReprintinginFoforCustomerTransaction:
                            configName = ConfigurationConstants.AllowReprintinginFoforCustomerTransaction;
                            break;
                        case KeyTypes.AllowReprintinginFoforWareHouseTransaction:
                            configName = ConfigurationConstants.AllowReprintinginFoforWareHouseTransaction;
                            break;
                        case KeyTypes.EditFOCKey:
                            configName = ConfigurationConstants.AllowEditFOCWithPasswordOnSales;
                            break;
                        case KeyTypes.VoidPaymentKey:
                            configName = ConfigurationConstants.ApplyKeyToVoidPayment;
                            break;
                        case KeyTypes.AcceptSalesLessThanReturn:
                            configName = ConfigurationConstants.ApplyKeyToSkipSalesLessThanReturn;
                            break;
                        case KeyTypes.SkipPromotionKey:
                            configName = ConfigurationConstants.ApplyKeyToSkipPromotion;
                            break;
                        case KeyTypes.SkipPromotionIndividuallyKey:
                            configName = ConfigurationConstants.ApplyKeyToSkipPromotionIndividually;
                            break;
                        case KeyTypes.ReturnGoodItemsKey:
                            configName = ConfigurationConstants.ApplyKeyToAllowReturnGoodItems;
                            break;
                        case KeyTypes.VoidInvoiceKey:
                            configName = ConfigurationConstants.AllowVoidInvoiceWithKey;
                            break;
                        case KeyTypes.ExchangeItemsBetweenVansKey:
                            configName = ConfigurationConstants.ApplyKeyToExchangeItemsBetweenVans;
                            break;
                        case KeyTypes.VisitCustomerOutOffRouteKey:
                            configName = ConfigurationConstants.ApplyKeyToVisitTerritoryCustomer;
                            break;
                        case KeyTypes.VisitCustomerWithoutBarcodeScanningKey:
                            configName = ConfigurationConstants.ApplyKeyToVisitCustomerWithoutBarcode;
                            break;
                        case KeyTypes.EndCustomerVisitWithoutBarcodeScanningKey:
                            configName = ConfigurationConstants.ApplyKeyToEndCustomerVisitWithoutBarcode;
                            break;
                        case KeyTypes.ChangeCurrencyExchangeRateForCustomerKey:
                            configName = ConfigurationConstants.SupportMultiCurrency;
                            break;
                        case KeyTypes.ChangeCurrencyExchangeRatePermanentlyKey:
                            configName = ConfigurationConstants.SupportMultiCurrency;
                            break;
                        case KeyTypes.AllowUnlimitedFOCKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowUnlimitedFOC;
                            break;
                        case KeyTypes.ChangeSystemDate:
                            configName = ConfigurationConstants.ApplyKeyToChangeSystemDate;
                            break;
                        case KeyTypes.AllowOrderCustomerExceedsLimitKey:
                            configName = ConfigurationConstants.AllowOrderCustomerExceedsLimitUsingKey;
                            break;
                        case KeyTypes.SkipCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyToSkipCustomer;
                            break;
                        case KeyTypes.AllowExchangeKey:
                            configName = ConfigurationConstants.ApplyKeyToAllowExchange;
                            break;
                        case KeyTypes.OrderToCustomerViolatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForOrderViolatedPaymentTerm;
                            break;
                        case KeyTypes.AllowPartialDeliveryInOrderKey:
                            configName = ConfigurationConstants.AllowPartialDeliveryUsingKeyInOrder;
                            break;
                        case KeyTypes.CreateOrderForOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForOrderOnHoldCustomer;
                            break;
                        case KeyTypes.AllowCustomerExceedContainersLimitKey:
                            configName = ConfigurationConstants.AllowExceedContainersLimitUsingKey;
                            break;
                        case KeyTypes.CreateReturnOrderForCustomerViolatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForReturnOrderViolatedPaymentTerm;
                            break;
                        case KeyTypes.CreateReturnOrderForOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForReturnOrderOnHoldCustomer;
                            break;
                        case KeyTypes.ApplayKeyToEditPriceOnReturn:
                            configName = ConfigurationConstants.ApplyKeyToEditPriceOnReturn;
                            break;
                        case KeyTypes.CreateExchangeOrderForOnHoldCustomerKey:
                            configName = ConfigurationConstants.ApplyKeyForExchangeOrderForOnHoldCustomer;
                            break;
                        case KeyTypes.CreateExchangeOrderForCustomerExceededCreditLimitKey:
                            configName = ConfigurationConstants.ApplyKeysForExchangeOrderOverCreditLimit;
                            break;
                        case KeyTypes.CreateExchangeOrderForCustomerViolatedPaymentTermKey:
                            configName = ConfigurationConstants.ApplyKeysForExchangeOrderViolatedPaymentTerm;
                            break;
                        case KeyTypes.AllowRejectDeliveryKey:
                            configName = ConfigurationConstants.AllowRejectDeliveryUsingKey;
                            break;
                        case KeyTypes.AllowRescheduleDeliveryKey:
                            configName = ConfigurationConstants.AllowRescheduleDeliveryUsingKey;
                            break;
                        case KeyTypes.ApplyCashDiscountKeyWithPassword:
                            configName = ConfigurationConstants.ApplyCashDiscountWithPassword;
                            break;
                        case KeyTypes.ApplySupervisorDiscountKeyWithPassword:
                            configName = ConfigurationConstants.ApplySupervisorDiscountWithPassword;
                            break;
                        case KeyTypes.AllowSalesModeSelectionForCashCustomers:
                            configName = ConfigurationConstants.AllowSalesModeSelectionForCashCustomers;
                            break;
                        case KeyTypes.ApplyKeyToPrintOriginalInvoice:
                            configName = ConfigurationConstants.ApplyKeyToPrintOriginalInvoice;
                            break;
                        case KeyTypes.ApplyKeyToSkipMustSellItem:
                            configName = ConfigurationConstants.ApplyKeyToSkipMustSellItem;
                            break;
                        case KeyTypes.IncreaseNumOfInvoiceDueDatesKey:
                            configName = ConfigurationConstants.ApplyKeyToIncreaseNumOfInvoiceDueDates;
                            break;
                        case KeyTypes.AllowVisitCustomerOutOfGeoFencingKey:
                            configName = ConfigurationConstants.AllowVisitCustomerOutOFGeoFencingWithKey;
                            break;
                        case KeyTypes.AllowSellingCustomersExceedingBillNumberKey:
                            configName = ConfigurationConstants.AllowSellingCustomersExceedingBillNumberWithKey;
                            break;
                        case KeyTypes.AllowCOSKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowCOS;
                            break;
                        case KeyTypes.AllowFixedIncentiveKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowFixedIncentive;
                            break;
                        case KeyTypes.AllowVariableIncentiveKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowVariableIncentive;
                            break;
                        case KeyTypes.AllowPrisonExpensesFOCKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowPrisonExpenses;
                            break;
                        case KeyTypes.AllowGuaranteesExpensesFOCKey:
                            configName = ConfigurationConstants.ApplyKeysToAllowGuaranteesExpenses;
                            break;
                        case KeyTypes.SellCustomerExceedsCommitmentLimitKey:
                            configName = ConfigurationConstants.AllowSellCustomerExceedsCommitmentLimitUsingKey;
                            break;
                        case KeyTypes.AllowReturnsKey:
                            configName = ConfigurationConstants.AllowReturnsKey;
                            break;
                        case KeyTypes.UnlimitedFOCForOnHoldCustomerKey:
                            configName = ConfigurationConstants.AllowUnlimitedFOCForOnHoldCustomerWithKey;
                            break;
                        case KeyTypes.UnlimitedFOCForCustomerViolatedPaymentTermKey:
                            configName = ConfigurationConstants.CheckPaymentTermViolationForUnlimitedFOCWithKey;
                            break;
                        case KeyTypes.UnlimitedFOCForCustomerExceedsCreditLimitKey:
                            configName = ConfigurationConstants.CheckCreditLimitForUnlimitedFOCWithKey;
                            break;
                        case KeyTypes.UnlimitedFOCForCustomersExceedingBillNumberKey:
                            configName = ConfigurationConstants.AllowUnlimitedFOCWhenCustomerExceedsBillNumberWithKey;
                            break;
                        case KeyTypes.ApplyKeyToEditCollectionDiscountPercentage:
                            configName = ConfigurationConstants.ApplyKeyToEditCollectionDiscountPercentage;
                            break;
                        case KeyTypes.AllowSkipStepKey:
                            configName = ConfigurationConstants.AllowSkipStepWithKey;
                            break;
                        case KeyTypes.AllowSkipCustomerVisitKey:
                            configName = ConfigurationConstants.AllowSkipCustomerVisitKey;
                            break;
                        case KeyTypes.AllowVoidExchangeWithKey:
                            configName = ConfigurationConstants.AllowVoidExchangeWithKey;
                            break;
                        case KeyTypes.AllowSkipSalesRestrictionItem:
                            configName = ConfigurationConstants.ApplyKeySalesRestriction;
                            break;
                        case KeyTypes.AllowEndOfDayBeforeVisitAllCustomers:
                            configName = ConfigurationConstants.ApplyKeyToSkipVisitALLCustomersBeforeEOD;
                            break;
                        case KeyTypes.AllowExchangeWhenReturnIsGreaterThanSalesKey:
                            configName = ConfigurationConstants.ApplyKeyToAllowExchangeWhenReturnIsGreaterThanSales;
                            break;
                        case KeyTypes.AllowExceedsOperatorCreditLimitKey:
                            configName = ConfigurationConstants.AllowExceedsOperatorCreditLimitUsingKey;
                            break;
                        case KeyTypes.AllowCashPaymentsKey:
                            configName = ConfigurationConstants.AllowCashPaymentsUsingKey;
                            break;
                        case KeyTypes.AllowExceedPDCLimitKey:
                            configName = ConfigurationConstants.AllowExceedPDCLimitWithKey;
                            break;
                        case KeyTypes.ApplyKeyForCreditSales:
                            configName = ConfigurationConstants.ApplyKeyForCreditSales;
                            break;
                        case KeyTypes.AllowEditNearExpiryPriceUsingKey:
                            configName = ConfigurationConstants.AllowEditNearExpiryPriceUsingKey;
                            break;
                        case KeyTypes.AllowEditNearExpiryPriceToBeOutOfRangeUsingKey:
                            configName = ConfigurationConstants.AllowEditNearExpiryPriceToBeOutOfRangeUsingKey;
                            break;
                        case KeyTypes.AllowSkipNearExpiryPromotionUsingKey:
                            configName = ConfigurationConstants.AllowSkipNearExpiryPromotionUsingKey;
                            break;
                        case KeyTypes.AllowNearExpiryFOCUsingKey:
                            configName = ConfigurationConstants.AllowNearExpiryFOCUsingKey;
                            break;
                        case KeyTypes.AllowEditNearExpiryDiscountUsingKey:
                            configName = ConfigurationConstants.AllowEditNearExpiryDiscountUsingKey;
                            break;
                        case KeyTypes.AllowUpdateNFCIDUsingKey:
                            configName = ConfigurationConstants.AllowUpdateNFCWithKey;
                            break;
                        case KeyTypes.AllowNFCScanUsingKey:
                            configName = ConfigurationConstants.ForceNFCScan;
                            break;
                    }
                    result = GetEmployeeHasPrivilegeKey(ref ReceiversList, configName, message.RequestedBy);
                }
                if (result == GlobalErrors.Success)
                {
                    var data = new Object();
                    data = new
                    {
                        registration_ids = notification.DeviceTokenList,
                        data = new
                        {
                            type = notification.NotificationTypeId,
                            notification
                        }
                    };

                    string jsonBody = JsonConvert.SerializeObject(data);
                    notification.jsonBody = jsonBody;
                    result = SendNotification(notification, null);
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetMessageBodyForDiscountKey(ref NotificationModel notification, KeyLimitModel message)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            List<NotificationTextLanguage> descriptionList;
            try
            {
                descriptionList = new List<NotificationTextLanguage>();
                string body = "";
                string arBody = "";
                notification.NotificationId = message.MessageID;
                notification.IsBOUser = true;
                notification.NotificationTypeId = NotificationTypes.DiscountKeys;
                notification.DiscountKeyModel = message;
                notification.BodyDescription = message.MessageBody;
                notification.SenderId = message.RequestedBy;
                notification.DeviceToken = message.SenderDeviceToken;
                NotificationTextLanguage t1 = new NotificationTextLanguage();
                NotificationTextLanguage t2 = new NotificationTextLanguage();

                if (LocalUtilities.MessageBodyForRequestKeyList == null)
                {
                    LocalUtilities.MessageBodyForRequestKeyList = new List<MessageBodyModel>();
                }
                if (message.TypeId == 1)
                {
                    body = ResourcesManager.TranslateKey(MessagesConstants.Desc_Discount_Key_Online, 1);
                    arBody = ResourcesManager.TranslateKey(MessagesConstants.Desc_Discount_Key_Online, 2);
                    t1.BodyDescription = body;
                    t1.LanguageId = 1;
                    t2.BodyDescription = arBody;
                    t2.LanguageId = 2;
                    descriptionList.Add(t1);
                    descriptionList.Add(t2);
                    descriptionList.ForEach(a =>
                    {
                        a.HeaderDescription = string.Empty;
                        a.BodyDescription = string.Format(ResourcesManager.TranslateKey(MessagesConstants.Desc_Discount_Key_Online, a.LanguageId), message.CustomerName.First(v => v.LanguageId == a.LanguageId).Description);
                    });
                }
                else
                {
                    body = ResourcesManager.TranslateKey(MessagesConstants.Desc_Foc_Key_Online, 1);
                    arBody = ResourcesManager.TranslateKey(MessagesConstants.Desc_Foc_Key_Online, 2);
                    t1.BodyDescription = body;
                    t1.LanguageId = 1;
                    t2.BodyDescription = arBody;
                    t2.LanguageId = 2;
                    descriptionList.Add(t1);
                    descriptionList.Add(t2);
                    descriptionList.ForEach(a =>
                    {
                        a.HeaderDescription = string.Empty;
                        a.BodyDescription = string.Format(ResourcesManager.TranslateKey(MessagesConstants.Desc_Foc_Key_Online, a.LanguageId),message.CustomerName.First(v=> v.LanguageId == a.LanguageId).Description);
                    });
                }
                notification.DescriptionsList = descriptionList;
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public async void SendMessageUsingFirebasePushNotificationDiscountLimit(KeyLimitModel message)
        {
            try
            {
                if (message.SenderDeviceToken != null && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                {

                    var serverKey = string.Format("key={0}", LocalUtilities.SonicFONotificationID);
                    var senderId = string.Format("id={0}", LocalUtilities.SonicFONotificationSenderID);

                    // Data to be sent
                    var data = new Object();
                    data = new
                    {
                        to = message.SenderDeviceToken,
                        data = new
                        {
                            type = NotificationTypes.DiscountKeys.GetHashCode(),
                            DicountKeyOnline = message
                        }
                    };
                    var jsonBody = JsonConvert.SerializeObject(data);

                    using (var httpRequest = new HttpRequestMessage(HttpMethod.Post, "https://fcm.googleapis.com/fcm/send"))
                    {
                        httpRequest.Headers.TryAddWithoutValidation("Authorization", serverKey);
                        httpRequest.Headers.TryAddWithoutValidation("Sender", senderId);
                        httpRequest.Content = new StringContent(jsonBody, Encoding.UTF8, "application/json");
                        ServicePointManager.ServerCertificateValidationCallback = delegate { return true; };
                        using (var httpClient = new HttpClient())
                        {
                            var res = await httpClient.SendAsync(httpRequest);
                            if (res.IsSuccessStatusCode)
                            {
                            }
                            else
                            {

                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
        }
        public GlobalErrors GetEchoNotificationCount(ref int NotificationCount)
        {
            return _notificationManager.GetEchoNotificationCount(ref NotificationCount);
        }
        #endregion
    }
}
