﻿using Newtonsoft.Json;
using Sonic.Web.Core;
using Sonic.Web.Core.Validation;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Model.ApiIntegration.OrderModel;
using Sonic.Web.Models;
using Sonic.Web.Resources;
using Sonic.Web.SecureLibrary;
using Sonic.Web.Services;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using InCube.Security;
using Microsoft.AspNetCore.Http;
using System.Collections.ObjectModel;
using Microsoft.AspNetCore.Mvc;

namespace Sonic.Web.Service
{
    public class OrderService
    {
        private readonly OrderManager _orderManager;
        private readonly EmployeeManager _employeeManager;
        private readonly PaymentManager _paymentManager;
        private readonly TransactionManager _transactionManager;
        private readonly CustomerManager _customerManager;
        private readonly ValidationService _validationService;
        private readonly PromotionManager _promotionManager;
        private readonly AccountManager _accountManager;
        private readonly IRequestRepository _requestRepository;
        private readonly DocumentSequenceManager _documentSequenceManager;
        private readonly DocumentSequenceService _documentSequenceService;
        private readonly ItemService itemService;
        private readonly TransactionManager transactionManager;
        private readonly ValidationService validationService;
        private readonly PromotionService promotionService;
        private readonly NotificationManager _notificationManager;
        private readonly MasterDataService _masterDataService;
        private readonly ValidationManager _validationManager;
        private readonly TransactionService _transactionService;
        private readonly EmailManager _emailManager;
        private readonly EchoManager _echoManager;
        private readonly EmailService _emailService;
        private readonly PaymentService _paymentService;
        private readonly PricesAndOffersManager _pricesAndOffersManager;
        private readonly DynamicReportManager _dynamicReportManager;
        public OrderService(IRequestRepository requestRepository)
        {
            _requestRepository = requestRepository;
            _orderManager = new OrderManager(_requestRepository);
            _employeeManager = new EmployeeManager(_requestRepository);
            _paymentManager = new PaymentManager(_requestRepository);
            _transactionManager = new TransactionManager(_requestRepository);
            _customerManager = new CustomerManager(_requestRepository);
            _validationService = new ValidationService(_requestRepository);
            _promotionManager = new PromotionManager(_requestRepository);
            _accountManager = new AccountManager(_requestRepository);
            _documentSequenceManager = new DocumentSequenceManager(requestRepository);
            _documentSequenceService = new DocumentSequenceService(requestRepository);
            itemService = new ItemService(_requestRepository);
            transactionManager = new TransactionManager(_requestRepository);
            validationService = new ValidationService(_requestRepository);
            promotionService = new PromotionService(_requestRepository);
            _notificationManager = new NotificationManager(_requestRepository);
            _masterDataService = new MasterDataService(_requestRepository);
            _validationManager = new ValidationManager(_requestRepository);
            _transactionService = new TransactionService(_requestRepository);
            _emailManager = new EmailManager(_requestRepository);
            _echoManager= new EchoManager(_requestRepository);
            _emailService = new EmailService(_requestRepository);
            _paymentService = new PaymentService(_requestRepository);
            _pricesAndOffersManager = new PricesAndOffersManager(_requestRepository);
            _dynamicReportManager = new DynamicReportManager (_requestRepository);
        }

        public GlobalErrors InvoiceOrder(OrderModel order, string voidedTransactionID, ref List<SerialCurrentStatusModel> invalidSerials, DBHelper<int> dbHelper, ref string errorMessage, ref bool invalidZATCATransaction, bool isPartialDelivery = false, Collection<IFormFile> attachedFiles = null)
        {
            if (!_orderManager.CheckOrderStatusBeforeAnyAction(order, null))
            {
                var result = GlobalErrors.DataAlreadyExists;
                return result;
            }
            List<PromotionsHistoryModel> oldPromotionsHistoryData = new List<PromotionsHistoryModel>();
            return InvoiceOrder(order, voidedTransactionID, ref invalidSerials, dbHelper, ref errorMessage, oldPromotionsHistoryData, ref invalidZATCATransaction, isPartialDelivery = false, attachedFiles);
        }

        public GlobalErrors InvoiceOrder(OrderModel order, string voidedTransactionID, ref List<SerialCurrentStatusModel> invalidSerials, DBHelper<int> dbHelper, ref string errorMessage, List<PromotionsHistoryModel> oldPromotionsHistoryData, ref bool invalidZATCATransaction, bool isPartialDelivery = false, Collection<IFormFile> attachedFiles = null)
        {
            var result = GlobalErrors.NotInitialized;
            var commitNow = false;
            decimal oldNetTotal = 0;
            string promotionsString = "";
            DateTime currentDate = DateTime.Now;
            List<RelatedToTargetPromotion> relatedToTargetSatisfiedPromotions = new List<RelatedToTargetPromotion>();
            try
            {
                TransactionModel transaction = null;
                bool transactionAlreadyExist = false;
                if (dbHelper == null)
                {
                    dbHelper = new DBHelper<int>();
                    dbHelper.BeginTransaction();
                    commitNow = true;
                }
                oldNetTotal = order.NetTotal;
                //if (order.OrderOperation == OrderOperation.FullDelivery) // if partial delivery , previous status should not be changed (approved or assigned) .
                //    order.PreviousOrderStatus = order.OrderStatus;
                //WarehouseTransactionModel warehouseTransaction = null;
                //result = _orderManager.GetOrderLoadRequest(order, ref warehouseTransaction, dbHelper);
                //if (result != GlobalErrors.Success) return result;
                //if (warehouseTransaction != null)
                //{
                //    order.WarehouseTransactionId = warehouseTransaction.TransactionId;
                //    order.WarehouseId = warehouseTransaction.WarehouseId;
                //    order.ReferenceWarehouseId = warehouseTransaction.RefWarehouseId;
                //}

                //if (order.OrderOperation == OrderOperation.FullDelivery && warehouseTransaction != null)
                //{
                //    //order.AllItems= _orderManager.GetOrderDetails(order, dbHelper);

                //    if (result != GlobalErrors.Success) return result;
                //    transactionAlreadyExist = true;
                //    result = _orderManager.GetInvoiceOrderID(order, dbHelper, ref transaction);
                //    if (result != GlobalErrors.Success) return result;
                //    result = _transactionManager.GetTransactionData(ref transaction, dbHelper);
                //    if (result != GlobalErrors.Success) return result;
                //    order.AllItems = transaction.AllItems;
                //    result = _orderManager.FillSerialsOfPacksOfOrderFromLoadRequest(order, dbHelper);
                //    transaction.AllItems = order.AllItems;
                //    if (result == GlobalErrors.Success)
                //    {
                //        bool allowBatchesInItems = !_requestRepository.Configurations.UseDefaultExpiryAndBatchForItems && _requestRepository.Configurations.AllowBatchSelectionInDelivery;
                //        if (allowBatchesInItems)
                //        {
                //            order.AllItems = _warehouseStockService.PreparesSavedItemsAsSelectedBatchesToGet(order.AllItems);
                //        }
                //    }
                //}
                bool isInvoiceOrderFromStorekeeper = (order.OrderStatus == OrderStatus.Prepared) ? true : false;
                if (!transactionAlreadyExist)
                {
                    transaction = PrepareTransaction(order, dbHelper, currentDate, isPartialDelivery, false, isInvoiceOrderFromStorekeeper);
                    if (transaction == null) return GlobalErrors.Error;
                }
                else
                {
                    result = _transactionManager.DeleteTransactionDetails(transaction, dbHelper);
                }
                if (!string.IsNullOrEmpty(order.WarehouseTransactionId) && order.WarehouseTransactionId != "-1")
                {
                    transaction.DeliverFromVehicle = true;
                    transaction.RefWarehouseId = order.ReferenceWarehouseId;
                    transaction.WarehouseId = order.WarehouseId;

                }

                if (order.OrderOperation == OrderOperation.PartialDelivery)
                {
                    transaction.CreationReason = TransactionCreationReason.PartialDeliveryInBO;
                    transaction.Notes = "Partial Deliver from " + voidedTransactionID;
                }

                transaction.AppliedAmount = transaction.NetTotal;
                result = SaveOrderInvoice(transaction, transactionAlreadyExist, order, ref invalidSerials, isInvoiceOrderFromStorekeeper, dbHelper, ref errorMessage);
                if (result == GlobalErrors.Success && order.OrderOperation != OrderOperation.Invoicing && !isInvoiceOrderFromStorekeeper)
                {
                    // create payment
                    if (_requestRepository.Configurations.CloseCashCustomerOrderInvoice
                        && transaction.SalesMode == SalesModes.CashSales
                        && result == GlobalErrors.Success)
                    {
                        SavePaymentModel savePaymentModel = new SavePaymentModel();
                        savePaymentModel.CustomerId = transaction.CustomerId;
                        savePaymentModel.OutletId = transaction.OutletId;
                        savePaymentModel.DivisionId = transaction.DivisionId;
                        savePaymentModel.EmployeeId = _requestRepository.CurrentOperator.EmployeeId;
                        PaymentModel payment = new PaymentModel();
                        payment.CustomerId = transaction.CustomerId;
                        payment.OutletId = transaction.OutletId;
                        payment.DivisionId = transaction.DivisionId;
                        payment.EmployeeId = transaction.EmployeeId;
                        payment.PaymentDate = DateTime.Now;
                        payment.PaymentTypeId = PaymentTypes.Cash;
                        payment.AppliedAmount = transaction.NetTotal;
                        savePaymentModel.PaymentsList.Add(payment);
                        savePaymentModel.TransactionsList.Add(transaction);
                        result = _paymentManager.SavePayments(savePaymentModel, dbHelper, ref errorMessage);
                    }
                }

                // Update Order status.
                if (result == GlobalErrors.Success)
                {
                    if (!isInvoiceOrderFromStorekeeper && (order.OrderOperation == OrderOperation.PartialDelivery || order.OrderOperation == OrderOperation.FullDelivery))
                    {
                        order.OrderStatus = OrderStatus.Delivered;
                        result = _orderManager.UpdateOrderAndAssignmentOnDelivery(order, !string.IsNullOrEmpty(order.WarehouseTransactionId), dbHelper);
                        if (result == GlobalErrors.Success)
                        {
                            result = _transactionManager.SetTransactionAsPosted(transaction.TransactionId, transaction.CustomerId, transaction.OutletId, dbHelper);
                        }

                    }
                    else
                    {
                        order.PreviousOrderStatus = (OrderStatus)order.OrderStatusId;
                        order.OrderStatus = OrderStatus.Invoiced;
                        result = _orderManager.UpdateOrderStatus(order, dbHelper);
                    }
                }
                // Saving in promotionBenefitHistory table
                if (result == GlobalErrors.Success)
                {
                    if (_requestRepository.Configurations.AllowPromotion)
                    {
                        if (transaction.CustomerPromotions != null && transaction.CustomerPromotions.Count > 0 &&
                            _requestRepository.Configurations.PromotionCalculationSequence == PromotionCalculationSequence.SequentialLevelsFlow.GetHashCode() && order.PromotionsForAllLevels != null && order.PromotionsForAllLevels.Count > 0)
                        {
                            transaction.CustomerPromotions = order.PromotionsForAllLevels;
                        }

                        if ((order.OrderOperation != OrderOperation.PartialDelivery && !order.IsThereAnyProductDiscountDefined) || (order.OrderOperation == OrderOperation.PartialDelivery && !_requestRepository.Configurations.RecalculatePromotionInDelivery))
                            result = _promotionManager.SavePromotionBenefitHistoryData(transaction, order, dbHelper); // to save promotions from order 

                        if (result == GlobalErrors.Success && transaction.CustomerPromotions != null && transaction.CustomerPromotions.Count > 0)
                        {
                            if (_promotionManager.SavePromotionBenefitHistoryData(CoreDataBaseConstants.TableNames.PromotionBenefitHistory, transaction.TransactionId, transaction.CustomerId, transaction.OutletId, transaction.DivisionId, transaction.EmployeeId, transaction.CustomerPromotions, PromotionBenefitTransactionTypes.SalesInvoice, currentDate, dbHelper) == GlobalErrors.Success) // to save achievement promotions
                            {
                                if (_promotionManager.SavePromotionsAgainstAchievementsData(transaction, order, oldPromotionsHistoryData, voidedTransactionID, oldNetTotal, dbHelper) != GlobalErrors.Success)
                                {
                                    result = GlobalErrors.Error;
                                    //Services.Instance.MsgGenerator.ShowMessage(MessagesText.FailedToSavePromotionAchievements);
                                }
                            }
                            else
                            {
                                result = GlobalErrors.Error;
                            }

                        }
                        if (result == GlobalErrors.Success)
                        {
                            var list = transaction.CustomerPromotions.Where(x => x.IsSatisfied && x.IsLoyaltyPromotion).ToList();
                            if (result == GlobalErrors.Success && list.Count > 0)
                            {
                                result = _promotionManager.SaveLoyality(list, transaction.CustomerId, transaction.OutletId, dbHelper);
                            }
                        }
                    }
                }
                if (result == GlobalErrors.Success && _requestRepository.Configurations.AllowCustomerAppLogin)
                {
                 _notificationManager.SendOrderUpdatedNotificationToCustomerApp(order.CustomerId, order.OutletId, order.EmployeeId, order.OrderId, order.OrderStatus, dbHelper);
                }
                if (result == GlobalErrors.Success)
                {
                    //List<AchievementTargetCustomer> neededTargets = new List<AchievementTargetCustomer>();
                    //result =_promotionManager.GetCustomerDefinedTargets(ref neededTargets,ref targetsString, transaction , dbHelper);
                    //if (result == GlobalErrors.Success)
                    //    result = _promotionManager.RunCalculateAchievementProcedure(transaction, false, dbHelper);
                    //if (result == GlobalErrors.Success)
                    //{
                    //    result = _promotionManager.RunCalculateSatisfiedPromotionsProcedure(false, transaction.CustomerId, transaction.OutletId, promotionsString, 0, 0, 0, false, "", false, ref relatedToTargetSatisfiedPromotions, -1, dbHelper);
                    //}
                    //if (result == GlobalErrors.Success)
                    //    result = _promotionManager.DeleteTempBenefitSequenceDetails();
                }
                if (result == GlobalErrors.Success && order.ContractedFOCs != null && order.ContractedFOCs.Count > 0)
                {
                    foreach (AppliedContractedFOCModel foc in order.ContractedFOCs)
                    {
                        result = _orderManager.UpdateContractedFOCDetails(transaction.CustomerId, transaction.OutletId, foc, dbHelper);
                        if (result != GlobalErrors.Success) break;
                        result = _orderManager.InsertIntoContractedFOCHistory(transaction.CustomerId, transaction.OutletId, transaction.TransactionId, transaction.TransactionDate, foc, dbHelper);
                        if (result != GlobalErrors.Success) break;
                    }
                    if (result == GlobalErrors.Success && order.ContractedFOCs.Any(x => x.FOCimagesSequence != null && x.FOCimagesSequence.Count > 0))
                    {
                        result = _orderManager.SaveFOCImages(order, transaction.TransactionId, dbHelper);
                        if (result == GlobalErrors.Success && attachedFiles != null && attachedFiles.Count > 0)
                        {
                            result = HandleFOCImages(order, transaction.TransactionId, attachedFiles);
                        }
                    }
                }
                #region[ZATCA]
                if (result == GlobalErrors.Success && _requestRepository.Configurations.LinkETransactionsWithReturns && commitNow)
                {
                    if (dbHelper != null)
                    {
                        if (result == GlobalErrors.Success || result == GlobalErrors.SuccessWithZeroRowAffected)
                        {
                            dbHelper.CommitTransaction();
                            GlobalErrors archivingResult = _transactionManager.ArchiveZATCATransactions(transaction); // If archiving fails, void the transaction.
                            if (_requestRepository.Configurations.ForceSendTransactionToZATCA && archivingResult != GlobalErrors.Success)
                            {
                                invalidZATCATransaction = true;
                                errorMessage = "Error while archiving ZATCA Transaction";
                                result = GlobalErrors.Error;
                                string voidedMessage = string.Empty;
                                order.OrderStatusFromUI = OrderStatus.Delivered;
                                GlobalErrors voidingResult = VoidOrderInvoiceAndUpdateOrder(order, ref voidedMessage); // TODO -> WHAT IF VOID FAILS ?
                                if (voidingResult == GlobalErrors.Success) errorMessage += ", the transaction '" + transaction.TransactionId + "' was voided automatically";
                                else errorMessage += ", an error occured while voiding the transaction: '" + transaction.TransactionId + "'";
                            }
                        }
                        else
                        {
                            dbHelper.RollBackTransaction();
                        }
                        dbHelper.Dispose();
                        dbHelper = null;
                    }
                }
                #endregion
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (commitNow && dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        if (result == GlobalErrors.NotEnoughQuantity)
                        {
                            errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Not_Enough_Stock_Order, _requestRepository.LanguageId);
                        }
                        dbHelper.RollBackTransaction();
                    }
                }
            }
            return result;
        }

        public GlobalErrors SaveOrderInvoice(TransactionModel transaction, bool saveDetailsOnly, OrderModel order, ref List<SerialCurrentStatusModel> invalidSerials, bool isInvoiceOrderFromStorekeeper, DBHelper<int> dbHelper, ref string errorMessage)

        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                if (!saveDetailsOnly)
                {
                    // Save Transaction Header
                    result = _transactionManager.SaveTransactionHeader(transaction, dbHelper);
                    if (result != GlobalErrors.Success) return result;
                }

                OrderOperation orderOperation = order.OrderOperation;
                // Save Transaction Details
                if ((orderOperation == OrderOperation.Invoicing && !_requestRepository.Configurations.AffectVanStockInDeliveryInFO && _requestRepository.Configurations.DeliveryMode.Contains(((int)DeliveryModes.BOInvoicingFODelivery).ToString()) && _requestRepository.Configurations.AffectMainWarehouseUponOrderBOInvoicing)
                    || (orderOperation != OrderOperation.Invoicing && _requestRepository.Configurations.DeliveryMode.Contains(((int)DeliveryModes.BOInvoicingAndDelivery).ToString()) && _requestRepository.Configurations.AffectMainWarehouseUponOrderBOInvoicing)
                    || (isInvoiceOrderFromStorekeeper))
                {
                    Dictionary<int, List<ItemBatch>> itemsBatches = null;
                    Dictionary<int, decimal> itemsAvlQtyInPcsList = null;

                    int trasactionSequence = 0;
                    //if (result == GlobalErrors.Success)
                    //{
                    //    result = _warehouseStockManager.GetMaxTransactionSequence(ref trasactionSequence, dbHelper);
                    //    if (result != GlobalErrors.Success)
                    //        return result;
                    //}
                    bool isThereItemInStock = true;
                    bool isThereItemNotInStock = false;
                    if (isInvoiceOrderFromStorekeeper)
                    {
                        isThereItemInStock = transaction.AllItems.FindIndex(i => i.InStock) > -1;
                        isThereItemNotInStock = transaction.AllItems.FindIndex(i => !i.InStock) > -1;
                    }
                    if (isThereItemInStock && result == GlobalErrors.Success)
                    {
                        //result = _warehouseStockManager.BuildItemsBatches(transaction, dbHelper, ref itemsBatches, ref itemsAvlQtyInPcsList);
                        //#region [Stock Availability]
                        //if (result == GlobalErrors.Success)
                        //{
                        //    if (isInvoiceOrderFromStorekeeper)
                        //    {
                        //        List<ItemPackModel> inStockList = transaction.AllItems.FindAll(x => x.InStock).ToList();
                        //        result = _warehouseStockManager.CheckCustomerTransactionStockAvailability(itemsAvlQtyInPcsList, inStockList, ref errorMessage);
                        //    }
                        //    else
                        //    {
                        //        result = _warehouseStockManager.CheckCustomerTransactionStockAvailability(itemsAvlQtyInPcsList, transaction.AllItems, ref errorMessage);
                        //    }
                        //}
                        //#endregion
                    }

                    //if (result == GlobalErrors.Success && _requestRepository.Configurations.EnableGeneratingSSCC)
                    //{
                    //    List<string> barcodes = new List<string>();
                    //    result = _orderManager.GetSSCCBarcodesOfOrder(transaction.SourceTransactionId, ref barcodes, dbHelper);
                    //    transaction.linkedSSCCBarcodes = string.Join(",", barcodes);
                    //    if (result == GlobalErrors.Success && isInvoiceOrderFromStorekeeper && transaction.linkedSSCCBarcodes != string.Empty)
                    //    {
                    //        result = _orderManager.InvoiceSSCCHeader(transaction.SourceTransactionId, transaction.TransactionId, SSCCStatuses.InvoiceToCustomer, dbHelper);
                    //        if (result == GlobalErrors.Success)
                    //        {
                    //            foreach (string ssccbarcode in barcodes)
                    //            {
                    //                result = _serialsManager.SaveInSSCCTransTracking(transaction.TransactionId, transaction.RefWarehouseId, transaction.WarehouseId, transaction.EmployeeId
                    //                    , -1, false, TransactionType.Sales.GetHashCode(), ssccbarcode, SSCCStatuses.InvoiceToCustomer.GetHashCode(), transaction.CustomerId, transaction.OutletId, dbHelper);
                    //                if (result != GlobalErrors.Success) break;
                    //            }
                    //        }
                    //    }
                    //}
                    if (order.OrderStatus == OrderStatus.Prepared)
                    {
                        _orderManager.PreparedOrderGetBatchAndSerials(order, ref transaction, dbHelper);
                    }
                    if (result == GlobalErrors.Success)
                    {
                        result = _transactionManager.GetDefaultPriceList(transaction, dbHelper);
                    }
                    if (isInvoiceOrderFromStorekeeper && isThereItemInStock && isThereItemNotInStock)
                    {
                        List<ItemPackModel> allItems = transaction.AllItems;
                        // In Stock
                        //transaction.AllItems = transaction.AllItems.FindAll(x => x.InStock).ToList();
                        //if (result == GlobalErrors.Success)
                        //{
                        //    result = _transactionManager.SavePacksDetails(transaction, trasactionSequence, ref itemsBatches, ref invalidSerials, dbHelper, isInvoiceOrderFromStorekeeper);
                        //}
                        //if (result == GlobalErrors.Success)
                        //    result = _warehouseStockManager.ArchivingWarehouseStockHistory(transaction.TransactionId, transaction.DivisionId, transaction.EmployeeId, dbHelper);
                        //if (result == GlobalErrors.Success)
                        //    result = _warehouseStockManager.DeleteWarehouseStockHistory(transaction.TransactionId, transaction.DivisionId, transaction.EmployeeId, transaction.RouteHistoryId, dbHelper);
                        //if (result == GlobalErrors.Success)
                        //{
                        //    result = _warehouseStockManager.DeleteZeroBatchesFromStock(transaction.WarehouseId, dbHelper);
                        //}

                        // Not in Stock
                        //transaction.WarehouseId = -1;
                        if (result == GlobalErrors.Success)
                        {
                            AppliedBatch appliedBatch = null;
                            decimal price = 0;
                            transaction.AllItems = allItems.FindAll(x => !x.InStock).ToList();
                            foreach (ItemPackModel pack in transaction.AllItems)
                            {
                                price = pack.Price;
                                if (pack.OrderDetailsBatches != null && pack.OrderDetailsBatches.Count > 0)
                                {
                                    foreach (ItemStockBatches selectedBatch in pack.OrderDetailsBatches)
                                    {
                                        appliedBatch = new AppliedBatch();
                                        appliedBatch.BatchNumber = selectedBatch.BatchNo;
                                        appliedBatch.ExpiryDate = selectedBatch.ExpiryDate;
                                        appliedBatch.SalesTransactionTypeId = selectedBatch.SalesTransactionTypeId;
                                        if (selectedBatch.SalesTransactionTypeId == SalesTransactionTypes.Promotion.GetHashCode() || selectedBatch.SalesTransactionTypeId == SalesTransactionTypes.FOC.GetHashCode() || selectedBatch.SalesTransactionTypeId == SalesTransactionTypes.FOCReturn.GetHashCode())
                                        {
                                            pack.Price = 0;
                                        }
                                        else
                                        {
                                            pack.Price = price;
                                        }
                                        pack.RequiredQty = selectedBatch.RequiredQty;
                                        if (result == GlobalErrors.Success)
                                        {
                                            result = transactionManager.SaveTransactionDetailsOnBatchLevel(transaction, pack, dbHelper, appliedBatch);
                                            if (result != GlobalErrors.Success)
                                                return result;
                                        }

                                    }
                                }
                                else
                                {
                                    result = _transactionManager.SaveTransactionDetails(transaction, false, dbHelper);
                                }
                            }
                            //if (result == GlobalErrors.Success)
                            //{
                            //    result = transactionManager.SaveSerialsToDb(transaction, ref invalidSerials, dbHelper, isInvoiceOrderFromStorekeeper);
                            //    if (result != GlobalErrors.Success)
                            //        return result;
                            //}
                        }
                        transaction.AllItems = allItems;
                    }
                    else
                    {
                        if (isThereItemInStock)
                        {
                            //if (result == GlobalErrors.Success)
                            //{
                            //    result = _transactionManager.SavePacksDetails(transaction, trasactionSequence, ref itemsBatches, ref invalidSerials, dbHelper, isInvoiceOrderFromStorekeeper);
                            //}
                            //if (result == GlobalErrors.Success)
                            //    result = _warehouseStockManager.ArchivingWarehouseStockHistory(transaction.TransactionId, transaction.DivisionId, transaction.EmployeeId, dbHelper);
                            //if (result == GlobalErrors.Success)
                            //    result = _warehouseStockManager.DeleteWarehouseStockHistory(transaction.TransactionId, transaction.DivisionId, transaction.EmployeeId, transaction.RouteHistoryId, dbHelper);
                            //if (result == GlobalErrors.Success)
                            //{
                            //    result = _warehouseStockManager.DeleteZeroBatchesFromStock(transaction.WarehouseId, dbHelper);
                            //}
                        }
                        else
                        {
                            transaction.WarehouseId = -1;
                            if (result == GlobalErrors.Success)
                            {
                                AppliedBatch appliedBatch = null;
                                foreach (ItemPackModel pack in transaction.AllItems)
                                {
                                    if (pack.OrderDetailsBatches != null && pack.OrderDetailsBatches.Count > 0)
                                    {
                                        foreach (ItemStockBatches selectedBatch in pack.OrderDetailsBatches)
                                        {
                                            appliedBatch = new AppliedBatch();
                                            appliedBatch.BatchNumber = selectedBatch.BatchNo;
                                            appliedBatch.ExpiryDate = selectedBatch.ExpiryDate;
                                            appliedBatch.SalesTransactionTypeId = selectedBatch.SalesTransactionTypeId;
                                            pack.RequiredQty = selectedBatch.RequiredQty;
                                            if (result == GlobalErrors.Success)
                                            {
                                                result = transactionManager.SaveTransactionDetailsOnBatchLevel(transaction, pack, dbHelper, appliedBatch);
                                                if (result != GlobalErrors.Success)
                                                    return result;
                                            }

                                        }
                                    }
                                    else
                                    {
                                        result = _transactionManager.SaveTransactionDetails(transaction, false, dbHelper);
                                    }
                                }
                                //if (result == GlobalErrors.Success)
                                //{
                                //    result = transactionManager.SaveSerialsToDb(transaction, ref invalidSerials, dbHelper, isInvoiceOrderFromStorekeeper);
                                //    if (result != GlobalErrors.Success)
                                //        return result;
                                //}
                            }
                        }
                    }

                }
                else
                {
                    int packSequence = 0;
                    AppliedBatch appliedBatch = new AppliedBatch();
                    foreach (ItemPackModel pack in transaction.AllItems)
                    {
                        if (pack.RequiredQty > 0)
                        {
                            packSequence++;
                            pack.Sequence = packSequence;
                            appliedBatch.ExpiryDate = LocalUtilities.DefaultExpiryDate;
                            appliedBatch.BatchNumber = LocalUtilities.DefaultBatchNo;
                            appliedBatch.PackId = pack.PackId;
                            appliedBatch.QuantityChange = pack.RequiredQty;
                            appliedBatch.Price = pack.Price;
                            appliedBatch.Tax = pack.CalculatedTax;
                            result = _transactionManager.GetDefaultPriceList(transaction, dbHelper);
                            if (result == GlobalErrors.Success)
                                result = _transactionManager.SaveTransactionDetailsOnBatchLevel(transaction, pack, dbHelper, appliedBatch);
                            if (result != GlobalErrors.Success)
                            {
                                return GlobalErrors.Error;
                            }
                        }
                    }
                }

                // update account balances
                //if (result == GlobalErrors.Success && !saveDetailsOnly && !_requestRepository.Configurations.AllowInvoiceOrderBeforeAssignOrder && orderOperation != OrderOperation.Invoicing && !isInvoiceOrderFromStorekeeper)
                //{
                //    result = _accountManager.UpdateAccountsAfterNewTransaction(dbHelper, transaction, ref errorMessage);
                //}

                // Update Max transaction Id
                if (result == GlobalErrors.Success && !saveDetailsOnly)
                {
                    result = _documentSequenceManager.UpdateMaxTransactionID(dbHelper, DocumentTypes.OrderInvoice, transaction.TransactionId, transaction.EmployeeId, transaction.DivisionId);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }


        public GlobalErrors VoidOrderInvoiceAndUpdateOrder(OrderModel order, ref string message)
        {
            var result = GlobalErrors.Success;
            DBHelper<int> dBHelper = null;
            try
            {
                string voidedTransactionID = string.Empty;

                dBHelper = new DBHelper<int>();
                dBHelper.BeginTransaction();
                if (!_orderManager.CheckOrderStatusBeforeAnyAction(order, dBHelper))
                {
                    result = GlobalErrors.DataAlreadyExists;
                    return result;
                }

                result = _orderManager.VoidOrderTransaction(order, false, ref voidedTransactionID, dBHelper, ref message);
                if (result == GlobalErrors.Success)
                {

                    OrderStatus orderStatus = OrderStatus.Assigned;
                    if (order.PreviousOrderStatus != OrderStatus.None)
                    {
                        orderStatus = order.PreviousOrderStatus;
                    }
                    else if (_orderManager.IsOrderConnectWithLoadReq(order.OrderId, dBHelper))
                    {
                        orderStatus = OrderStatus.LoadRequestAccepted;
                    }
                    else if (_requestRepository.Configurations.AllowAutoDispatchOrderForCustomer(order.CustomerId, order.OutletId))
                    {
                        orderStatus = OrderStatus.New;

                    }
                    else if (_requestRepository.Configurations.AllowInvoiceOrderBeforeAssignOrder)
                    {
                        orderStatus = OrderStatus.Approved;
                    }


                    if (result == GlobalErrors.Success)
                    {
                        result = _orderManager.UpdateOrderStatusesWithActualQty(order, orderStatus, DeliveryStatus.Closed, dBHelper);

                        if (result == GlobalErrors.Success && !_requestRepository.Configurations.AllowInvoiceOrderBeforeAssignOrder)
                        {
                            if (result == GlobalErrors.Success && (orderStatus == OrderStatus.New || orderStatus == OrderStatus.Approved))
                            {
                                result = _orderManager.RemoveOrderAssignemnt(order, dBHelper);
                            }
                            else
                            {
                                result = _orderManager.UpdateDeliveryStatusForOrder(order, DeliveryStatus.Assigned, dBHelper);
                            }
                        }

                    }
                    #region[Contracted FOC - Shelve Incentive]
                    if (result == GlobalErrors.Success)
                    {
                        result = _orderManager.RestoreVoidedTransactionShelveIncentive(order.CustomerId, order.OutletId, voidedTransactionID, dBHelper);
                    }
                    #endregion
                    #region[Validate Voided Transaction]
                    if (result == GlobalErrors.Success)
                    {
                        // voidedTransactionID
                        result = _orderManager.ValidateVoidedTransactionStatus(voidedTransactionID, order.CustomerId, order.OutletId, dBHelper);
                        if (result == GlobalErrors.Success)
                        {
                            result = _orderManager.ValidateVoidedTransactionPaymentStatus(voidedTransactionID, order.CustomerId, order.OutletId, dBHelper);
                        }
                    }
                    #endregion
                }


            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (dBHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dBHelper.CommitTransaction();
                    }
                    else
                    {
                        dBHelper.RollBackTransaction();
                    }
                }

            }
            return result;
        }


        public GlobalErrors IsOrderValidToInvoice(ref OrderModel order, ref string message)
        {
            var result = GlobalErrors.Success;
            try
            {
                result = _orderManager.GetLastOrderStatus(ref order, null);

                if (_requestRepository.Configurations.AllowInvoiceOrderBeforeAssignOrder)
                {
                    if ((order.OrderOperation != OrderOperation.PartialDelivery && order.OrderStatus == OrderStatus.Assigned) || (order.OrderOperation != OrderOperation.PartialDelivery && order.OrderStatus != OrderStatus.Approved
                       && order.OrderStatus != OrderStatus.LoadRequestAccepted))
                    {
                        message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId);
                        return GlobalErrors.Error;
                    }
                }
                else
                {
                    if (_requestRepository.Configurations.EnableOrderPreparationCycle && order.OrderOperation == OrderOperation.Invoicing)
                    {
                        if (order.OrderStatus != OrderStatus.Prepared)
                        {
                            message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId);
                            return GlobalErrors.Error;
                        }
                    }
                    else if ((order.OrderOperation == OrderOperation.PartialDelivery && order.OrderStatus != OrderStatus.Delivered && order.OrderStatus != OrderStatus.Invoiced)
                        || (order.OrderOperation != OrderOperation.PartialDelivery && order.OrderStatus != OrderStatus.Assigned
                  && order.OrderStatus != OrderStatus.LoadRequestAccepted && order.OrderStatus != OrderStatus.Invoiced && order.OrderStatus != OrderStatus.Approved)
                  )
                    {
                        message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId);
                        return GlobalErrors.Error;
                    }
                }



                order.TransactionId = _documentSequenceService.GetMaxDocumentSequence(_requestRepository.Configurations.AllowInvoiceOrderBeforeAssignOrder ? _requestRepository.CurrentOperator.EmployeeId : order.EmployeeId, DocumentTypes.OrderInvoice, order.DivisionId);

                if (order.TransactionId == string.Empty)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Document_Sequence, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                if (order.OrderOperation == OrderOperation.Invoicing && order.OrderStatus == OrderStatus.Assigned && _requestRepository.Configurations.ForceLoadRequestForOrders && _requestRepository.Configurations.AffectVanStockInDeliveryInFO)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Must_Send_Load_Req, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                // No Employee Account
                bool isEmpAccountExist = false;
                //result = _employeeManager.CheckEmployeeAccount(order.EmployeeId, ref isEmpAccountExist);
                //if (!isEmpAccountExist)
                //{
                //    message = ResourcesManager.TranslateKey(MessagesConstants.DescNoAccountForEmployeeToCreateOrderInvoice, _requestRepository.LanguageId);
                //    return GlobalErrors.Error;
                //}

                // Fill Selected outlet Data
                order.Outlet = _customerManager.FillOutletData(order.CustomerId, order.OutletId, order.DivisionId);
                TransactionCommonData transactionCommonData = FillTransactionCommonData(order);

                //// No outlet Account
                //if (order.Outlet.Account == null)
                //{
                //    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Account_For_Outlet, _requestRepository.LanguageId);
                //    return GlobalErrors.Error;
                //}
                if (order.Outlet.Inactive)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Cannot_Deliver_For_Inactive_Customer, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                // Is Violate Payment Terms
                if (_validationService.IsCustomerViolatePaymentTerms(transactionCommonData))
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Customer_Violated_Payment_Term, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }

                // Is Exceeds Customer Limit
                if (_validationService.IsExceedsCustomerLimit(transactionCommonData))
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Exceed_Credit_Limit, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }

                //IsExceedsCustomerBillToBillLimit
                if (_customerManager.IsExceedsCustomerBillToBillLimit(transactionCommonData.CustomerId, transactionCommonData.OutletId, transactionCommonData.DivisionId))
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.MessageErrorCustomerHasUnpaidtransactions, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }

                // Check quantity in warehouse stock
             
                if (order.OrderOperation == OrderOperation.FullDelivery)
                {
                    order.OrderHasSerialItems = _orderManager.CheckIfOrderHasSerialItems(order.OrderId);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }


        public GlobalErrors CheckAndAssignOrders(List<OrderModel> orders, ref string message)
        {
            var result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();

                result = AssignOrdersToDriver(orders, dbHelper, ref message);
                if (result == GlobalErrors.NotEnoughQuantity)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Not_Enough_Stock_Order, _requestRepository.LanguageId);
                }

                if (result == GlobalErrors.Success)
                {
                    dbHelper.CommitTransaction();
                }
                else
                {
                    dbHelper.RollBackTransaction();
                }

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors ReassignOrders(List<OrderModel> orders , ref string messageResult)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;            
            try
            {
                result = _orderManager.ReassignOrders(orders ,ref messageResult);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }            
            return result;
        }

        public GlobalErrors UpdateWarehouseOrder(OrderModel order)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.EditWarehouseOrder(order);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors CheckAndAppoveOrders(OrderModel order, ref string message)
        {
            var result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();

                if (order.OrderTypeId == OrderTypes.Return)
                {
                    result = ApproveReturnOrder(order, ref message, dbHelper);
                }
                else
                {
                    #region[Approval Cycle For Orders]
                    bool applyApprovalCycleOnOrders = _requestRepository.Configurations.ApplyApprovalCycleInWeb.Contains(((int)ApprovalCycleOperations.Orders).ToString());
                    if (applyApprovalCycleOnOrders && order.ApprovalStatus > -1)
                    {
                        result = SaveOrderApprovalCycle(order, dbHelper);
                    }
                    else 
                    {
                        result = GlobalErrors.Success;
                    }
                    #endregion

                    if((result == GlobalErrors.Success &&  order.ApprovalStatus == -1 && applyApprovalCycleOnOrders) || !applyApprovalCycleOnOrders)
                        result = ApproveOrder(order, ref message, dbHelper);
                    
                }

                if (result == GlobalErrors.Success)
                {
                    dbHelper.CommitTransaction();
                }
                else
                {
                    dbHelper.RollBackTransaction();
                }

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        private GlobalErrors SaveOrderApprovalCycle(OrderModel order, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                int actualNewLevel = -1;
                int newStatus = -1;
                int approvalOperationTypeId = (int)ApprovalCycleOperations.Orders;
                string organizationFilter = string.Empty;
                string restrictApprovalCycleToOrganizationLevel = _requestRepository.Configurations.RestrictApprovalCycleToOrganizationLevel;
                if (!string.IsNullOrEmpty(restrictApprovalCycleToOrganizationLevel) && restrictApprovalCycleToOrganizationLevel.Contains(((int)ApprovalCycleOperations.Orders).ToString()))
                {
                    if (order.OrganizationId > -1) 
                    {
                        organizationFilter = $@" AND SelectedOrganizationID = {order.OrganizationId}";
                        result = GlobalErrors.Success;
                    }
                }
                else
                {
                    result = GlobalErrors.Success;
                }
                if (result == GlobalErrors.Success)
                {
                    result = _pricesAndOffersManager.ApproveOperation(-1, approvalOperationTypeId, ref actualNewLevel, ref newStatus, organizationFilter, dbHelper);
                }
                if (result == GlobalErrors.Success)
                {
                    result = _orderManager.UpdateOrderApprovalStatus(order, newStatus, dbHelper);

                    if (result == GlobalErrors.Success)
                    {
                        int approvalHistoryId = -1;
                        result = _pricesAndOffersManager.SaveInApprovalCycleHistory(-1 , order.ApprovalStatus, approvalOperationTypeId, newStatus, ref approvalHistoryId, dbHelper);
                        if (result == GlobalErrors.Success) 
                        {
                            result = _pricesAndOffersManager.SaveInOrderApprovalCycleHistory(approvalHistoryId , order.OrderId, order.CustomerId, order.OutletId, order.DivisionId, dbHelper);
                        }
                        order.ApprovalStatus = newStatus;
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors PrepareOrdersItems(OrderModel order, ref List<ItemPackModel> AllItems)
        {
            var result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();
                if (!_orderManager.CheckOrderStatusBeforeAnyAction(order, dbHelper))
                {
                    result = GlobalErrors.DataAlreadyExists;
                    return result;
                }
                order.AllItems = _orderManager.GetOrderItemsToLoad(order, dbHelper);


                if (result == GlobalErrors.Success)
                {
                    order.PreviousOrderStatus = order.OrderStatus;
                    if (_requestRepository.Configurations.AutoExecuteLoadRequestInBO && !_requestRepository.Configurations.AllowBatchSelectionInWarehouseOperations)
                        order.OrderStatus = OrderStatus.LoadRequestAccepted;
                    else
                        order.OrderStatus = OrderStatus.WaitForLoad;

                    result = _orderManager.UpdateOrderAfterWarehouseTransaction(order, dbHelper);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                }
            }
            return result;
        }
        public GlobalErrors SendOrderLoadRequest(List<OrderModel> orders,string orderIDs, DBHelper<int> dbHelper)
        {
            var result = GlobalErrors.NotInitialized;
            int divisionID = -1;
            string warehouseTransactionId = string.Empty;
            int warehouseID = -1;
            int refWarehouseID = -1;
            int driverID = -1;
            OrderStatus status = OrderStatus.None;
            List<ItemPackModel> allItems = new List<ItemPackModel>();
            bool committNow = false;
            try
            {
                if (dbHelper == null)
                {
                    dbHelper = new DBHelper<int>();
                    dbHelper.BeginTransaction();
                    committNow = true;
                }
                foreach (OrderModel order in orders)
                {
                    divisionID = order.DivisionId;
                    warehouseTransactionId = order.WarehouseTransactionId;
                    warehouseID = order.vehicleId; 
                    refWarehouseID = order.ReferenceWarehouseId;
                    if (order.OrderStatus == OrderStatus.Approved)
                        driverID = order.DriverId;
                    else
                        driverID = order.EmployeeId;

                    if (!_orderManager.CheckOrderStatusBeforeAnyAction(order, dbHelper))
                    {
                        result = GlobalErrors.DataAlreadyExists;
                        break;
                    }
                    if (result == GlobalErrors.DataAlreadyExists)
                        break;
                    if (_requestRepository.Configurations.AutoExecuteLoadRequestInBO && !_requestRepository.Configurations.AllowBatchSelectionInWarehouseOperations)
                        order.OrderStatus = OrderStatus.LoadRequestAccepted;
                    else
                        order.OrderStatus = OrderStatus.WaitForLoad;
                    status = order.OrderStatus;
                }
                if (result == GlobalErrors.DataAlreadyExists)
                    return result;

                allItems = _orderManager.GetOrderItems(orderIDs, divisionID, dbHelper);

                if (result == GlobalErrors.Success)
                {
                    result = _orderManager.UpdateOrdersAfterWarehouseTransaction(orderIDs, status, warehouseTransactionId,divisionID, dbHelper);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (committNow && dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                }
            }
            return result;
        }
        public GlobalErrors AssignOrdersToDriver(List<OrderModel> orders, DBHelper<int> dbHelper, ref string errorMesage)
        {
            var result = GlobalErrors.NotInitialized;
            int helperId = -1, driverId = -1, supervisorId = -1, salesRepId = -1, salesManagerId = -1;
            try
            {
                _employeeManager.GetEmployeeRelatedStaff(orders.FirstOrDefault().DriverId, dbHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepId, ref salesManagerId);

                foreach (var order in orders)
                {

                    order.HelperId = helperId;
                    order.SupervisorId = supervisorId;
                    order.SalesRepId = salesRepId;
                    order.SalesManagerId = _employeeManager.GetSupervisorSalesManager(order.DriverId, dbHelper);
                    order.OrganizationId = _employeeManager.GetEmployeeOrganization(order.DriverId, dbHelper);
                    order.PreviousOrderStatus = order.OrderStatus;
                    order.OrderStatusFromUI = order.OrderStatus;
                    result = _orderManager.InsertIntoDeliveryAssignment(order, dbHelper);
                    if (!_requestRepository.Configurations.AllowAutoDispatchOrderForCustomer(order.CustomerId,order.OutletId))
                    {
                        if (!_orderManager.CheckOrderStatusBeforeAnyAction(order, dbHelper))
                        {
                            result = GlobalErrors.DataAlreadyExists;
                            return result;
                        }
                    }
                    if (result == GlobalErrors.Success && _requestRepository.Configurations.EnableSalesDistributionChannel)
                    {
                        // check if driver territory's sales distribution channels include order's sales distributed channel
                        bool isOrderSDCWithinTerritorySDCs = false;
                        result = _orderManager.CheckTerritorySDCWithOrderSDC(order.DriverId, order.SDCId, ref isOrderSDCWithinTerritorySDCs, dbHelper);
                        if (result == GlobalErrors.Success)
                        {
                            if (!isOrderSDCWithinTerritorySDCs)
                            {
                                errorMesage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Cannot_Dispatch_Order_Out_Of_Territory_SDCS, _requestRepository.LanguageId).Replace("{0}",order.OrderId);
                                result = GlobalErrors.Error;
                            }
                        }
                    }
                    if (result == GlobalErrors.Success)
                    {
                        if (_requestRepository.Configurations.AffectVanStockInDeliveryInFO && _requestRepository.Configurations.ForceLoadRequestForOrders && (_requestRepository.Configurations.DeliveryMode.Contains(((int)DeliveryModes.FOInvoicingAndDelivery).ToString()) || _requestRepository.Configurations.DeliveryMode.Contains(((int)DeliveryModes.BOInvoicingFODelivery).ToString()) || _requestRepository.Configurations.DeliveryMode.Contains(((int)DeliveryModes.BOInvoicingThirdPartyDelivery).ToString())) && order.OrderTypeId != OrderTypes.Return)
                        {
                            if (!_requestRepository.Configurations.AggregateOrdersToOneLoadRequest)
                            {
                                //order.vehicleId = _warehouseManager.GetVehicleConnectedToEmployee(order.DriverId, dbHelper);
                                //order.ReferenceWarehouseId = _warehouseManager.GetVehicleLoadingWH(order.vehicleId);
                                order.AllItems = _orderManager.GetOrderItemsToLoad(order, dbHelper);
                                if (_requestRepository.Configurations.ApplyDivisionsOnDocumentSequence)
                                    order.WarehouseTransactionId = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, order.DriverId, DocumentTypes.WarhouseTransaction, order.DivisionId);
                                else
                                    order.WarehouseTransactionId = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, order.DriverId, DocumentTypes.WarhouseTransaction, -1);

                                if (string.IsNullOrEmpty(order.WarehouseTransactionId)) return GlobalErrors.Error;


                            }
                        }
                        else
                        {
                            order.OrderStatus = OrderStatus.Assigned;

                            result = _orderManager.UpdateOrderStatus(order, dbHelper);
                            if (result == GlobalErrors.Success)
                            {
                                result = _orderManager.InsertCustomerIntoEmployeeTerritory(order.CustomerId, order.OutletId, order.DriverId, dbHelper);

                                if (result == GlobalErrors.SuccessWithZeroRowAffected)
                                {
                                    result = GlobalErrors.Success;
                                }
                            }
                            if (result == GlobalErrors.Success && order.EnableSendNotification)
                            {
                                order.AllItems = _orderManager.GetOrderDetails(order, dbHelper);
                                _notificationManager.SendSonicNotifiactionToAndroid(order, order.DriverId, NotificationTypes.Delivery , false);

                            }
                        }
                        if (result == GlobalErrors.Success && _requestRepository.Configurations.AllowCustomerAppLogin)
                        {
                            _notificationManager.SendOrderUpdatedNotificationToCustomerApp(order.CustomerId, order.OutletId, order.EmployeeId, order.OrderId, order.OrderStatus, dbHelper);
                        }
                    }

                    if (result != GlobalErrors.Success) break;

                }

                if (_requestRepository.Configurations.AffectVanStockInDeliveryInFO && _requestRepository.Configurations.ForceLoadRequestForOrders && _requestRepository.Configurations.AggregateOrdersToOneLoadRequest && (_requestRepository.Configurations.DeliveryMode.Contains(((int)DeliveryModes.FOInvoicingAndDelivery).ToString()) || _requestRepository.Configurations.DeliveryMode.Contains(((int)DeliveryModes.BOInvoicingFODelivery).ToString()) || _requestRepository.Configurations.DeliveryMode.Contains(((int)DeliveryModes.BOInvoicingThirdPartyDelivery).ToString())))
                {
                    bool _isAllReturnOrders = true;
                    string orderIDs = string.Empty;
                    List<ItemPackModel> allItems = new List<ItemPackModel>();
                    List<int> deliveryPersons = new List<int>();
                    List<int> divisionsList = new List<int>();
                    errorMesage = string.Empty;
                    if (orders != null && orders.Count == 1)
                    {
                        if (orders[0].OrderTypeId == OrderTypes.Return)
                        {
                            errorMesage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Message_No_Load_Request_For_Return_Orders, _requestRepository.LanguageId);
                            return GlobalErrors.Error;
                        }
                    }

                    if (!string.IsNullOrEmpty(orderIDs))
                        orderIDs = orderIDs.TrimEnd(',');

                    if (result == GlobalErrors.Success)
                    {
                        result = SendOrderLoadRequest(orders, orderIDs, dbHelper);
                    }
                    if (result != GlobalErrors.Success)
                    {
                        if (_isAllReturnOrders && (orders != null && orders.Count > 0))
                        {
                            errorMesage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Message_No_Load_Request_For_Return_Orders, _requestRepository.LanguageId);
                            return GlobalErrors.Error;
                        }
                    }
                   

                }
                    return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public async Task<object> GetDashboardsData(bool isToday, bool isMonthly, bool isAnnual)
        {
            var topItems = new List<OrderDashboards>();
            var statuses = new List<OrderDashboards>();
            var allOrders = new VolumeDashboards();
            var allPending = new VolumeDashboards();
            var toDelivery = new VolumeDashboards();
            var delivered = new VolumeDashboards();

            try
            {
                topItems = await _orderManager.GetTopItemsDashboard(isToday, isMonthly, isAnnual);
                statuses = await _orderManager.GetOrdersByStatus(isToday, isMonthly, isAnnual);
                allOrders = await _orderManager.GetOverAllOrders(isToday, isMonthly, isAnnual);
                allPending = await _orderManager.GetAllPendingOrdersDashboard(isToday, isMonthly, isAnnual);
                toDelivery = await _orderManager.GetReadyToDeliveryOrder(isToday, isMonthly, isAnnual);
                delivered = await _orderManager.GetDeliveredOrdersValue(isToday, isMonthly, isAnnual);

                var dashboardData = new
                {
                    topItems = new
                    {
                        xAxis = topItems.Select(x => x.Name).ToList(),
                        yAxis = topItems.Select(x => x.Value).ToList(),
                    },
                    orderByStatus = statuses,
                    overAllOrders = allOrders,
                    currentPending = allPending,
                    readyToDelivery = toDelivery,
                    deliveredOrders = delivered
                };
                return dashboardData;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }

        public GlobalErrors GetOrder(OrderListFilter filter, ref OrderModel order)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.GetOrderHeader(filter, ref order);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetInvoicedOrderForView(OrderListFilter filter, ref OrderModel order)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.GetInvoicedOrderForView(filter, ref order);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public bool CheckIfOrderInPreparation(OrderModel order)
        {
            try
            {
                return _orderManager.CheckIfOrderInPreparation(order);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return false;
            }
        }
        public GlobalErrors SetOrderInPreparation(OrderModel order)
        {
            try
            {
                return _orderManager.SetOrderInPreparation(order);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetOrderList(OrderListFilter filter, ref SharedTableResult<OrderModel> orderList , bool isExportMode = false)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;

                switch (filter.OrderOperation)
                {
                    case OrderOperation.GetRejected:
                        result = _orderManager.GetRejectedOrderList(filter, ref orderList);
                        break;
                    case OrderOperation.Cancel:
                        result = _orderManager.GetCancelledOrderList(filter, ref orderList);
                        break;
                    case OrderOperation.GetPending:
                    case OrderOperation.GetApproved:
                        result = _orderManager.GetNewOrderList(filter, ref orderList);
                        break;

                    case OrderOperation.GetInvoice:
                    case OrderOperation.GetAssigned:
                    case OrderOperation.GetInvoiced:
                    case OrderOperation.GetCollected:
                    case OrderOperation.GetPrepared:
                    case OrderOperation.GetCancelPrepared:
                        result = _orderManager.GetAssignedOrderList(filter, ref orderList ,isExportMode);
                        break;
                    case OrderOperation.GetForSSCC:
                        result = _orderManager.DeleteOrderInPreparation();
                        if (result == GlobalErrors.Success)
                            result = _orderManager.GetAssignedOrderList(filter, ref orderList);
                        break;
                    case OrderOperation.Reprint:
                        result = _orderManager.GetPreparedOrderList(filter, ref orderList);
                        break;
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors HandleFOCImages(OrderModel order, string transactionId, Collection<IFormFile> attachedFiles)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;

            try
            {
                string imgExtensions = LocalUtilities.ImageExtension;
                string rootPath = Path.Combine(LocalUtilities.FOImagesFolderPath, "-1", ImageMode.ShelfIncentiveImages.ToString());
                int sequenceNumber = 1;
                foreach (var Foc in order.ContractedFOCs)
                {
                    foreach (int seq in Foc.FOCimagesSequence)
                    {
                        var file = attachedFiles[seq];
                        string imageName = "-" + Foc.ContractId.ToString() + "-" + order.CustomerId.ToString() + "-" + order.OutletId + "-" + transactionId.ToString() + imgExtensions;
                        string fullPath = Path.Combine(rootPath);
                        if (file != null && file.Length > 0)
                        {
                            if (!Directory.Exists(fullPath))
                            {
                                Directory.CreateDirectory(fullPath);
                            }
                            if (File.Exists(fullPath))
                            {
                                File.Delete(fullPath);
                            }
                            string imageFileName = (sequenceNumber.ToString() + imageName).ToString();
                            string imageFilePath = Path.Combine(fullPath, imageFileName);
                            using (var stream = new FileStream(imageFilePath, FileMode.Create))
                            {
                                file.CopyTo(stream);
                                result = GlobalErrors.Success;
                            }
                            sequenceNumber++;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        // 0 

        // 9
        // 2


        public TransactionModel PrepareTransaction(OrderModel order, DBHelper<int> dBHelper, bool isPartialDelivery = false, bool isReturnMode = false, bool isInvoiceOrderFromStorekeeper = false)
        {
            return PrepareTransaction(order, dBHelper, DateTime.Now, isPartialDelivery, isReturnMode, isInvoiceOrderFromStorekeeper);
        }
        public TransactionModel PrepareTransaction(OrderModel order, DBHelper<int> dBHelper,DateTime currentDate, bool isPartialDelivery = false, bool isReturnMode = false, bool isInvoiceOrderFromStorekeeper = false)
        {
            TransactionModel transaction = new TransactionModel();
            bool getDetailsFromOrderObj = false;
            try
            {
                var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;
               
                if(isReturnMode)
                    transaction.TransactionId = _documentSequenceService.GetMaxDocumentSequence(_requestRepository.CurrentOperator.EmployeeId, DocumentTypes.Return, order.DivisionId, dBHelper);
                else
                    transaction.TransactionId = _documentSequenceService.GetMaxDocumentSequence(order.EmployeeId, DocumentTypes.OrderInvoice, order.DivisionId, dBHelper);

                transaction.WarehouseId = order.WarehouseId;
                transaction.SourceTransactionId = order.OrderId.Trim();
                transaction.TransactionDate = currentDate;
                transaction.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
                transaction.CreatedDate = currentDate;
                transaction.CustomerId = order.CustomerId;
                transaction.OutletId = order.OutletId;
                transaction.EmployeeId = order.EmployeeId;
                transaction.DivisionId = order.DivisionId;
                transaction.OrganizationId = order.OrganizationId;
                transaction.TransactionType = TransactionType.Sales;
                transaction.TransactionTypeId = TransactionType.Sales.GetHashCode();
                transaction.GrossTotal = LocalUtilities.GetRoundedDecimal(order.GrossTotal, numberOfDigits);
                transaction.NetTotal = LocalUtilities.GetRoundedDecimal(order.NetTotal, numberOfDigits);
                transaction.RemainingAmount = LocalUtilities.GetRoundedDecimal(order.NetTotal, numberOfDigits);
                transaction.Tax = LocalUtilities.GetRoundedDecimal(order.Tax, numberOfDigits);
                transaction.CalculatedRetailTax = LocalUtilities.GetRoundedDecimal(order.CalculatedRetailTax, numberOfDigits);
                transaction.RoundingDifference = order.RoundingDifference;
                if (((transaction.GrossTotal + transaction.Tax + transaction.CalculatedRetailTax) - (transaction.NetTotal + transaction.RoundingDifference)) > 0)
                    transaction.Discount = (transaction.GrossTotal + transaction.Tax + transaction.CalculatedRetailTax) - (transaction.NetTotal + transaction.RoundingDifference);

                #region [Fill employee related data]

                int helperId = -1, driverId = -1, supervisorId = -1, salesRepId = -1, salesManagerId = -1;

                _employeeManager.GetEmployeeRelatedStaff(transaction.EmployeeId, dBHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepId, ref salesManagerId);
                transaction.HelperId = helperId;
                transaction.DriverId = driverId;
                transaction.SupervisorId = supervisorId;
                transaction.SalesRepId = salesRepId;
                transaction.DriverId = driverId;
                transaction.SalesManagerId = salesManagerId;

                #endregion [Fill employee related data]
                transaction.RouteHistoryId = -1;
                transaction.RouteId = -1;
                transaction.Posted = false;
                transaction.Voided = false;

                if (_requestRepository.Configurations.EnableSalesDistributionChannel) 
                {
                    transaction.SDCId = order.SDCId;
                }
                transaction.TransactionStatusId = TransactionStatues.None.GetHashCode();
                List<Promotion> targetPromotions = order.CustomerPromotions.FindAll(x => x.PromotionType == PromotionTypes.TargetPromotion || x.PromotionType == PromotionTypes.DynamicTargetPromotion || x.RelatedToTarget).ToList();
                if (order.OrderOperation == OrderOperation.FullDelivery)
                {
                    transaction.CreationReason = TransactionCreationReason.OrderManualDeliveryBOInvoice;
                    bool areThereContractedFOCs = order.ContractedFOCs != null && order.ContractedFOCs.Count > 0;
                    if (_requestRepository.Configurations.AllowBatchSelectionInDelivery || (targetPromotions != null && targetPromotions.Count > 0) || areThereContractedFOCs)
                    {
                        getDetailsFromOrderObj = true;
                    }
                }
                else if (order.OrderOperation == OrderOperation.PartialDelivery)
                {
                    transaction.CreationReason = TransactionCreationReason.PartialDeliveryInBO;
                    getDetailsFromOrderObj = true;

                }
                else if (isInvoiceOrderFromStorekeeper)
                {
                    transaction.CreationReason = TransactionCreationReason.OrderDeliveryBOInvoice;
                    order.OrderOperation = OrderOperation.FullDelivery;
                    getDetailsFromOrderObj = true;
                    transaction.RefWarehouseId = -1;
                    GlobalErrors result = GlobalErrors.NotInitialized;

                    if (result == GlobalErrors.Success)
                    {
                        DeliveryAssignmentModel deliveryAssignment = new DeliveryAssignmentModel();
                        result = _orderManager.GetOrderDeliveryDetails(order, ref deliveryAssignment, dBHelper);
                        if (result == GlobalErrors.Success && deliveryAssignment != null)
                        {
                            transaction.RefWarehouseId = deliveryAssignment.VehicleId;
                        }
                    }
                }
                else if (isReturnMode)
                {
                    getDetailsFromOrderObj = true;
                }
                else
                {
                    transaction.CreationReason = TransactionCreationReason.OrderDeliveryBOInvoice;
                }

                transaction.Outlet = _customerManager.FillOutletData(transaction.CustomerId, transaction.OutletId, transaction.DivisionId, dBHelper);
                transaction.AccountId = -1;
                transaction.CurrencyId = -1;
                transaction.SalesMode = !_orderManager.AllowSalesModeSelectionForCustomerInOrder(order.Outlet.CustomerTypeId) ? (SalesModes)order.Outlet.CustomerTypeId : (SalesModes)order.SalesMode;
                transaction.LPONumber = order.LPO;
                List<ItemPackModel> orderItems = new List<ItemPackModel>(order.AllItems);
                ItemPackModel orderItem = null;

                if (order.OrderOperation == OrderOperation.FullDelivery || order.OrderOperation == OrderOperation.PartialDelivery) // this was added to add achievement promotions to the transaction 
                    transaction.CustomerPromotions = order.CustomerPromotions;
                //get the current entered details and batches in delivery
                if (!isInvoiceOrderFromStorekeeper)
                {
                    if (!getDetailsFromOrderObj)
                    {
                        order.AllItems = _orderManager.GetOrderDetails(order, dBHelper);
                    }
                    //if (isInvoiceOrderFromStorekeeper)
                    //{
                    //    order.AllItems = _warehouseStockService.PrepareAddedItemsAsBatchesLinesToSaveOrderPreperation(order.AllItems);
                    //}
                    else
                    {
                        //if (_requestRepository.Configurations.AllowBatchSelectionInDelivery && !(_requestRepository.Configurations.RestrictReturnFromInvoices && isReturnMode))
                        //    order.AllItems = _warehouseStockService.PrepareAddedItemsAsBatchesLinesToSave(order.AllItems);
                    }
                }

                foreach (ItemPackModel orderPack in order.AllItems)
                {
                    ItemPackModel pack = new ItemPackModel();

                    if (orderItems.Find(orgPack => orgPack.PackId == orderPack.PackId && orgPack.SalesTransactionTypeId == orderPack.SalesTransactionTypeId) != null)
                        orderItem = orderItems.Find(orgPack => orgPack.PackId == orderPack.PackId && orgPack.SalesTransactionTypeId == orderPack.SalesTransactionTypeId);

                 
                    pack.ItemId = orderPack.ItemId;
                    pack.PackId = orderPack.PackId;
                    pack.BasePrice = orderPack.BasePrice;
                    pack.Price = orderPack.Price;
                    pack.UsedPriceListId = orderPack.UsedPriceListId;
                    pack.RequiredQty = orderPack.RequiredQty;
                    pack.Sequence = orderPack.Sequence;
                    pack.ReSellable = orderPack.ReSellable;
                    pack.ReturnReasonId = orderPack.ReturnReasonId;
                    pack.ConsumerPrice = orderPack.ConsumerPrice;
                    pack.ConsumerPriceListId = orderPack.ConsumerPriceListId;
                    pack.PackStatusId = orderPack.PackStatusId > 0 ? orderPack.PackStatusId : -1;
                    pack.ReturnReasonId = orderPack.ReturnReasonId > 0 ? orderPack.ReturnReasonId : -1;
                    pack.InStock = orderPack.InStock;
                    pack.InSCS = orderPack.InSCS;
                    pack.OrderDetailsBatches = orderPack.OrderDetailsBatches;
                    if (!getDetailsFromOrderObj && !isReturnMode)
                    {
                        pack.BatchNo = LocalUtilities.DefaultBatchNo;
                        pack.ExpiryDate = LocalUtilities.DefaultExpiryDate;
                    }
                    else
                    {
                        pack.BatchNo = orderPack.BatchNo;
                        pack.ExpiryDate = isReturnMode ? orderPack.ExpiryDateModel.Date : orderPack.ExpiryDate;
                    }

                    pack.SalesTransactionTypeId = orderPack.SalesTransactionTypeId;
                    pack.PiecesInPack = orderPack.PiecesInPack;
                    pack.Tax = orderPack.Tax;
                    pack.BaseTax = orderPack.BaseTax;
                    pack.RetailValue = orderPack.CalculatedRetailTax / orderPack.RequiredQty;
                    pack.RetailPrice = 1;
                    pack.RetailTax = pack.RetailValue * 100;
                    pack.Discount = orderPack.Discount;
                    if (isPartialDelivery || isReturnMode)
                    {
                        pack.DiscountTypeId = orderPack.DiscountTypeId;
                    }
                    else
                    {
                        pack.DiscountTypeId = DiscountValueTypes.Percentage.GetHashCode();
                    }
                    // here we want to add achievement promotions to order packs
                    if (orderItem != null)
                    {
                        pack.PromotedDiscount = orderItem.PromotedDiscount;
                        orderPack.ExciseContractPromotedDiscount = orderItem.ExciseContractPromotedDiscount;
                        orderPack.RetailCalculatedPromotedDiscount = orderItem.RetailCalculatedPromotedDiscount;
                        orderPack.WholeSalesPromotedDiscount = orderItem.WholeSalesPromotedDiscount;
                        orderPack.WholeSalesCalPromotedDiscount = orderItem.WholeSalesCalPromotedDiscount;
                        orderPack.DynamicPromotedDiscount = orderItem.DynamicPromotedDiscount;
                        orderPack.DynamicCalTargetPromotedDiscount = orderItem.DynamicCalTargetPromotedDiscount;
                        orderPack.ItemPromotedDiscount = orderItem.ItemPromotedDiscount; // for related to target promotions ;

                        pack.RetailCalculatedPromotedDiscount = orderItem.RetailCalculatedPromotedDiscount;
                        pack.WholeSalesCalPromotedDiscount = orderItem.WholeSalesCalPromotedDiscount;
                        pack.DynamicCalTargetPromotedDiscount = orderItem.DynamicCalTargetPromotedDiscount;
                    }

                    ConvertDiscountAndTax(pack, orderPack, order.CustomerId, order.OutletId);

                    transaction.AllItems.Add(pack);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                transaction = null;
            }
            return transaction;
        }

        private void ConvertDiscountAndTax(ItemPackModel pack, ItemPackModel orderPack, int customerId, int outletId)
        {
            decimal tempValue;
            decimal addedExciseTax = 0;
            bool applyRoundingPerLine = _requestRepository.Configurations.ApplyRoundingOnCalculations;
            try
            {
                if (_requestRepository.Configurations.IncludeExciseTaxInDiscount)
                {
                    addedExciseTax = pack.RetailValue * pack.RequiredQty;
                    if (applyRoundingPerLine)
                    {
                        addedExciseTax = LocalUtilities.GetRoundedDecimal(addedExciseTax, _requestRepository.Configurations.NumberOfDigits);
                    }
                }
                var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;
                tempValue = (pack.RequiredQty * pack.Price) + addedExciseTax;
                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                    tempValue = LocalUtilities.GetRoundedDecimal(tempValue, numberOfDigits);

                if (pack.DiscountTypeId == (int)DiscountValueTypes.Percentage)
                    pack.CalculatedDiscount = (tempValue * orderPack.Discount / 100);
                else if (pack.DiscountTypeId == (int)DiscountValueTypes.Amount)
                    pack.CalculatedDiscount = orderPack.Discount;

                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                    pack.CalculatedDiscount = LocalUtilities.GetRoundedDecimal(pack.CalculatedDiscount, numberOfDigits);


                tempValue = tempValue - pack.CalculatedDiscount;

                // Item Promo discount
                if (orderPack.ItemPromotedDiscount > 100)
                    pack.ItemPromotedDiscount = 100;
                else
                {
                    //Already Calculated
                    pack.ItemPromotedDiscount = orderPack.ItemPromotedDiscount;
                }
                pack.RelatedToTargetDiscount = orderPack.RelatedToTargetDiscount;
                pack.CalculatedItemPromotedDiscount = tempValue * pack.ItemPromotedDiscount / 100;

                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                {
                    pack.CalculatedItemPromotedDiscount = LocalUtilities.GetRoundedDecimal(pack.CalculatedItemPromotedDiscount, numberOfDigits);
                }

                // All items Discount
                if (orderPack.AllItemDiscount > 100)
                    pack.AllItemDiscount = 100;
                else
                {
                    //Already Calculated
                    pack.AllItemDiscount = orderPack.AllItemDiscount; // tempValue * orderPack.AllItemDiscount / 100;////////////////////////////
                }
                pack.CalculatedAllItemPromotedDiscount = tempValue * pack.AllItemDiscount / 100;

                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                {
                    pack.CalculatedAllItemPromotedDiscount = LocalUtilities.GetRoundedDecimal(pack.CalculatedAllItemPromotedDiscount, numberOfDigits);
                }

                // Retail Promo discount
                if (orderPack.ExciseContractPromotedDiscount > 100)
                    pack.ExciseContractPromotedDiscount = tempValue;
                else
                {
                    //Already Calculated
                    pack.ExciseContractPromotedDiscount = orderPack.ExciseContractPromotedDiscount; //tempValue * orderPack.ItemPromotedDiscount / 100;
                }

                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                {
                    pack.RetailCalculatedPromotedDiscount = LocalUtilities.GetRoundedDecimal(pack.RetailCalculatedPromotedDiscount, numberOfDigits);
                }

                // WholeSales Promo discount
                if (orderPack.WholeSalesPromotedDiscount > 100)
                    pack.WholeSalesPromotedDiscount = tempValue;
                else
                {
                    //Already Calculated
                    pack.WholeSalesPromotedDiscount = orderPack.WholeSalesPromotedDiscount; //tempValue * orderPack.ItemPromotedDiscount / 100;
                }
                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                {
                    pack.WholeSalesCalPromotedDiscount = LocalUtilities.GetRoundedDecimal(pack.WholeSalesCalPromotedDiscount, numberOfDigits);
                }

                // Dynamic Promo discount
                if (orderPack.DynamicPromotedDiscount > 100)
                    pack.DynamicPromotedDiscount = tempValue;
                else
                {
                    //Already Calculated
                    pack.DynamicPromotedDiscount = orderPack.DynamicPromotedDiscount; //tempValue * orderPack.ItemPromotedDiscount / 100;
                }
                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                {
                    pack.DynamicCalTargetPromotedDiscount = LocalUtilities.GetRoundedDecimal(pack.DynamicCalTargetPromotedDiscount, numberOfDigits);
                }

                //Promoted discount
                pack.PromotedDiscount = (pack.AllItemDiscount + pack.ItemPromotedDiscount + pack.ExciseContractPromotedDiscount + pack.WholeSalesPromotedDiscount + pack.DynamicPromotedDiscount);
                if (pack.PromotedDiscount > 100)
                    pack.PromotedDiscount = 100;
                pack.CalculatedPromotedDiscount = (tempValue) * pack.PromotedDiscount / 100;
                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                {
                    pack.CalculatedPromotedDiscount = LocalUtilities.GetRoundedDecimal((tempValue) * pack.PromotedDiscount / 100, numberOfDigits);
                }

                pack.CalculatedDiscount += pack.CalculatedPromotedDiscount;

                tempValue -= pack.CalculatedPromotedDiscount;
                // Header discount
                if (orderPack.HeaderDiscount > 100)
                    pack.HeaderDiscount = tempValue;
                else
                    pack.HeaderDiscount = tempValue * orderPack.HeaderDiscount / 100;

                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                    pack.HeaderDiscount = LocalUtilities.GetRoundedDecimal(pack.HeaderDiscount, numberOfDigits);

                pack.CalculatedDiscount += pack.HeaderDiscount;

                // Base Discount
                pack.BaseDiscount = orderPack.BaseDiscount;
                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                    pack.BaseDiscount = LocalUtilities.GetRoundedDecimal(pack.BaseDiscount, numberOfDigits);

                // Shelf Incentive Discount
                // calculated based on -> [tempValue = gross - calculatedDiscount - calculatePromotedDiscount]
                pack.ShelfIncentiveDiscount = orderPack.ShelfIncentiveDiscount;
                if (orderPack.ShelfIncentiveDiscount > 100)
                    pack.ShelfIncentiveDiscountAmount = tempValue;
                else
                    pack.ShelfIncentiveDiscountAmount = tempValue * orderPack.ShelfIncentiveDiscount / 100;

                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                    pack.ShelfIncentiveDiscountAmount = LocalUtilities.GetRoundedDecimal(pack.ShelfIncentiveDiscountAmount, numberOfDigits);

                pack.CalculatedDiscount += pack.ShelfIncentiveDiscountAmount;


                // Retail TAX
                if (!_requestRepository.Configurations.AllowRetailTaxOnItemsForCustomer(customerId, outletId))
                {
                    pack.CalculatedRetailTax = 0;
                }
                else
                {
                    pack.CalculatedRetailTax = orderPack.CalculatedRetailTax;
                    if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                        pack.CalculatedRetailTax = LocalUtilities.GetRoundedDecimal(pack.CalculatedRetailTax, numberOfDigits);

                }

                // TAX AND BASE TAX
                decimal gross = pack.RequiredQty * pack.Price;
                decimal baseGross = pack.RequiredQty * pack.BasePrice;
                if (_requestRepository.Configurations.ApplyRoundingOnCalculations)
                {
                    gross = LocalUtilities.GetRoundedDecimal(gross, numberOfDigits);
                    baseGross = LocalUtilities.GetRoundedDecimal(baseGross, numberOfDigits);
                }
                if (_requestRepository.Configurations.CalculateTaxBeforeDiscounts)
                {
                    pack.CalculatedTax = (gross + pack.CalculatedRetailTax) * pack.Tax / 100;
                    pack.CalculatedBaseTax = (baseGross + pack.CalculatedRetailTax) * pack.BaseTax / 100;
                }
                else
                {
                    if(pack.SalesTransactionTypeId != SalesTransactionTypes.FreeReturn)
                    {
                        decimal shelveIncentiveDiscount = _requestRepository.Configurations.CalculateShelfIncentiveAfterTax ? pack.ShelfIncentiveDiscountAmount : 0;
                        pack.CalculatedTax = (gross + pack.CalculatedRetailTax - (pack.CalculatedDiscount - shelveIncentiveDiscount)) * pack.Tax / 100;
                        pack.CalculatedBaseTax = (baseGross + pack.CalculatedRetailTax - (pack.CalculatedDiscount - shelveIncentiveDiscount)) * pack.BaseTax / 100;
                    }
                    else
                    {
                        pack.CalculatedTax = orderPack.CalculatedTax;
                    }
                }
                if (_requestRepository.Configurations.AllowTaxOverFreeItems && pack.SalesTransactionTypeId == SalesTransactionTypes.FOC)
                {
                    pack.Tax = pack.BaseTax;
                }

                if (_requestRepository.Configurations.AllowTaxOverFreeItems && (pack.SalesTransactionTypeId == SalesTransactionTypes.FOC || pack.SalesTransactionTypeId == SalesTransactionTypes.Promotion))
                {
                    if (_requestRepository.Configurations.ExcludeBasePriceFromFreeItemsVAT)
                    {
                        pack.CalculatedTax = (pack.CalculatedRetailTax) * pack.BaseTax / 100;
                        pack.CalculatedBaseTax = (pack.CalculatedRetailTax) * pack.BaseTax / 100;
                    }
                    else
                    {
                        pack.CalculatedTax = (baseGross + pack.CalculatedRetailTax) * pack.BaseTax / 100;
                        pack.CalculatedBaseTax = (baseGross + pack.CalculatedRetailTax) * pack.BaseTax / 100;
                    }
                }
                if (_requestRepository.Configurations.ApplyRoundingOnCalculations && !_requestRepository.Configurations.ApplyZATCARoundingRules)
                {
                    pack.CalculatedTax = LocalUtilities.GetRoundedDecimal(pack.CalculatedTax, numberOfDigits);
                    pack.CalculatedBaseTax = LocalUtilities.GetRoundedDecimal(pack.CalculatedBaseTax, numberOfDigits);
                }
                pack.GrossTotal = gross;
                pack.NetTotal = pack.SalesTransactionTypeId != SalesTransactionTypes.FreeReturnAsDiscount? gross - pack.CalculatedDiscount + pack.CalculatedRetailTax + pack.CalculatedTax : 0;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
        }
        public GlobalErrors IsReturnOrderValidToProcess(ref OrderModel order, ref string message)
        {
            var result = GlobalErrors.Success;
            try
            {
                if (!_orderManager.CheckOrderStatusBeforeAnyAction(order,null))
                {
                    result = GlobalErrors.DataAlreadyExists;
                    return result;
                }
                order.TransactionId = _documentSequenceService.GetMaxDocumentSequence(_requestRepository.Configurations.AllowInvoiceOrderBeforeAssignOrder ? _requestRepository.CurrentOperator.EmployeeId : order.EmployeeId, DocumentTypes.Return, order.DivisionId);
                if (order.TransactionId == string.Empty)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Document_Sequence, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors IsOrderValidToAssign(ref OrderModel order, ref string message)
        {
            var result = GlobalErrors.Success;
            try
            {
                result = _orderManager.GetLastOrderStatus(ref order, null);

                if (order.OrderStatus != OrderStatus.Approved)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Canceled, _requestRepository.LanguageId); // this order is Canceled ;
                    return GlobalErrors.Error;
                }

                if (_requestRepository.Configurations.ControlTransactionOnDivisionLevel)
                {
                    if (!_employeeManager.EmployeeHasAccessToDivision(order.DriverId, order.DivisionId))
                    {
                        message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Employee_Not_Access_Division, _requestRepository.LanguageId);// Employee don't have access on the selected division
                        return GlobalErrors.Error;
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }

            return result;
        }



        public GlobalErrors SaveOrderSonic(OrderModel order, ref string message)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                #region[Get Order Approval Status]
                int approvalStatus = -1;
                if (_requestRepository.Configurations.ApplyApprovalCycleInWeb.Contains(((int)ApprovalCycleOperations.Orders).ToString()))
                {
                    DBHelper<int> dbHelper = new DBHelper<int>();
                    SettingManager settingManager = new SettingManager(_requestRepository);
                    result = settingManager.GetCurrentOperationStatus((int)ApprovalCycleOperations.Orders, ref approvalStatus, dbHelper);
                }
                else
                {
                    approvalStatus = -1;
                }
                #endregion

                var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;
                var documentType = order.OrderTypeId == OrderTypes.Return ? DocumentTypes.ReturnOrder : DocumentTypes.Order;
                order.OrderId = _documentSequenceService.GetMaxDocumentSequence(_requestRepository.CurrentOperator.EmployeeId, documentType, order.DivisionId);
                if (string.IsNullOrEmpty(order.OrderId))
                {
                    return GlobalErrors.Error;
                }
               
                if (_requestRepository.Configurations.OrdersToBeApprovedInFO.Contains("2"))
                {
                    order.OrderStatus = OrderStatus.NewWebOrder;
                }
                else
                {
                    order.OrderStatus = OrderStatus.New;
                }
                
                if (order.OrderTypeId != OrderTypes.Return)
                {
                    order.OrderTypeId = OrderTypes.Sales;
                }
                order.ApprovalStatus = approvalStatus;
                order.NetTotal = LocalUtilities.GetRoundedDecimal(order.NetTotal, numberOfDigits);
                order.GrossTotal = LocalUtilities.GetRoundedDecimal(order.GrossTotal, numberOfDigits);
                order.Discount = LocalUtilities.GetRoundedDecimal(order.Discount, numberOfDigits);
                order.Tax = LocalUtilities.GetRoundedDecimal(order.Tax, numberOfDigits);
                order.CalculatedRetailTax = LocalUtilities.GetRoundedDecimal(order.CalculatedRetailTax, numberOfDigits);
                order.PromotedDiscount = LocalUtilities.GetRoundedDecimal(order.PromotedDiscount, numberOfDigits);
                order.HeaderDiscount = LocalUtilities.GetRoundedDecimal(order.HeaderDiscount, numberOfDigits);

                order.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
                order.CreationSource = CreationSource.SonicWeb;
                order.OrganizationId = _requestRepository.CurrentOperator.OrganizationId;
                if (!(order.OrderTypeId == OrderTypes.Return && _requestRepository.Configurations.RestrictReturnFromInvoices))
                {
                    order.AllItems = new List<ItemPackModel>();
                    order.AllItems.AddRange(order.SoldItems);
                    order.AllItems.AddRange(order.PromotedItems);
                }

                order.Notes = new List<string>();
                if (order.Remarks != null && !string.IsNullOrEmpty(order.Remarks.Trim()))
                {
                    order.Notes.Add(order.Remarks);
                }
                // Saving
                result = SaveOrder(order, ref message); 
                
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors CreateComplementaryOrder(OrderModel order, DBHelper<int> dBHelper)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {

                var complementaryOrder = new OrderModel();
                var filter = new OrderListFilter();
                filter.OrderId = order.OrderId;
                filter.CustomerId = order.CustomerId;
                filter.OutletId = order.OutletId;
                filter.DivisionId = order.DivisionId;
                result = _orderManager.GetOrderHeader(filter, ref complementaryOrder, dBHelper);
                if (result == GlobalErrors.Success)
                    complementaryOrder.AllItems = FillComplementaryItems(order, ref complementaryOrder, dBHelper);
                if (complementaryOrder.AllItems == null) return GlobalErrors.Error;
                if (complementaryOrder.AllItems.Count > 0)
                {
                    complementaryOrder.OrderId = _documentSequenceService.GetMaxDocumentSequence(_requestRepository.CurrentOperator.EmployeeId, DocumentTypes.Order, order.DivisionId);
                    if (string.IsNullOrEmpty(order.OrderId))
                    {
                        return GlobalErrors.Error;
                    }
                    complementaryOrder.EmployeeId = _requestRepository.CurrentOperator.EmployeeId;
                    complementaryOrder.ReferenceOrderID = order.OrderId;
                    complementaryOrder.OrderTypeId = OrderTypes.Sales;
                    complementaryOrder.DesiredDeliveryDate = DateTime.Now;
                    complementaryOrder.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
                    complementaryOrder.CreationSource = CreationSource.SonicWeb;
                    complementaryOrder.OrganizationId = _requestRepository.CurrentOperator.OrganizationId;
                    int helperId = -1, driverId = -1, supervisorId = -1, salesRepID = -1, salesManagerId = -1;
                    _employeeManager.GetEmployeeRelatedStaff(complementaryOrder.EmployeeId, dBHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepID, ref salesManagerId);
                    complementaryOrder.HelperId = helperId;
                    complementaryOrder.DriverId = driverId;
                    complementaryOrder.SupervisorId = supervisorId;
                    complementaryOrder.SalesRepId = salesRepID;
                    complementaryOrder.DriverId = driverId;
                    complementaryOrder.SalesManagerId = salesManagerId;

                    //complementaryOrder.NetTotal = complementaryOrder.NetTotal - order.NetTotal;
                    //complementaryOrder.GrossTotal = complementaryOrder.GrossTotal - order.GrossTotal;
                    //complementaryOrder.Tax = complementaryOrder.Tax - order.Tax;
                    //complementaryOrder.Discount = complementaryOrder.Discount - order.Discount;
                    //complementaryOrder.CalculatedRetailTax = complementaryOrder.CalculatedRetailTax - order.CalculatedRetailTax;
                    complementaryOrder.RouteHistoryId = -1;
                    complementaryOrder.OrderStatus = OrderStatus.ComplementaryOrder;

                    result = _orderManager.SaveOrderHeaders(complementaryOrder, dBHelper, 5);
                    if (result == GlobalErrors.Success)
                    {
                        result = _orderManager.SaveOrderDetails(complementaryOrder, dBHelper);
                    }
                    if (result == GlobalErrors.Success)
                    {
                        result = _documentSequenceManager.UpdateMaxTransactionID(dBHelper, DocumentTypes.Order, complementaryOrder.OrderId, _requestRepository.CurrentOperator.EmployeeId, order.DivisionId);
                    }
                }
                else
                {
                    return GlobalErrors.Success;
                }

                return result;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        private List<ItemPackModel> FillComplementaryItems(OrderModel order, ref OrderModel complementaryOrder, DBHelper<int> dBHelper)
        {
            var complementaryItems = new List<ItemPackModel>();
            try
            {
                var complementaryItem = new ItemPackModel();
                decimal grossTotal = 0;
                decimal netTotal = 0;
                decimal taxTotal = 0;
                decimal retailTaxTotal = 0;
                decimal discountTotal = 0;
                List<ItemPackModel> originalOrderItems = complementaryOrder.SoldItems;
                originalOrderItems.ForEach(orgItem =>
                {
                    var newItem = order.AllItems.Find(x => x.PackId == orgItem.PackId && x.SalesTransactionTypeId == orgItem.SalesTransactionTypeId);
                    if (newItem == null)
                    {
                        newItem = new ItemPackModel();
                        newItem.RequiredQty = 0;
                        newItem.Discount = 0;
                        newItem.CalculatedRetailTax = 0;
                    }

                    if (orgItem.RequiredQty > newItem.RequiredQty)
                    {
                        complementaryItem = new ItemPackModel();
                        complementaryItem.PackId = orgItem.PackId;
                        complementaryItem.RequiredQty = orgItem.RequiredQty - newItem.RequiredQty;
                        complementaryItem.Price = orgItem.Price;
                        complementaryItem.Tax = orgItem.Tax;
                        complementaryItem.BaseTax = orgItem.BaseTax;
                        complementaryItem.DiscountTypeId = orgItem.DiscountTypeId.GetHashCode();
                        if (orgItem.DiscountTypeId == DiscountValueTypes.Amount.GetHashCode())
                        {
                            complementaryItem.Discount = orgItem.Discount - newItem.Discount;
                        }
                        else
                        {
                            complementaryItem.Discount = orgItem.Discount;
                        }

                        complementaryItem.SalesTransactionTypeId = orgItem.SalesTransactionTypeId;
                        complementaryItem.BasePrice = orgItem.BasePrice;
                        complementaryItem.Foc = 0;
                        complementaryItem.ActualDeliveredQuantity = 0;
                        complementaryItem.PromotedDiscount = 0;
                        complementaryItem.FinalDiscountPercentage = 0;
                        complementaryItem.AdditionalDiscount = 0;
                        complementaryItem.StockStatusId = orgItem.StockStatusId;
                        complementaryItem.UsedPriceListId = orgItem.UsedPriceListId;
                        complementaryItem.RequestedQuantity = complementaryItem.RequiredQty;
                        complementaryItem.CalculatedRetailTax = orgItem.CalculatedRetailTax - newItem.CalculatedRetailTax;
                        ConvertDiscountAndTax(complementaryItem, complementaryItem, order.CustomerId, order.OutletId);
                        grossTotal += complementaryItem.GrossTotal;
                        netTotal += complementaryItem.NetTotal;
                        taxTotal += complementaryItem.CalculatedTax;
                        retailTaxTotal += complementaryItem.CalculatedRetailTax;
                        discountTotal += complementaryItem.CalculatedDiscount;
                        complementaryItems.Add(complementaryItem);
                    }


                });
                complementaryOrder.NetTotal = netTotal;
                complementaryOrder.GrossTotal = grossTotal;
                complementaryOrder.Tax = taxTotal;
                complementaryOrder.CalculatedRetailTax = retailTaxTotal;
                complementaryOrder.Discount = discountTotal;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
            return complementaryItems;
        }
        public GlobalErrors SaveOrderCRM(OrderCRMModel order, ref string ErrorMsg)
        {
            var result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                OrderModel orderModel = new OrderModel();
                orderModel.OrderDate = new DateTime();
                orderModel.Outlet = new CustomerOutletModel();
                orderModel.Outlet.CustomerId = int.Parse(EncryptionManager.Instance.DecryptData(order.CustomerIdSignature));
                orderModel.Outlet.OutletId = int.Parse(EncryptionManager.Instance.DecryptData(order.OutletIdSignature));

                orderModel.CustomerId = orderModel.Outlet.CustomerId;
                orderModel.OutletId = orderModel.Outlet.OutletId;
                orderModel.WarehouseId = order.WarehouseId;
                order.SoldItems.ForEach(i =>
                {
                    i.RequestedQuantity = i.RequiredQty;
                });
                if (order.PromotedItems != null)
                {
                    order.PromotedItems.ForEach(i =>
                    {
                        i.RequestedQuantity = i.RequiredQty;
                    });
                }

                dbHelper = new DBHelper<int>(_requestRepository.CurrentOperator.EmployeeId);
                dbHelper.BeginTransaction();


                orderModel.SoldItems = order.SoldItems;
                orderModel.PromotedItems = order.PromotedItems;
                orderModel.OrderStatus = OrderStatus.New;
                orderModel.OrderTypeId = OrderTypes.Sales;
                orderModel.DivisionId = -1;
                orderModel.OrderId = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, _requestRepository.CurrentOperator.EmployeeId, DocumentTypes.Order, orderModel.DivisionId, 2);
                if (string.IsNullOrEmpty(order.OrderId) || string.IsNullOrEmpty(order.OrderId.Trim()))
                {
                    return GlobalErrors.Error;
                }
                orderModel.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
                orderModel.CreationSource = CreationSource.SonicCRM;
                orderModel.OrganizationId = _requestRepository.Configurations.RestrictDataToCustomerOrganization ? order.OrganizationId : _requestRepository.CurrentOperator.OrganizationId;
                orderModel.AllItems = new List<ItemPackModel>();
                orderModel.AllItems.AddRange(orderModel.SoldItems);
                orderModel.AllItems.AddRange(orderModel.PromotedItems);
                orderModel.OrderSourceId = order.OrderSourceId;
                orderModel.EmployeeId = _requestRepository.CurrentOperator.EmployeeId;
                orderModel.DesiredDeliveryDate = order.DesiredDeliveryDateModel.Date;
                orderModel.Notes = new List<string>();
                orderModel.CustomerPromotions = order.CustomerPromotions;
                if (order.Remarks != null && !string.IsNullOrEmpty(order.Remarks.Trim()))
                {
                    orderModel.Notes.Add(order.Remarks);
                }
                orderModel.NetTotal = order.NetTotal;
                orderModel.GrossTotal = order.GrossTotal;
                orderModel.Tax = order.Tax;
                orderModel.CalculatedRetailTax = order.CalculatedRetailTax;
                orderModel.Discount = order.Discount;
                orderModel.ResumeDateModel = order.ResumeDateModel;
                if (order.ResumeDateModel != null && order.ResumeDateModel.Date != DateTime.MinValue)
                {
                    orderModel.ResumeDate = order.ResumeDateModel.Date;
                }
                else
                {
                    orderModel.ResumeDate = DateTime.MinValue;
                }
                orderModel.DeliveryChargesId = order.DeliveryChargesId;
                orderModel.DeliveryValue = order.DeliveryValue;
                orderModel.DeliveryTax = order.DeliveryTax;

                orderModel.RecurringChangedReasonId = order.RecurringChangedReasonId;
                orderModel.RecurringStatusId = order.RecurringStatusId;
                orderModel.RecurringComment = order.RecurringComment;
                orderModel.RecurringTypeId = order.RecurringTypeId;
                orderModel.RecurringValue = order.RecurringValue;
                orderModel.CreationSource = CreationSource.SonicCRM;
                if (order.SourceOrderId != null)
                {
                    // check if unique
                    bool uniqueCode = true;
                    result = _masterDataService.CheckIfUniqueValue(CoreDataBaseConstants.TableNames.SalesOrder, CoreDataBaseConstants.QueryColumnsNames.SourceOrderID, order.SourceOrderId, "", ref uniqueCode, dbHelper);
                    if (!uniqueCode)
                    {
                        ErrorMsg = "Source Order ID must be unique";
                        return GlobalErrors.ValueIsMoreThanRequested;
                    }

                }
                orderModel.SourceOrderId = order.SourceOrderId;
                orderModel.PaymentMethodTypeID = order.PaymentMethodTypeID;
                if (order.PaymentReferenceNumber != null)
                {
                    // check if unique
                    bool uniqueCode = true;
                    result = _masterDataService.CheckIfUniqueValue(CoreDataBaseConstants.TableNames.SalesOrder, CoreDataBaseConstants.QueryColumnsNames.PaymentReferenceNumber, order.PaymentReferenceNumber, "", ref uniqueCode, dbHelper);
                    if (!uniqueCode)
                    {
                        ErrorMsg = "Payment Reference Number must be unique";
                        return GlobalErrors.ValueIsMoreThanRequested;
                    }
                }
                orderModel.PaymentReferenceNumber = order.PaymentReferenceNumber;

                string message = string.Empty;
                result = SaveOrder(orderModel, ref message, false, dbHelper);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _requestRepository.CurrentOperator.EmployeeId);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors SaveOrder(OrderModel order)
        {
            string message = string.Empty;
            return SaveOrder(order, ref message, false, null);
        }
        public GlobalErrors SaveOrder(OrderModel order, ref string message)
        {
            return SaveOrder(order, ref message, false, null);
        }
        public GlobalErrors SaveOrder(OrderModel order, ref string message, bool isEcho, DBHelper<int> dbHelper, bool isCustomerApp = false)
        {
            var result = GlobalErrors.NotInitialized;
            bool AllowPromotion = true;
            bool commitNow = false;
            int employeeId = -1;
            string employeeCodeName = string.Empty;
            int uploaded = -1;
            try
            {
                if (dbHelper == null)
                {
                    dbHelper = new DBHelper<int>(_requestRepository.CurrentOperator.EmployeeId);
                    dbHelper.BeginTransaction();
                    commitNow = true;
                }
                int helperId = -1, driverId = -1, supervisorId = -1, salesRepID = -1, salesManagerId = -1;
                _employeeManager.GetEmployeeRelatedStaff(order.EmployeeId, dbHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepID, ref salesManagerId);
                order.HelperId = helperId;
                order.DriverId = driverId;
                order.SupervisorId = supervisorId;
                order.SalesRepId = salesRepID;
                order.DriverId = driverId;
                order.SalesManagerId = salesManagerId;
                var documentType = order.OrderTypeId == OrderTypes.Return ? DocumentTypes.ReturnOrder : DocumentTypes.Order;
                result = _documentSequenceManager.UpdateMaxTransactionID(dbHelper, documentType, order.OrderId, isEcho ? order.EmployeeId : _requestRepository.CurrentOperator.EmployeeId, order.DivisionId);
                if (order.OrderStatus == OrderStatus.NewWebOrder)
                {
                    string divisions = string.Join(",", order.AllItems.Select(Division => Division.DivisionId).Distinct());
                    employeeId = _orderManager.GetEmployeeUnderCustTerritory(ref employeeId, ref uploaded , order.CustomerId, order.OutletId, divisions, ref employeeCodeName ,dbHelper);
                    order.EmployeeCodeName = employeeCodeName;
                    if (employeeId == -1)
                    {
                        message = ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Active_Employee, _requestRepository.LanguageId);
                        return result = GlobalErrors.Error;
                    }
                }
                if (isEcho)
                {
                    //in case Order comming from Echo the Order must have combination for customer outlet Ids 
                    order.OrderId = "CO-" + order.CustomerId + "-" + order.OutletId + "-" + order.OrderId.ToString().PadLeft(7, '0');
                }
                if (result == GlobalErrors.Success)
                {
                    result = _orderManager.SaveOrderHeaders(order, dbHelper, 5 , employeeId);
                }
                if (result == GlobalErrors.Success)
                {
                    result = _orderManager.SaveOrderDetails(order, dbHelper);
                }
                if (result == GlobalErrors.Success && AllowPromotion && order.CustomerPromotions != null && order.CustomerPromotions.Count > 0)
                {
                    if (_requestRepository.Configurations.PromotionCalculationSequence == PromotionCalculationSequence.SequentialLevelsFlow.GetHashCode() && order.PromotionsForAllLevels != null && order.PromotionsForAllLevels.Count > 0)
                    {
                        order.CustomerPromotions = order.PromotionsForAllLevels;
                    }
                    result = _promotionManager.SavePromotionBenefitHistoryData(CoreDataBaseConstants.TableNames.PromotionBenefitHistory,order.OrderId, order.Outlet.CustomerId, order.Outlet.OutletId, order.DivisionId, order.EmployeeId, order.CustomerPromotions, PromotionBenefitTransactionTypes.SalesOrder, dbHelper, isCustomerApp);
                }
                if (result == GlobalErrors.Success && order.Notes != null && order.Notes.Count > 0)
                {
                    result = _orderManager.SaveOrderNotes(order, dbHelper);
                }
                if (result == GlobalErrors.Success && order.RecurringTypeId > 0 && order.RecurringValue != null && !string.IsNullOrEmpty(order.RecurringValue.Trim()))
                {
                    result = _orderManager.SaveRecurringOrder(order, dbHelper);
                }
                if (result == GlobalErrors.Success && order.OrderStatus != OrderStatus.NewWebOrder && _requestRepository.Configurations.AllowAutoApproveOrderForCustomer(order.CustomerId,order.OutletId))
                {
                    order.PreviousOrderStatus = order.OrderStatus;
                    order.OrderStatus = OrderStatus.Approved;
                    result = ApproveOrder(order, ref message, dbHelper);
                }
                if (result == GlobalErrors.Success && isEcho && _requestRepository.Configurations.EnableCustomerAppPoints && _requestRepository.Configurations.PointsWeight > 0)
                {
                    //Save Customer Points
                    decimal val = order.NetTotal * _requestRepository.Configurations.PointsWeight;
                    result = _orderManager.UpdateCustomersPoints(order.OrderId, CustomerPointsAction.NewOrder, val, dbHelper);
                }
         
                //send notifaction if new order                
                if (result == GlobalErrors.Success && order.OrderStatus == OrderStatus.NewWebOrder && uploaded == 1)
                {
                    order.CreatedDate = DateTime.Now;
                    if (employeeId > -1)
                    {
                        result = _orderManager.GetCustomerCodeName(ref order , order.CustomerId);
                        _notificationManager.SendSonicNotifiactionToAndroid(order, employeeId, NotificationTypes.NewWebOrder , true);
                    }
                }
                if (_requestRepository.Configurations.ApplyApprovalCycleInWeb.Contains(((int)ApprovalCycleOperations.Orders).ToString()))
                {
                    if (result == GlobalErrors.Success && order.ApprovalStatus == -1)
                    {
                        int approvalOperationTypeId = (int)ApprovalCycleOperations.Orders;
                        order.OrderStatus = OrderStatus.Approved;
                        result = _orderManager.UpdateOrderStatus(order, dbHelper);

                        if (result == GlobalErrors.Success)
                        {
                            int approvalHistoryId = -1;
                            result = _pricesAndOffersManager.SaveInApprovalCycleHistory(-1, order.ApprovalStatus, approvalOperationTypeId, order.ApprovalStatus, ref approvalHistoryId, dbHelper);
                            if (result == GlobalErrors.Success)
                            {
                                result = _pricesAndOffersManager.SaveInOrderApprovalCycleHistory(approvalHistoryId, order.OrderId, order.CustomerId, order.OutletId, order.DivisionId, dbHelper);
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (commitNow && dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }


        public GlobalErrors EditOrderCRM(OrderCRMModel order)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                OrderModel orderModel = new OrderModel();
                orderModel.OrderDate = new DateTime();
                orderModel.Outlet = new CustomerOutletModel();
                orderModel.Outlet.CustomerId = int.Parse(EncryptionManager.Instance.DecryptData(order.CustomerIdSignature));
                orderModel.Outlet.OutletId = int.Parse(EncryptionManager.Instance.DecryptData(order.OutletIdSignature));
                order.SoldItems.ForEach(i =>
                {
                    i.RequestedQuantity = i.RequiredQty;
                });
                if (order.PromotedItems != null)
                {
                    order.PromotedItems.ForEach(i =>
                    {
                        i.RequestedQuantity = i.RequiredQty;
                    });
                }
                orderModel.SoldItems = order.SoldItems;
                orderModel.PromotedItems = order.PromotedItems;
                orderModel.OrderStatus = OrderStatus.New;
                orderModel.OrderTypeId = OrderTypes.Sales;
                orderModel.DivisionId = -1;
                orderModel.OrderId = order.OrderId;
                if (string.IsNullOrEmpty(order.OrderId))
                {
                    return GlobalErrors.Error;
                }
                orderModel.CreationSource = CreationSource.SonicCRM;
                orderModel.AllItems = new List<ItemPackModel>();
                orderModel.AllItems.AddRange(orderModel.SoldItems);
                orderModel.AllItems.AddRange(orderModel.PromotedItems);
                orderModel.OrderSourceId = -1;
                orderModel.ModifyReasonId = order.ModifyReasonId;
                orderModel.ModifySourceId = order.ModifySourceId;
                orderModel.EmployeeId = _requestRepository.CurrentOperator.EmployeeId;
                orderModel.DesiredDeliveryDate = order.DesiredDeliveryDateModel.Date;
                orderModel.Notes = new List<string>();
                orderModel.Remarks = order.Remarks;
                orderModel.CustomerPromotions = order.CustomerPromotions;
                if (order.Remarks != null && !string.IsNullOrEmpty(order.Remarks.Trim()))
                {
                    orderModel.Notes.Add(order.Remarks);
                }
                orderModel.NetTotal = order.NetTotal;
                orderModel.GrossTotal = order.GrossTotal;
                orderModel.Tax = order.Tax;
                orderModel.CalculatedRetailTax = order.CalculatedRetailTax;
                orderModel.Discount = order.Discount;

                orderModel.ResumeDateModel = order.ResumeDateModel;
                if (order.ResumeDateModel != null && order.ResumeDateModel.Date != DateTime.MinValue)
                {
                    orderModel.ResumeDate = order.ResumeDateModel.Date;
                }
                else
                {
                    orderModel.ResumeDate = DateTime.MinValue;
                }
                orderModel.RecurringChangedReasonId = order.RecurringChangedReasonId;
                orderModel.RecurringStatusId = order.RecurringStatusId;
                orderModel.RecurringComment = order.RecurringComment;
                orderModel.RecurringTypeId = order.RecurringTypeId;
                orderModel.RecurringValue = order.RecurringValue;
                orderModel.CreationSource = CreationSource.SonicCRM;

                orderModel.DeliveryChargesId = order.DeliveryChargesId;
                orderModel.DeliveryValue = order.DeliveryValue;
                orderModel.DeliveryTax = order.DeliveryTax;

                result = _orderManager.EditOrder(orderModel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public bool CheckOrderStatusBeforeAnyAction(OrderModel order)
        {
            try
            {
                return _orderManager.CheckOrderStatusBeforeAnyAction(order, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return false;
        }
        public bool CheckPreparedOrderStatusBeforeAnyAction(OrderModel order)
        {
            try
            {
                return _orderManager.CheckPreparedOrderStatusBeforeAnyAction(order);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return false;
        }
        public bool IsThereRelatedToTargetPromDependsOnThisOrder(OrderModel order  , ref string message)
        {
            bool isValid = true; 
            try
            {
                if (_promotionManager.IsAnyLaterTransactionAchievedRelatedToTargetPromotion(order))
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_You_Cannot_Void_Invoice_With_Related_Target_Prom, _requestRepository.LanguageId);
                    isValid = false ;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return isValid;
        }
        public GlobalErrors DeleteOrderCRM(OrderCRMModel order)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                OrderModel orderModel = new OrderModel();
                orderModel.OrderDate = new DateTime();
                orderModel.Outlet = new CustomerOutletModel();
                orderModel.Outlet.CustomerId = order.CustomerId;
                orderModel.Outlet.OutletId = order.OutletId;
                orderModel.OrderStatus = OrderStatus.Deleted;
                orderModel.DivisionId = -1;
                orderModel.OrderId = order.OrderId;
                if (string.IsNullOrEmpty(order.OrderId))
                {
                    return GlobalErrors.Error;
                }
                orderModel.ModifyReasonId = order.ModifyReasonId;
                orderModel.ModifySourceId = order.ModifySourceId;
                result = _orderManager.DeleteOrder(orderModel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public string ApproveSelectedOrders(ref int countOfFaild, List<OrderModel> orders)
        {
            GlobalErrors result = GlobalErrors.Success;
            StringBuilder messageResult = new StringBuilder();
            string message = string.Empty;
            List<int> deliveryPersons = new List<int>();
            List<int> divisionsList = new List<int>();
            try
            {
                foreach (OrderModel order in orders)
                {
                    var operation = new OrderOperationModel();
                    var originalStatus = order.OrderStatusId;
                    if (!_orderManager.CheckOrderStatusBeforeAnyAction(order, null))
                    {
                        countOfFaild++;
                        order.ValidationMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId); ;
                        messageResult.AppendLine(string.Format("  </br> </br>  {0} : {1} ", order.OrderId, order.ValidationMessage));
                        continue; // move to next order
                    }

                    var resultValidation = IsValidOperation(order, ref operation);
                    if (operation.IsValidOperation)
                    {
                        order.OrderStatusId = originalStatus;

                       
                        if (result == GlobalErrors.Success)
                        {
                            result = CheckAndAppoveOrders(order, ref message);
                            if (result == GlobalErrors.NotEnoughQuantity && string.IsNullOrEmpty(message))
                            {
                                message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Not_Enough_Stock_Order, _requestRepository.LanguageId);
                            }
                            else if (result != GlobalErrors.Success && string.IsNullOrEmpty(message))
                            {
                                message = ResourcesManager.TranslateKey(MessagesConstants.MessageErrorHappened, _requestRepository.LanguageId);
                            }
                        }
                        if (result != GlobalErrors.Success)
                        {
                            countOfFaild++;
                            order.ValidationMessage = message;
                            messageResult.AppendLine(string.Format("  </br> </br>  {0} : {1} ", order.OrderId, order.ValidationMessage));
                            continue; // move to next order
                        }
                    }
                    else
                    {
                        countOfFaild++;
                        order.ValidationMessage = operation.Message;
                        messageResult.AppendLine(string.Format("  </br> </br>  {0} : {1} ", order.OrderId, order.ValidationMessage));
                        continue; // move to next order
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return messageResult.ToString();
        }
        public GlobalErrors UpdateOrder(OrderModel order, ref string message)
        {
            var result = GlobalErrors.NotInitialized;
            List<int> deliveryPersons = new List<int>();
            List<int> divisionsList = new List<int>();

            try
            {
                if (!_orderManager.CheckOrderStatusBeforeAnyAction(order, null))
                {
                    result = GlobalErrors.DataAlreadyExists;
                    return result;
                }
                switch (order.OrderOperation)
                {
                    case OrderOperation.Approve:
                            result = GlobalErrors.Success;
                        if (result == GlobalErrors.Success)
                            result = CheckAndAppoveOrders(order, ref message);
                        if (result == GlobalErrors.NotEnoughQuantity)
                        {
                            if (string.IsNullOrEmpty(message))
                                message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Not_Enough_Stock_Order, _requestRepository.LanguageId);
                        }
                        break;
                    case OrderOperation.Cancel:
                        DBHelper<int> dbHelper = new DBHelper<int>();
                        dbHelper.BeginTransaction();
                        order.Outlet = new CustomerOutletModel();
                        order.Outlet.CustomerId = order.CustomerId;
                        order.Outlet.OutletId = order.OutletId;

                        // save cancel reason note without delete old saved notes
                        int maxNoteId = _orderManager.GetMaxOrderNoteId(order, dbHelper);
                        if (maxNoteId != -1)
                        {
                            result = _orderManager.AddOrderNote(order, order.Remarks, maxNoteId, dbHelper);
                        }
                        if (result == GlobalErrors.Success)
                        {
                            result = _orderManager.UpdateOrderStatus(order, dbHelper);
                        }
                        if (result == GlobalErrors.Success)
                        {
                            result = _orderManager.DeleteEchoFirstOrderCustomerOutlet(order.OrderId, dbHelper);
                        }
                        if (result == GlobalErrors.Success)
                        {
                            result = _orderManager.RemoveSSCCAssignmentFromOrder(order.OrderId, dbHelper);
                        }
                        if (result == GlobalErrors.Success && order.OrderTypeId == OrderTypes.Return)
                        {
                            result = _orderManager.DeleteFromForcedReturnInvoicesPacks(order.OrderId, order.CustomerId, order.OutletId, dbHelper);
                        }
                        if (result == GlobalErrors.Success && _requestRepository.Configurations.AllowCustomerAppLogin)
                        {
                            _notificationManager.SendOrderUpdatedNotificationToCustomerApp(order.CustomerId, order.OutletId, order.EmployeeId, order.OrderId, order.OrderStatus, dbHelper);
                        }
                        if (result == GlobalErrors.Success)
                        {
                            dbHelper.CommitTransaction();
                        }
                        else
                        {
                            dbHelper.RollBackTransaction();
                        }
                        break;
                    case OrderOperation.EditValues:
                        result = _orderManager.EditOrder(order);
                        break;
                    case OrderOperation.DisApprove:
                        result = _orderManager.UpdateOrderStatus(order);
                        break;
                    case OrderOperation.GetRejected:
                        result = ApproveRejectedOrder(order);
                        break;
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        private GlobalErrors ApproveRejectedOrder(OrderModel order)
        {
            //OrderModel selectedOrder = order;
            var result = GlobalErrors.NotInitialized;
            DBHelper<int> dBHelper = null;
            try
            {
                string voidedTransactionID = string.Empty;
                string message = string.Empty;
                dBHelper = new DBHelper<int>();
                dBHelper.BeginTransaction();
                if (order.OrderStatus == OrderStatus.Rejected)
                {
                    if (order.OrderTypeId == OrderTypes.ExchangeSales)
                    {
                        string sourceId = string.Empty;
                        result = _orderManager.GetExchangeReturnOrder(order, dBHelper, ref sourceId);
                        if (result == GlobalErrors.Success)
                        {
                            order.SourceOrderId = sourceId;
                        }
                    }
                    else
                    {
                        if (_requestRepository.Configurations.DeliveryMode.Contains(DeliveryModes.BOInvoicingAndDelivery.GetHashCode().ToString()) ||
                            _requestRepository.Configurations.DeliveryMode.Contains(DeliveryModes.BOInvoicingFODelivery.GetHashCode().ToString()))
                        {
                            TransactionModel transaction = new TransactionModel();
                            result = _orderManager.GetInvoiceOrderID(order, dBHelper, ref transaction);
                            if (transaction != null)
                            {
                                //result = _orderManager.VoidOrderTransaction(order, false, ref voidedTransactionID, dBHelper, ref message);
                            }
                            order.OrderStatus = OrderStatus.RejectCompleted;
                            result = _orderManager.UpdateOrderStatus(order);
                            if (result == GlobalErrors.Success)
                            {
                                result = _orderManager.UpdateDeliveryStatusForOrder(order, DeliveryStatus.RejectCompleted, dBHelper);
                            }
                        }

                    }
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dBHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dBHelper.CommitTransaction();
                    }
                    else
                    {
                        dBHelper.RollBackTransaction();
                    }
                    dBHelper.Dispose();
                    dBHelper = null;
                }
            }
        }
        public GlobalErrors EditRecurringOrder(OrderCRMModel order)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.UpdateRecurringOrder(order);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        private GlobalErrors ApproveOrder(OrderModel order, ref string message, DBHelper<int> dbHelper)
        {
            var result = GlobalErrors.Error;
            try
            {
                if (CheckBeforeApproval(order, dbHelper, ref message, -1, string.Empty, order.OrderOperation) == GlobalErrors.Success) // TODO, do not repeat check if it occured previously in approval cycle validations.
                {
                    result = _orderManager.UpdateOrderStatus(order, dbHelper);

                    if (_requestRepository.Configurations.AllowAutoDispatchOrderForCustomer(order.CustomerId, order.OutletId))
                    {
                        List<OrderModel> Orders = new List<OrderModel>();
                        order.DriverId = order.EmployeeId;
                        order.ScheduleDateModel = order.DesiredDeliveryDateModel;
                        Orders.Add(order);
                        result = AssignOrdersToDriver(Orders, dbHelper, ref message);
                    }
                    else if (result == GlobalErrors.Success && _requestRepository.Configurations.AllowCustomerAppLogin)
                    {
                        _notificationManager.SendOrderUpdatedNotificationToCustomerApp(order.CustomerId, order.OutletId, order.EmployeeId, order.OrderId, order.OrderStatus, dbHelper);
                    }
                }

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        private GlobalErrors ApproveReturnOrder(OrderModel order, ref string message, DBHelper<int> dbHelper)
        {
            var result = GlobalErrors.Error;

            try
            {
                if (!_requestRepository.Configurations.AutoAssignReturnOrder || order.CreationSource != CreationSource.InFrontOffice)
                {
                    order.OrderStatus = OrderStatus.Approved;
                    result = _orderManager.UpdateOrderStatus(order, dbHelper);
                }
                else
                if (_requestRepository.Configurations.AutoAssignReturnOrder && order.CreationSource == CreationSource.InFrontOffice && order.EmployeeId > -1)
                {
                    List<OrderModel> orders = new List<OrderModel>();
                    int driverId = order.EmployeeId;
                    var date = order.DesiredDeliveryDateModel;
                    var today = DateTime.Now;
                    order.DriverId = driverId;
                    order.ScheduleDateModel = date;
                    if (today.Day == date.Day && today.Month == date.Month && today.Year == date.Year)
                    {
                        order.EnableSendNotification = true;
                    }
                    else
                    {
                        order.EnableSendNotification = false;
                    }
                    orders.Add(order);
                    result = AssignOrdersToDriver(orders, dbHelper, ref message);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        private GlobalErrors CheckBeforeApproval(OrderModel order, DBHelper<int> dbHelper, ref string message, int currentLevel, string currentOrderIds,OrderOperation orderOperation)
        {
            var result = GlobalErrors.Success;
            bool isApprove = false;
            try
            {
                if(orderOperation == OrderOperation.Approve)
                {
                    isApprove = true;
                }
                //if (_requestRepository.Configurations.ForceStockCheckingBeforeApprovingOrder && order.OrderTypeId != OrderTypes.Return)
                //    result = _orderManager.CheckStockAvailableQuantityForOrder(order, dbHelper, ref message, currentLevel, currentOrderIds, isApprove);

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public  GlobalErrors GetMainWarehouseForEmployeeAsString(ref string warehouseIds)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                return _orderManager.GetMainWarehouseForEmployeeAsString(ref warehouseIds);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

            private TransactionCommonData FillTransactionCommonData(OrderModel order)
        {
            TransactionCommonData transationCommonData = new TransactionCommonData();
            try
            {
                transationCommonData.Outlet = order.Outlet;
                transationCommonData.NetTotal = order.NetTotal;
                transationCommonData.OrderStatus = order.OrderStatus;
                transationCommonData.ItemFormProcessMode = ItemFormProcessMode.Delivery;
                transationCommonData.CustomerId = order.CustomerId;
                transationCommonData.OutletId = order.OutletId;
                transationCommonData.DivisionId = order.DivisionId;
                transationCommonData.EmployeeId = order.DriverId;
                transationCommonData.SalesMode = order.SalesMode;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return transationCommonData;
        }

        public GlobalErrors IsValidToVoidOrderInvoice(ref string message, OrderModel order)
        {
            var result = GlobalErrors.Success;
            var deliveryAssignment = new DeliveryAssignmentModel();
            try
            {
                result = _orderManager.GetLastOrderStatus(ref order,null);

                result = _orderManager.GetOrderDeliveryDetails(order, ref deliveryAssignment);

                if (result != GlobalErrors.Success) return GlobalErrors.Error;
                bool deliveryAssignmentUploaded = _orderManager.IsDeliveryAssignmentUploaded(order);

                bool isAnyTransactionAchievedPromotion = _promotionManager.IsAnyTransactionAchievedPromotion(order);
                bool isAnyTransactionAchievedRelatedToTargetPromotion = _promotionManager.IsAnyLaterTransactionAchievedRelatedToTargetPromotion(order);
                bool isAnyReturnsDependantOnTransaction = _transactionManager.IsAnyReturnDependantOnOrderTransaction(order);
                bool isDeliveredFromFO = _transactionManager.IsOrderDeliveredInFO(order);
                if (order.OrderStatus != OrderStatus.Invoiced && order.OrderStatus != OrderStatus.Delivered)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Invoiced_Or_Delivered, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                if (isDeliveredFromFO)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Cannot_Void_Tran_Delivered_In_FO, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                if (isAnyTransactionAchievedPromotion)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_You_Cannot_Void_Invoice_Previous_Ach, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                if (isAnyTransactionAchievedRelatedToTargetPromotion)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_You_Cannot_Void_Invoice_With_Related_Target_Prom, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                if (deliveryAssignmentUploaded)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Invoice_Uploaded, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                if (isAnyReturnsDependantOnTransaction)
                {
                    message = ResourcesManager.TranslateKey(MessagesConstants.Desc_Cannot_Void_Transaction_Used_In_Return, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }



        public GlobalErrors GetOrderSources(ref List<OrderSourceModel> sourceList)
        {
            try
            {
                return _orderManager.GetOrderSources(ref sourceList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetDeliveryChargesList(int orderSourceID, ref List<DeliveryChargesModel> deliveryChargesList)
        {
            try
            {
                return _orderManager.GetDeliveryChargesList(orderSourceID, ref deliveryChargesList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetOrderModificationReasons(ref List<OrderSourceModel> list, OrderModificationTypes type)
        {
            try
            {
                return _orderManager.GetOrderModificationReasons(ref list, type);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetOrderModificationSources(ref List<OrderSourceModel> list, OrderModificationTypes type)
        {
            try
            {
                return _orderManager.GetOrderModificationSources(ref list, type);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetOrderRecurringChangeReason(ref List<RecurringChangedReasoneModel> reasonList)
        {
            try
            {
                return _orderManager.GetOrderRecurringChangeReason(ref reasonList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public string GetMaxOrderId()
        {
            try
            {
                return _documentSequenceService.GetMaxDocumentSequence(_requestRepository.CurrentOperator.EmployeeId, DocumentTypes.Order, -1);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return string.Empty;
            }
        }

        public GlobalErrors GetLastOrderDetails(string orderId, int customerId, int outletId, ref List<ItemPackModel> list)
        {
            try
            {
                return _orderManager.GetLastOrderDetails(orderId, customerId, outletId, ref list);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetSuggestedOrderItems(int customerId, int outletId, ref List<ItemPackModel> list)
        {
            try
            {
                return _orderManager.GetSuggestedOrderItems(customerId, outletId, ref list);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetStockStatuses(ref List<SalesOrderTypeModel> list)
        {
            try
            {
                return _orderManager.GetStockStatuses(ref list);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors IsValidOperation(OrderModel order, ref OrderOperationModel orderOperation)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                switch (order.OrderOperation)
                {
                    case OrderOperation.Add:
                        string message = string.Empty;
                        result = CheckBeforeApproval(order, null, ref message, -1, string.Empty, order.OrderOperation);
                        if (result != GlobalErrors.Success)
                        {
                            orderOperation.Message = message;
                            orderOperation.OnlyShowWarning = true;
                            orderOperation.IsValidOperation = false;

                        }
                        else
                        {
                            orderOperation.IsValidOperation = true;
                        }

                        break;
                    case OrderOperation.EditValues:
                    case OrderOperation.Approve:
                    case OrderOperation.Cancel:
                        result = IsValidToEdit(order, ref orderOperation);
                        break;
                }

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        private GlobalErrors IsValidToEdit(OrderModel order, ref OrderOperationModel orderOperation)
        {
            var result = GlobalErrors.Success;
            try
            {
                string message = string.Empty;
                result = _orderManager.GetLastOrderStatus(ref order,null);

                if (order.OrderStatus == OrderStatus.New || order.OrderStatus == OrderStatus.Approved)
                {
                   
                  
                }
                else
                {
                    orderOperation.IsValidOperation = false;
                    orderOperation.Message = "order status not compatable with operation";
                    return result;
                }

                orderOperation.IsValidOperation = true;
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetOrderDetailsCRM(int customerId, int outletId, string orderId, ref OrderCRMModel orderModel)
        {
            try
            {
                return _orderManager.GetOrderDetailsCRM(customerId, outletId, orderId, ref orderModel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetRecurringHistory(ref List<RecurringModel> list)
        {
            try
            {
                return _orderManager.GetRecurringHistory(ref list);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                list = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetRecurringHistory(int customerId, int outletId, ref SharedTableResult<RecurringModel> list)
        {
            try
            {
                return _orderManager.GetRecurringHistory(customerId, outletId, ref list);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                list = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetRecurringDetails(int customerId, int outletId, string orderId, int recurId, ref OrderCRMModel order)
        {
            try
            {
                return _orderManager.GetRecurringDetails(customerId, outletId, orderId, recurId, ref order);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                order = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetRecurringDetails(RecurringModel recurringOrder, ref OrderCRMModel order)
        {
            try
            {
                return _orderManager.GetRecurringDetails(recurringOrder, ref order);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                order = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors CheckSelectedDateHasOrder(OrderCRMModel order, DateTime date, bool checkOrdersForToday, ref OrderDateValidationResponse resultData)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                int customerId = int.Parse(EncryptionManager.Instance.DecryptData(order.CustomerIdSignature));
                int outletId = int.Parse(EncryptionManager.Instance.DecryptData(order.OutletIdSignature));
                result = _orderManager.CheckSelectedDateHasOrder(order.OrderId, customerId, outletId, date, checkOrdersForToday, ref resultData);
            }
            catch (Exception ex)
            {
                string orderSig = "Order Signature : " + (order == null ? "Null" : order.CustomerIdSignature + " ## " + order.OutletIdSignature);
                string deliveryD = "Order DesiredDeliveryDateModel : " + (order.DesiredDeliveryDateModel == null ? "Null" : order.DesiredDeliveryDateModel.Date + "");
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, _requestRepository.CurrentOperator.EmployeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, orderSig, null, "", _requestRepository.CurrentOperator.EmployeeId);
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, deliveryD, null, "", _requestRepository.CurrentOperator.EmployeeId);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetEmployeeAssignedOrders(int employeeID, ref List<OrderModel> orderModels)
        {
            try
            {
                return _orderManager.GetEmployeeAssignedOrdersList(employeeID, ref orderModels);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors ChangeOrderAssignment(int employeeID, string changeDriverOrders, DBHelper<int> dbHelper)
        {
            try
            {
                return _orderManager.UpdateOrderAssignment(employeeID, changeDriverOrders, dbHelper);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors DeleteOrderAssignment(string changeDriverOrders, DBHelper<int> dbHelper)
        {
            try
            {
                return _orderManager.DeleteOrderAssignment(changeDriverOrders, dbHelper);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors UpdateOrderStatus(string orderID, int orderStatusID, int previousOrderStatusID, DBHelper<int> dbHelper)
        {
            try
            {
                return _orderManager.UpdateOrderStatus(orderID, orderStatusID, previousOrderStatusID, dbHelper);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors DeleteMultiOrders(List<OrderModel> ordersList, ref string errorMessage, ref int countOfFaild)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            StringBuilder messageResult = new StringBuilder();
            try
            {

                foreach (OrderModel order in ordersList)
                {
                    var operationOrder = new OrderOperationModel();
                    if (!_orderManager.CheckOrderStatusBeforeAnyAction(order, null))
                    {
                        countOfFaild++;
                        order.ValidationMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Order_Status_Changed, _requestRepository.LanguageId);
                        messageResult.AppendLine(string.Format(" {0} : {1}  </br>  ", order.OrderId, order.ValidationMessage));
                        continue; // move to next order
                    }

                    var resultValidation = IsValidOperation(order, ref operationOrder);

                    if (operationOrder.IsValidOperation)
                    {
                        dbHelper = new DBHelper<int>();
                        dbHelper.BeginTransaction();
                        order.Outlet = new CustomerOutletModel();
                        order.Outlet.CustomerId = order.CustomerId;
                        order.Outlet.OutletId = order.OutletId;
                        order.OrderStatusId = OrderStatus.Canceled.GetHashCode();
                        // save cancel reason note without delete old saved notes
                        int maxNoteId = _orderManager.GetMaxOrderNoteId(order, dbHelper);
                        if (maxNoteId != -1)
                        {
                            result = _orderManager.AddOrderNote(order, order.Remarks, maxNoteId, dbHelper);
                            if (result == GlobalErrors.Success)
                            {
                                result = _orderManager.UpdateOrderStatus(order, dbHelper);
                            }
                        }
                        else
                        {
                            result = GlobalErrors.Error;
                        }

                        if (dbHelper != null)
                        {
                            if (result == GlobalErrors.Success)
                            {
                                dbHelper.CommitTransaction();
                            }
                            else
                            {
                                dbHelper.RollBackTransaction();
                            }
                            dbHelper.Dispose();
                            dbHelper = null;
                        }
                        if (result != GlobalErrors.Success)
                        {
                            return result;
                        }


                    }
                    else
                    {
                        countOfFaild++;
                        order.ValidationMessage = operationOrder.Message;
                        messageResult.AppendLine(string.Format(" {0} : {1}  </br>  ", order.OrderId, order.ValidationMessage));
                        continue; // move to next order

                    }
                }
                errorMessage = messageResult.ToString();
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        //public string DeleteMultiOrders(ref int countOfFaild,List<OrderModel> ordersList)
        //{
        //    GlobalErrors result = GlobalErrors.NotInitialized;
        //    DBHelper<int> dbHelper = null;
        //    StringBuilder messageResult = new StringBuilder();
        //    try
        //    {
        //        dbHelper = new DBHelper<int>();
        //        dbHelper.BeginTransaction();
        //        foreach (OrderModel order in ordersList)
        //        {
        //            var operationOrder = new OrderOperationModel();
        //            var resultValidation = IsValidOperation(order, ref operationOrder);
        //            if (operationOrder.IsValidOperation)
        //            {

        //                order.Outlet = new CustomerOutletModel();
        //                order.Outlet.CustomerId = order.CustomerId;
        //                order.Outlet.OutletId = order.OutletId;
        //                // save cancel reason note without delete old saved notes
        //                int maxNoteId = _orderManager.GetMaxOrderNoteId(order, dbHelper);
        //                if (maxNoteId != -1)
        //                {
        //                    result = _orderManager.AddOrderNote(order, order.Remarks, maxNoteId, dbHelper);
        //                    if (result == GlobalErrors.Success)
        //                    {
        //                        result = _orderManager.UpdateOrderStatus(order, dbHelper);
        //                    }
        //                }
        //                else
        //                {
        //                    result = GlobalErrors.Error;
        //                }
        //                //if (result != GlobalErrors.Success)
        //                //{
        //                //    return result;
        //                //}
        //            }
        //            else
        //            {
        //                countOfFaild++;
        //                order.ValidationMessage = operationOrder.Message;
        //                messageResult.AppendLine(string.Format("  </br> </br>  {0} : {1} ", order.OrderId, order.ValidationMessage));
        //                continue; // move to next order
        //            }
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        result = GlobalErrors.Error;
        //    }
        //    finally
        //    {
        //        if (dbHelper != null)
        //        {
        //            if (result == GlobalErrors.Success)
        //            {
        //                dbHelper.CommitTransaction();
        //            }
        //            else
        //            {
        //                dbHelper.RollBackTransaction();
        //            }
        //            dbHelper.Dispose();
        //            dbHelper = null;
        //        }
        //    }
        //    return messageResult.ToString();
        //}
        public GlobalErrors DispatchReturnOrder(OrderModel order, ref bool IsAnyDeliveryAssignmentUploaded, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                // Check if delivery assignment is uploaded
                result = _orderManager.IsAnyDeliveryAssignmentUploadedPerSelectedOrder(order, ref IsAnyDeliveryAssignmentUploaded, dbHelper);
                if (result == GlobalErrors.Success)
                {
                    if (IsAnyDeliveryAssignmentUploaded)
                    {
                        return GlobalErrors.Error;
                    }

                    int helperId = -1, driverId = -1, supervisorId = -1, salesRepId = -1, salesManagerId = -1;
                    _employeeManager.GetEmployeeRelatedStaff(order.DriverId, dbHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepId, ref salesManagerId);
                    order.HelperId = helperId;
                    order.SupervisorId = supervisorId;
                    order.SalesRepId = salesRepId;
                    order.SalesManagerId = salesManagerId;
                    order.OrganizationId = _employeeManager.GetEmployeeOrganization(order.DriverId, dbHelper);

                    result = _orderManager.InsertIntoDeliveryAssignment(order, dbHelper);
                    if (result == GlobalErrors.Success)
                    {
                        order.PreviousOrderStatus = order.OrderStatus;
                        order.OrderStatus = OrderStatus.Assigned;
                        result = _orderManager.UpdateOrderStatus(order, dbHelper);
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetOrderTypes(List<int> orderTypesFilter, ref List<OrderTypeModel> orderTypes)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.GetOrderTypes(orderTypesFilter, ref orderTypes);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }


        public GlobalErrors CheckResellableItems(List<int> orderItemsPackStatuses, ref bool isContainResellable)
        {
            try
            {
                string packStatuses = string.Join(",", orderItemsPackStatuses.Distinct().ToList());
                if(!string.IsNullOrEmpty(packStatuses))
                {
                    return _orderManager.CheckResellableItems(packStatuses, ref isContainResellable);
                } 
                else
                {
                    return GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }


        public GlobalErrors CheckForPostedCustomerTransactions(OrderModel order, ref bool showWaitForUpdateWarning)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                bool hasPostedTransactions = false;
                result = _orderManager.CheckForPostedCustomerTransactions(order, ref hasPostedTransactions);
                if(result == GlobalErrors.Success)
                {
                    if(hasPostedTransactions)
                    {
                        //List<ItemPackModel> updatedItems = new List<ItemPackModel>();
                        //result = itemService.UpdateCustomerReturnBatches(order, ref updatedItems);
                        showWaitForUpdateWarning = true;

                    }
                    else
                    {
                        showWaitForUpdateWarning = false;
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }


        public GlobalErrors GetCollectedOrderForView(OrderListFilter filter, ref OrderModel order)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.GetCollectedOrderForView(filter, ref order);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors IsValidToApprove(List<OrderModel> orders, ref List<OrderModel> validOrdersToApprove, ref List<OrderModel> invalidOrdersToApprove, ref string message)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                StringBuilder messageResult = new StringBuilder();


                List<int> approvalcycleValidations = null;
                int approvalOperationTypeId = (int)ApprovalCycleOperations.Orders;
                bool isValid = false;
                
                List<string> ids = new List<string>();
                string currentOrderIds = string.Empty;
                orders.ForEach(x =>
                {
                    ids.Add("'" + x.OrderId + "'");
                });
                currentOrderIds = string.Join(",", ids);
                
                foreach (OrderModel order in orders)
                {
                    string validationMessage = string.Empty;
                    approvalcycleValidations = new List<int>();
                    string organizationFilter = string.Empty;
                    string restrictApprovalCycleToOrganizationLevel = _requestRepository.Configurations.RestrictApprovalCycleToOrganizationLevel;
                    if (!string.IsNullOrEmpty(restrictApprovalCycleToOrganizationLevel) && restrictApprovalCycleToOrganizationLevel.Contains(((int)ApprovalCycleOperations.Orders).ToString()))
                    {
                        if (order.OrganizationId > -1)
                        {
                            organizationFilter = $@" AND ApprovalCycleLevels.SelectedOrganizationID = {order.OrganizationId}";
                        }
                    }
                    int currentLevel = -1;
                    result = _pricesAndOffersManager.IsValidToApprove(order.ApprovalStatus, approvalOperationTypeId, -1, organizationFilter, ref isValid, ref approvalcycleValidations, ref currentLevel);
                    if (result != GlobalErrors.Success) break;

                    if (isValid)
                    {
                        // check approval cycle validations....
                        result = RunOrderApprovalCycleLevelValidations(approvalcycleValidations,  order, ref isValid, ref validationMessage, currentLevel, currentOrderIds, order.OrderOperation);
                        if (result != GlobalErrors.Success) break;

                        if (isValid) 
                        {
                            //result = _pricesAndOffersManager.ApproveOperation(-1, approvalOperationTypeId, ref actualNewLevel, ref newStatus, organizationFilter, null);
                            //if (result != GlobalErrors.Success) break;
                            //order.ApprovalStatus = newStatus;
                            validOrdersToApprove.Add(order);

                        }
                        else
                        {
                            messageResult.AppendLine(string.Format(" {0} : {1}  </br>  ", order.OrderId, validationMessage));
                            invalidOrdersToApprove.Add(order);
                        }
                    }
                    else 
                    {
                        order.ValidationMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Not_Valid_To_Approve, _requestRepository.LanguageId);
                        messageResult.AppendLine(string.Format(" {0} : {1}  </br>  ", order.OrderId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Not_Valid_To_Approve, _requestRepository.LanguageId)));
                        invalidOrdersToApprove.Add(order);
                    }
                }

                message = messageResult.ToString();
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors RunOrderApprovalCycleLevelValidations(List<int> validationIds, OrderModel order, ref bool isValid, ref string validationMessage, int currentLevel, string currentOrderIds, OrderOperation orderOperation)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                TransactionCommonData transactionCommonData = new TransactionCommonData();
                transactionCommonData.ItemFormProcessMode = ItemFormProcessMode.Orders;
                transactionCommonData.DivisionId = order.DivisionId;
                transactionCommonData.Outlet = _customerManager.FillOutletData(order.CustomerId, order.OutletId, order.DivisionId);
                transactionCommonData.SalesMode = order.SalesMode;
                transactionCommonData.NetTotal = LocalUtilities.GetRoundedDecimal(order.NetTotal, _requestRepository.Configurations.NumberOfDigits);
                if (validationIds.Contains((int)ApprovalCycleValidations.PaymentTerm))
                {
                    isValid = !_validationService.IsCustomerViolatePaymentTerms(transactionCommonData);
                    if (!isValid)
                    {
                        order.ValidationMessage = string.Format(ResourcesManager.TranslateKey(MessagesConstants.Desc_Payment_term_Message, _requestRepository.LanguageId),
                             ResourcesManager.TranslateKey(MessagesConstants.Desc_Customer_Violated_Payment_Term, _requestRepository.LanguageId),
                             transactionCommonData.Outlet.CustOutPaymentTerm.PaymentTermDescription);
                    }
                }
                if (isValid && validationIds.Contains((int)ApprovalCycleValidations.CreditLimit))
                {
                    if (_requestRepository.Configurations.DeductPendingOrdersFromAvailableCredit)
                    {
                        transactionCommonData.DraftTransactionId = order.OrderId;
                    }
                    isValid = !_validationService.IsExceedsCustomerLimit(transactionCommonData);
                    if (!isValid)
                    {
                        order.ValidationMessage = string.Format(ResourcesManager.TranslateKey(MessagesConstants.Desc_Excced_Credit_Limit_Message, _requestRepository.LanguageId),
                            ResourcesManager.TranslateKey(MessagesConstants.Desc_Exceed_Credit_Limit, _requestRepository.LanguageId),
                            Math.Round(transactionCommonData.Outlet.Account.CreditLimit),
                            Math.Round(transactionCommonData.Outlet.Account.Balance),
                            Math.Round(transactionCommonData.Outlet.Account.PendingOrdersTotal));
                    }
                }
                if (isValid && validationIds.Contains((int)ApprovalCycleValidations.OnHoldStatus))
                {
                    isValid = !_validationService.IsCustomerOnHold(transactionCommonData);
                    if (!isValid)
                    {
                        order.ValidationMessage = ResourcesManager.TranslateKey(MessagesConstants.Error_Customer_OnHold, _requestRepository.LanguageId);
                    }
                }
                if (isValid && validationIds.Contains((int)ApprovalCycleValidations.WarehouseStock))
                {
                    result = CheckBeforeApproval(order, null, ref validationMessage, currentLevel, currentOrderIds, orderOperation);
                    if (result == GlobalErrors.NotEnoughQuantity) 
                    {
                        isValid = false;
                        result = GlobalErrors.Success;
                    }
                    if (result == GlobalErrors.Success) 
                    {
                        order.SkipCheckQuantityBeforeApproval = true;
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors IsDeliveryAssignmentUploaded(List<OrderModel> orders)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                foreach(OrderModel order in orders)
                {
                    if(_orderManager.IsDeliveryAssignmentUploaded(order, null))
                    {
                        order.IsUploaded = true;
                    }
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }

        #region APIs

        public GlobalErrors GetOrderStatus(string OrderNumber, ref OrderStatusAPIModel Status)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.GetOrderStatus(OrderNumber, ref Status);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetOrdersHistory(string SiteCode, string From_Date, string To_Date, ref List<OrderAPIModel> OrderList)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            List<ItemPackAPIModel> ItemsList = new List<ItemPackAPIModel>();
            try
            {
                result = _orderManager.GetOrdersList(SiteCode, From_Date, To_Date, ref OrderList);

                if (result == GlobalErrors.Success && OrderList != null && OrderList.Count > 0)
                {
                    foreach (OrderAPIModel order in OrderList)
                    {
                        ItemsList = new List<ItemPackAPIModel>();
                        result = _orderManager.GetOrderItems(SiteCode, order.OrderNumber, ref ItemsList);
                        if (result != GlobalErrors.Success)
                            return GlobalErrors.Error;

                        order.Details = ItemsList;
                        order.TotalItems = ItemsList.Count;
                    }
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors SaveOrderFromAPI(SendOrderAPIModel order)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                // Saving
                result = _orderManager.SaveOrderFromAPI(order);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetSiteUpdatedProducts(string SiteCode, ref ProductsAPIModel Products)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.GetSiteUpdatedProducts(SiteCode, ref Products);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetUpdatedProducts(string PriceListCode, ref ProductsAPIModel Products)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _orderManager.GetUpdatedProducts(PriceListCode, ref Products);
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors PrepareEDIOrder(EDIOrderInputModel order, ref EDIOrderOutputModel orderProcessingResult)
        {
            var result = GlobalErrors.NotInitialized;
            int customerID = -1, outletID = -1, loggedInUserID = -1;
            List<ItemPackModel> requiredItems = new List<ItemPackModel>();
            decimal transactionNetTotal = 0, grossTotal = 0, taxTotal = 0, exciseTaxTotal = 0, discountTotal = 0, headerDiscountTotal = 0, promotedDiscount = 0;
            OrderModel orderModel = new OrderModel();
            var takePromotions = false;
            List<Promotion> customerPromotions = new List<Promotion>();
            try
            {
                #region Preparing Data
                result = _customerManager.GetIDsByCustomerCode(order.SiteCode, ref customerID, ref outletID);
                if (customerID < 0 || outletID < 0)
                {
                    orderProcessingResult.Message = string.Format(MessagesConstants.Error_Wrong_Site_Code, order.SiteCode);
                    return GlobalErrors.Error;
                }
                if (_requestRepository.CurrentOperator != null)
                    loggedInUserID = _requestRepository.CurrentOperator.EmployeeId;
                CustomerOutletModel customerOutlet = _customerManager.FillOutletData(customerID, outletID, -1);
                if (order.AllItems != null && order.AllItems.Count > 0)
                {
                    var anyDuplicateCode = order.AllItems.GroupBy(requestedItem => requestedItem.ItemCode).Any(code => code.Skip(1).Any());
                    if (anyDuplicateCode)
                    {
                        orderProcessingResult.Message = "Requested Order has duplicate items";
                        return GlobalErrors.Error;
                    }
                    List<ItemPackModel> allAvailableItems = new List<ItemPackModel>();
                    result = itemService.GetAllItemsList(-1, customerID, outletID, ref allAvailableItems);
                    if (result != GlobalErrors.Success || allAvailableItems.Count == 0)
                    {
                        orderProcessingResult.Message = MessagesConstants.Error_Cannot_Retrieve_Items;
                        return GlobalErrors.Error;
                    }
                    requiredItems = allAvailableItems.Where(Item => order.AllItems.Any(APIItem => APIItem.ItemCode.Equals(Item.ItemCode)))
                    .ToList();
                    if (requiredItems == null || requiredItems.Count <= 0)
                    {
                        orderProcessingResult.Message = MessagesConstants.Error_Cannot_Retrieve_Items;
                        return GlobalErrors.Error;
                    }
                    if (requiredItems.Count != order.AllItems.Count)
                    {
                        var noneExsistanceItem = order.AllItems.Where(orderedItem => requiredItems.Any(reqItem => reqItem.ItemCode != orderedItem.ItemCode)).First();
                        orderProcessingResult.Message = string.Format(MessagesConstants.Error_Item_Codes_DNE, noneExsistanceItem.ItemCode);
                        return GlobalErrors.Error;
                    }
                    var itemZeroPriced = requiredItems.Where(reqItem => reqItem.Price <= 0).FirstOrDefault();
                    if (itemZeroPriced != null)
                    {
                        orderProcessingResult.Message = string.Format(MessagesConstants.Error_Item_Has_Zero_Price, itemZeroPriced.ItemCode);
                        return GlobalErrors.Error;
                    }
                    requiredItems.ForEach(item =>
                    {
                        var x = order.AllItems.Find(apiItem => apiItem.ItemCode == item.ItemCode);
                        if (x != null)
                        {
                            item.RequiredQty = x.Quantity;
                            item.RequestedQuantity = x.Quantity;
                            item.SalesTransactionTypeId = SalesTransactionTypes.Sales;
                            item = transactionManager.CalculatePackTotals(item);
                            transactionNetTotal += item.NetTotal;
                            grossTotal += item.GrossTotal;
                            taxTotal += item.CalculatedTax;
                            exciseTaxTotal += item.CalculatedRetailTax;
                            discountTotal += item.CalculatedDiscountTotal;
                            headerDiscountTotal += item.HeaderDiscount;
                        }
                    });
                }
                else
                {
                    orderProcessingResult.Message = MessagesConstants.Error_NoItems_InEDIOrder;
                    return GlobalErrors.Error;
                }
                #endregion
                #region Fill Transaction Common Data
                var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;
                TransactionCommonData transactionCommonData = new TransactionCommonData();
                transactionCommonData.CustomerId = customerID;
                transactionCommonData.OutletId = outletID;
                transactionCommonData.Outlet = customerOutlet;
                transactionCommonData.NetTotal = LocalUtilities.GetRoundedDecimal(transactionNetTotal, numberOfDigits);
                transactionCommonData.GrossTotal = LocalUtilities.GetRoundedDecimal(grossTotal, numberOfDigits);
                transactionCommonData.Tax = LocalUtilities.GetRoundedDecimal(taxTotal, numberOfDigits);
                transactionCommonData.ExciseTax = LocalUtilities.GetRoundedDecimal(exciseTaxTotal, numberOfDigits);
                transactionCommonData.OrderId = _documentSequenceService.GetMaxDocumentSequence(loggedInUserID, DocumentTypes.Order, -1);
                transactionCommonData.ItemFormProcessMode = ItemFormProcessMode.Orders;
                transactionCommonData.DraftTransactionId = transactionCommonData.OrderId;
                transactionCommonData.SoldItems = requiredItems;
                #endregion
                result = ValidateEDIOrder(transactionCommonData, ref orderProcessingResult);
                if (result == GlobalErrors.Success)
                {
                    result = validationService.CheckPromotions(transactionCommonData, ref customerPromotions, true);
                    if (result == GlobalErrors.Success && customerPromotions.Count > 0)
                    {
                        transactionCommonData.CustomerPromotions = customerPromotions;
                        result = promotionService.TakePromotions(ref transactionCommonData, true);
                        if (result == GlobalErrors.Success)
                        {
                            takePromotions = transactionCommonData.CustomerPromotions.Any(promotion => promotion.IsTaken);
                            if (takePromotions)
                            {
                                // to calculate net totals again after taking promotions
                                transactionNetTotal = 0; grossTotal = 0; exciseTaxTotal = 0; discountTotal = 0; headerDiscountTotal = 0; taxTotal = 0;
                                transactionCommonData.SoldItems.ForEach(item =>
                                {
                                    item = transactionManager.CalculatePackTotals(item);
                                    item.SalesTransactionTypeId = SalesTransactionTypes.Sales;
                                    transactionNetTotal += item.NetTotal;
                                    grossTotal += item.GrossTotal;
                                    taxTotal += item.CalculatedTax;
                                    exciseTaxTotal += item.CalculatedRetailTax;
                                    discountTotal += item.CalculatedDiscountTotal;
                                    headerDiscountTotal += item.HeaderDiscount;
                                    promotedDiscount += item.PromotedDiscount;
                                });
                            }
                            if (result == GlobalErrors.Success && transactionCommonData.PromotedItems.Count > 0)
                            {
                                orderModel.PromotedItems = transactionCommonData.PromotedItems;
                                orderModel.AllItems = transactionCommonData.SoldItems;
                                orderModel.PromotedItems.ForEach(promotedItem =>
                                {
                                    orderModel.AllItems.Add(promotedItem);
                                });
                            }
                        }
                        else
                        {
                            orderProcessingResult.Message = MessagesConstants.Error_Taking_Promotions;
                            return GlobalErrors.Error;
                        }
                    }
                    else
                    {
                        orderModel.AllItems = transactionCommonData.SoldItems;
                    }
                    #region FillOrder
                    orderModel.OrderId = transactionCommonData.OrderId;
                    orderModel.NetTotal = LocalUtilities.GetRoundedDecimal(transactionNetTotal, numberOfDigits);
                    orderModel.LPO = order.PoNumber;
                    orderModel.GrossTotal = LocalUtilities.GetRoundedDecimal(grossTotal, numberOfDigits);
                    orderModel.SoldItems = requiredItems;
                    orderModel.CustomerId = customerID;
                    orderModel.OutletId = outletID;
                    orderModel.CreatedBy = loggedInUserID;
                    orderModel.CreationSource = CreationSource.SonicAPI;
                    orderModel.DesiredDeliveryDate = DateTime.Now.DayOfWeek == DayOfWeek.Thursday ? DateTime.Today.AddDays(2) : DateTime.Today.AddDays(1);
                    orderModel.DivisionId = -1;
                    orderModel.EmployeeId = loggedInUserID;
                    orderModel.OrderDate = DateTime.Now;
                    orderModel.OrderStatus = OrderStatus.New;
                    orderModel.Discount = LocalUtilities.GetRoundedDecimal(discountTotal, numberOfDigits);
                    orderModel.Tax = LocalUtilities.GetRoundedDecimal(taxTotal, numberOfDigits);
                    orderModel.HeaderDiscount = LocalUtilities.GetRoundedDecimal(headerDiscountTotal, numberOfDigits);
                    orderModel.CalculatedRetailTax = LocalUtilities.GetRoundedDecimal(exciseTaxTotal, numberOfDigits);
                    orderModel.PromotedDiscount = LocalUtilities.GetRoundedDecimal(promotedDiscount, numberOfDigits);
                    orderModel.Outlet = customerOutlet;
                    orderModel.SalesMode = customerOutlet.CustomerTypeId;
                    if (takePromotions)
                    {
                        orderModel.CustomerPromotions = transactionCommonData.CustomerPromotions;
                    }
                    #endregion
                    if (result == GlobalErrors.Success)
                    {
                        result = _orderManager.SaveEDIOrder(orderModel, ref orderProcessingResult);
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                orderProcessingResult.Message = ex.Message + ex.InnerException;
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors ValidateEDIOrder(TransactionCommonData transactionCommonData, ref EDIOrderOutputModel orderProcessingResult)
        {
            var result = GlobalErrors.Success;
            try
            {
                if (validationService.IsCustomerOnHold(transactionCommonData))
                {
                    orderProcessingResult.Message = MessagesConstants.Error_Customer_OnHold;
                    return GlobalErrors.Error;
                }
                if (validationService.IsExceedsCustomerLimit(transactionCommonData))
                {
                    orderProcessingResult.Message = MessagesConstants.Error_Customer_Exceeds_Limit;
                    return GlobalErrors.Error;
                }
                if (validationService.IsCustomerViolatePaymentTerms(transactionCommonData))
                {
                    orderProcessingResult.Message = MessagesConstants.Error_Customer_Violates_Payment_Terms;
                    return GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                orderProcessingResult.Message = ex.Message + ex.InnerException;
                result = GlobalErrors.Error;
            }
            return result;
        }

        #endregion


        #region [ECHO]

        public GlobalErrors CheckOutOrderForEcho(string path, List<ItemPackModel> soldItems, ref List<TransactionCommonData> transactionCommonDataList, ref string closeToPromotionIds)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                transactionCommonDataList = new List<TransactionCommonData>();
                TransactionCommonData transactionCommon = new TransactionCommonData();
                List<Promotion> customerPromotions = new List<Promotion>();
                List<int> closeToPrmotions = new List<int>();

                if (_requestRepository.CurrentEchoOperator.isMultiUser)
                {
                    List<ItemPackModel> soldItemsPerDiv = new List<ItemPackModel>();
                    foreach (OperatorCustomerModel custOperatorModel in _requestRepository.CurrentEchoOperator.OperatorCustomers)
                    {
                        result = EchoReadPromotionFromFile(path, ref customerPromotions);
                        soldItemsPerDiv = soldItems.Where(item => item.MainDivisionId == custOperatorModel.MainDivisionId).ToList();
                        transactionCommon = PrepareTransactionCommonDataForEcho(customerPromotions, custOperatorModel.CustomerId, custOperatorModel.OutletId, custOperatorModel.RouteEmployeeId, soldItemsPerDiv, ref closeToPrmotions, EchoService.CustomerAppSettings.ShowApplied, EchoService.CustomerAppSettings.ShowAchievements);
                        if (transactionCommon != null)
                        {
                            transactionCommon.CalculatedTotalDiscount = transactionCommon.SoldItems.Sum(x => x.CalculatedDiscount);
                            transactionCommon.CalculatedTotalTax = transactionCommon.SoldItems.Sum(x => x.CalculatedTax);
                            bool isNewCustomer = false;
                            var res = _orderManager.IsEchoFirstOrderCustomerOutlet(ref isNewCustomer);
                            if (isNewCustomer)
                            {
                                result = EchoReadPromotionFromFile(path, ref customerPromotions);
                                if (result != GlobalErrors.Success) return result;
                                decimal netTotal = transactionCommon.NetTotal;
                                decimal oldDiscount = transactionCommon.CalculatedTotalDiscount;
                                soldItemsPerDiv.ForEach(item =>
                                {
                                    item.Discount += _requestRepository.Configurations.EchoFirstDiscountCustomerOutlet;
                                    item.NetTotal = 0;
                                    item.GrossTotal = 0;
                                });
                                transactionCommon = PrepareTransactionCommonDataForEcho(customerPromotions, custOperatorModel.CustomerId, custOperatorModel.OutletId, custOperatorModel.RouteEmployeeId, soldItemsPerDiv, ref closeToPrmotions, EchoService.CustomerAppSettings.ShowApplied, EchoService.CustomerAppSettings.ShowAchievements);
                                if (transactionCommon != null)
                                {
                                    transactionCommon.CalculatedTotalDiscount = transactionCommon.SoldItems.Sum(x => x.CalculatedDiscountTotal);
                                    transactionCommon.CalculatedDiscountTotalBeforFirstUserDiscount = transactionCommon.CalculatedTotalDiscount - oldDiscount;

                                    transactionCommon.CalculatedTotalTax = transactionCommon.SoldItems.Sum(x => x.CalculatedTax);
                                    transactionCommon.NetTotalBeforFirstUserDiscount = netTotal;
                                }

                            }
                            soldItems.ForEach(item =>
                            {
                                item.CalculatedItemPromotedDiscount += (item.Price * item.Quantity) * (item.ItemPromotedDiscount / 100);
                                item.CalculatedAllItemPromotedDiscount = (item.Price * item.Quantity) * (item.AllItemDiscount / 100);
                            });
                            transactionCommon.TotalCalculatedItemPromotedDiscount = soldItems.Sum(i => i.CalculatedItemPromotedDiscount);
                            transactionCommon.TotalCalculatedAllItemPromotedDiscount = soldItems.Sum(i => i.CalculatedAllItemPromotedDiscount);

                            transactionCommonDataList.Add(transactionCommon);
                            soldItems.RemoveAll(allItems => soldItemsPerDiv.Exists(divItems => divItems.PackId == allItems.PackId));
                            result = GlobalErrors.Success;
                            if (soldItems.Count == 0) break;
                        }
                        else
                        {
                            result = GlobalErrors.Error;
                            break;
                        }
                    }
                    closeToPromotionIds = string.Join(",", closeToPrmotions);
                    return result;
                }
                else
                {
                    result = EchoReadPromotionFromFile(path, ref customerPromotions);
                    transactionCommon = PrepareTransactionCommonDataForEcho(customerPromotions, _requestRepository.CurrentEchoOperator.CustomerId, _requestRepository.CurrentEchoOperator.OutletId, _requestRepository.CurrentEchoOperator.RouteEmployeeId, soldItems, ref closeToPrmotions, EchoService.CustomerAppSettings.ShowApplied, EchoService.CustomerAppSettings.ShowAchievements);
                    if (transactionCommon != null)
                    {
                        transactionCommon.CalculatedTotalDiscount = transactionCommon.SoldItems.Sum(x => x.CalculatedDiscountTotal);
                        transactionCommon.CalculatedTotalTax = transactionCommon.SoldItems.Sum(x => x.CalculatedTax);
                        bool isNewCustomer = false;
                        var res = _orderManager.IsEchoFirstOrderCustomerOutlet(ref isNewCustomer);
                        if (isNewCustomer)
                        {
                            decimal netTotal = transactionCommon.NetTotal;
                            decimal oldDiscount = transactionCommon.CalculatedTotalDiscount;
                            soldItems.ForEach(item =>
                            {
                                item.Discount += _requestRepository.Configurations.EchoFirstDiscountCustomerOutlet;
                                item.NetTotal = 0;
                                item.GrossTotal = 0;
                                item.CalculatedTax = 0;
                            });
                            transactionCommon.NetTotal = 0;
                            transactionCommon.GrossTotal = 0;
                            result = EchoReadPromotionFromFile(path, ref customerPromotions);
                            if (result != GlobalErrors.Success) return result;
                            transactionCommon = PrepareTransactionCommonDataForEcho(customerPromotions, _requestRepository.CurrentEchoOperator.CustomerId, _requestRepository.CurrentEchoOperator.OutletId, _requestRepository.CurrentEchoOperator.RouteEmployeeId, soldItems, ref closeToPrmotions, EchoService.CustomerAppSettings.ShowApplied, EchoService.CustomerAppSettings.ShowAchievements);

                            if (transactionCommon != null)
                            {
                                List<ItemPackModel> itemPacks = new List<ItemPackModel>();
                                result = _orderManager.GetEchoFirstOrderFreeItems(ref itemPacks);
                                if (result != GlobalErrors.Success) return result;
                                transactionCommon.PromotedItems.AddRange(itemPacks);
                                transactionCommon.CalculatedTotalDiscount = transactionCommon.SoldItems.Sum(x => x.CalculatedDiscountTotal);
                                transactionCommon.CalculatedDiscountTotalBeforFirstUserDiscount = transactionCommon.CalculatedTotalDiscount - oldDiscount;

                                transactionCommon.CalculatedTotalTax = transactionCommon.SoldItems.Sum(x => x.CalculatedTax);
                                transactionCommon.NetTotalBeforFirstUserDiscount = netTotal;

                            }

                        }
                        soldItems.ForEach(item =>
                        {
                            item.CalculatedItemPromotedDiscount += (item.Price * item.RequiredQty) * (item.ItemPromotedDiscount / 100);
                            item.CalculatedAllItemPromotedDiscount = (item.Price * item.RequiredQty) * (item.AllItemDiscount / 100);
                        });
                        transactionCommon.TotalCalculatedItemPromotedDiscount = soldItems.Sum(i => i.CalculatedItemPromotedDiscount);
                        transactionCommon.TotalCalculatedAllItemPromotedDiscount = soldItems.Sum(i => i.CalculatedAllItemPromotedDiscount);

                        transactionCommonDataList.Add(transactionCommon);
                        closeToPromotionIds = string.Join(",", closeToPrmotions);

                        List<Promotion> achivements = new List<Promotion>();
                        // transactionCommon.ItemFormProcessMode = ItemFormProcessMode.Delivery;
                        bool hasWholeSalesTargetPromo = false;
                        bool hasDynamicTargetSalesAmountPromo = false;

                        if (result == GlobalErrors.Success)
                        {
                            if (EchoService.CustomerAppSettings.ShowAchievements)
                            {
                                bool hasSelection = false;
                                bool hasLoyalityPromotion = false;
                                bool hasNoSalesCertainPeriod = false;
                                result = _validationService.CheckPromotions(transactionCommon, ref achivements, false, ref hasWholeSalesTargetPromo, ref hasDynamicTargetSalesAmountPromo, true, true, true, ref hasSelection, ref hasLoyalityPromotion, true, ref hasNoSalesCertainPeriod);
                                if (result == GlobalErrors.Success)
                                {
                                    if (achivements != null && achivements.Count > 0)
                                        //achivements.RemoveAll(achive => achive.PromotionOutputs.Count == 0);
                                        transactionCommon.Achivements = achivements;

                                }
                            }
                        }



                        else return GlobalErrors.Error;
                        return GlobalErrors.Success;
                    }
                    else return GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors EchoReadPromotionFromFile(string rootPath, ref List<Promotion> customerPromotions)
        {
            try
            {
                var jsonFilePath = string.Format("Echo\\offers\\{0}.json", _requestRepository.CurrentEchoOperator.OperatorId);
                var path = Path.Combine(rootPath, "wwwroot", jsonFilePath);
                if (System.IO.File.Exists(path))
                {
                    customerPromotions = new List<Promotion>();
                    var jsonObj = System.IO.File.ReadAllText(path);
                    if (jsonObj != null)
                        customerPromotions = JsonConvert.DeserializeObject<List<Promotion>>(jsonObj);
                    return GlobalErrors.Success;

                }
                else
                {
                    GlobalErrors result = promotionService.EchoGeneratePromotionFile(rootPath, ref customerPromotions);
                    return result;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        private TransactionCommonData PrepareTransactionCommonDataForEcho(List<Promotion> customerPromotions, int customerId, int outletId, int employeeId, List<ItemPackModel> soldItems, ref List<int> closeToSatisfiedPromotions, bool showApplied, bool ShowAchievements)
        {
            TransactionCommonData transactionCommonData;
            try
            {
                var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;
                transactionCommonData = new TransactionCommonData();
                decimal transactionNetTotal = 0, grossTotal = 0, taxTotal = 0, discountTotal = 0, promotedDiscount = 0;
                var takePromotions = false;
                List<Promotion> loyalities = new List<Promotion>();
                soldItems.ForEach(s =>
               {
                   if (s.RequiredQty > 0)
                   {
                       s.SalesTransactionTypeId = SalesTransactionTypes.Sales;
                       s = transactionManager.CalculatePackTotals(s);
                       transactionNetTotal += s.NetTotal;
                       grossTotal += s.GrossTotal;
                       taxTotal += s.CalculatedTax;
                   }
               });

                transactionCommonData.CustomerId = customerId;
                transactionCommonData.OutletId = outletId;
                transactionCommonData.EmployeeId = employeeId;
                transactionCommonData.Outlet = _customerManager.FillOutletData(customerId, outletId, -1, null, true);
                transactionCommonData.GrossTotal = LocalUtilities.GetRoundedDecimal(grossTotal, numberOfDigits);
                transactionCommonData.Tax = LocalUtilities.GetRoundedDecimal(taxTotal, numberOfDigits);
                transactionCommonData.NetTotal = LocalUtilities.GetRoundedDecimal(transactionNetTotal, numberOfDigits);
                transactionCommonData.ItemFormProcessMode = ItemFormProcessMode.Orders;
                transactionCommonData.SoldItems = soldItems;

                if (customerPromotions.Count > 0)
                {
                    foreach (Promotion promotion in customerPromotions)
                    {
                        promotion.EchoShowAchinment = ShowAchievements;
                        promotion.IsTaken = false;
                        int _numberOfWholeSaleContract = 0;
                        //// here
                        promotion.IsSatisfied = _promotionManager.IsPromotionInputsSatisfied(promotion, transactionCommonData, ref _numberOfWholeSaleContract);
                        if (showApplied)
                        {
                            if (promotion.IsSatisfied && !promotion.IsLoyaltyPromotion && promotion.PromotionType != PromotionTypes.TargetPromotion && promotion.PromotionType != PromotionTypes.DynamicTargetPromotion && !promotion.RelatedToTarget)
                            {
                                transactionCommonData.EchoAppliedPromotions.Add(promotion);
                            }
                        }


                        if (promotion.PromotionType == PromotionTypes.StepsValues)
                        {
                            promotion.IsRepeated = false;
                        }
                        if (!promotion.IsSatisfied)
                        {
                            List<int> promoItems = promotion.PromotionInputs.SelectMany(input => input.PromotionOptionDetails).ToList().FindAll(ddd => ddd.InputOptionDetailTypeId == PromotionInputDetailTypes.ItemQuantity).Select(d => d.ItemId).Distinct().ToList();
                            if (promoItems.Count > 0)
                            {
                                if (soldItems.Any(a => promoItems.Contains(a.ItemId)))
                                {
                                    closeToSatisfiedPromotions.Add(promotion.PromotionId);
                                }
                            }
                        }
                    }
                    customerPromotions.RemoveAll(a => a.IsSatisfied == false);
                }
                if (customerPromotions.Count > 0)
                {
                    transactionCommonData.CustomerPromotions = customerPromotions;
                    // Cashing dynamic promotions
                    transactionCommonData.Achivements = customerPromotions.Where(x => x.PromotionType == PromotionTypes.DynamicTargetPromotion || x.PromotionType == PromotionTypes.TargetPromotion || x.RelatedToTarget || x.IsLoyaltyPromotion).ToList();
                    transactionCommonData.CustomerPromotions.RemoveAll(x => x.PromotionType == PromotionTypes.DynamicTargetPromotion || x.PromotionType == PromotionTypes.TargetPromotion || x.RelatedToTarget || x.IsLoyaltyPromotion);

                    GlobalErrors result = promotionService.TakePromotions(ref transactionCommonData, true);
                    if (result == GlobalErrors.Success)
                    {
                        takePromotions = transactionCommonData.CustomerPromotions.Any(promotion => promotion.IsTaken);
                        if (takePromotions)
                        {
                            // to calculate net totals again after taking promotions
                            transactionNetTotal = 0; grossTotal = 0; discountTotal = 0; taxTotal = 0;
                            transactionCommonData.SoldItems.ForEach(item =>
                            {
                                item = transactionManager.CalculatePackTotals(item);
                                item.SalesTransactionTypeId = SalesTransactionTypes.Sales;
                                transactionNetTotal += item.NetTotal;
                                grossTotal += item.GrossTotal;
                                taxTotal += item.CalculatedTax;
                                //exciseTaxTotal += item.CalculatedRetailTax;
                                discountTotal += item.CalculatedDiscountTotal;
                                promotedDiscount += item.PromotedDiscount;
                                item.NetTotal = LocalUtilities.GetRoundedDecimal(item.NetTotal, numberOfDigits);
                                item.CalculatedDiscountTotal = LocalUtilities.GetRoundedDecimal(item.CalculatedDiscountTotal, numberOfDigits);
                            });
                            transactionCommonData.GrossTotal = LocalUtilities.GetRoundedDecimal(grossTotal, numberOfDigits);
                            transactionCommonData.Tax = LocalUtilities.GetRoundedDecimal(taxTotal, numberOfDigits);
                            transactionCommonData.HeaderDiscount = LocalUtilities.GetRoundedDecimal(discountTotal, numberOfDigits);
                            transactionCommonData.NetTotal = transactionCommonData.GrossTotal + transactionCommonData.Tax - transactionCommonData.HeaderDiscount;
                        }
                        return transactionCommonData;
                    }
                    else return null;
                }
                else
                {
                    return transactionCommonData;
                    //Save Directly there is no Promotions
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
        }
        public GlobalErrors SaveOrderEcho(ref List<TransactionCommonData> transactionLsit)
        {
            var result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            var ordersToBeSent = new List<OrderModel>();
            bool isNewCustomer = false;
            List<Promotion> achivements = new List<Promotion>();
            string errorMessage = "";
            try
            {
                var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();
                foreach (TransactionCommonData transaction in transactionLsit)
                {

                    if (transaction.SoldItems == null || transaction.SoldItems.Count == 0)
                    {
                        continue;
                    }
                    OrderModel orderModel = new OrderModel();
                    orderModel.OrderDate = DateTime.Now;
                    orderModel.Outlet = new CustomerOutletModel();
                    orderModel.Outlet.CustomerId = transaction.CustomerId;
                    orderModel.Outlet.OutletId = transaction.OutletId;
                    orderModel.CustomerId = transaction.CustomerId;
                    orderModel.OutletId = transaction.OutletId;
                    orderModel.deliveryLocation = transaction.DeliveryLocation;
                    transaction.SoldItems.ForEach(i =>
                    {
                        i.RequestedQuantity = i.RequiredQty;
                    });
                    if (transaction.PromotedItems != null)
                    {
                        transaction.PromotedItems.ForEach(i =>
                        {
                            i.RequestedQuantity = i.RequiredQty;
                        });
                    }
                    orderModel.SoldItems = transaction.SoldItems;
                    orderModel.PromotedItems = transaction.PromotedItems;
                    orderModel.OrderStatus = _requestRepository.Configurations.ApproveCustomerAppOrdersInFO ? OrderStatus.New : OrderStatus.New;
                    orderModel.OrderTypeId = OrderTypes.Sales;
                    orderModel.DivisionId = -1;

                    // Get Max Order ID
                    orderModel.OrderId = _documentSequenceService.GetMaxDocumentSequence(transaction.EmployeeId, DocumentTypes.Order, orderModel.DivisionId);
                    if (string.IsNullOrEmpty(orderModel.OrderId))
                        return GlobalErrors.Error;
                    orderModel.CreatedBy = transaction.EmployeeId;
                    orderModel.CreationSource = CreationSource.Echo;
                    orderModel.OrganizationId = _requestRepository.CurrentEchoOperator.RouteOrganizationId;
                    orderModel.AllItems = new List<ItemPackModel>();
                    orderModel.AllItems.AddRange(orderModel.SoldItems);
                    orderModel.AllItems.AddRange(orderModel.PromotedItems);
                    orderModel.OrderSourceId = -1;
                    orderModel.EmployeeId = transaction.EmployeeId;
                    if (transaction.DesiredDeliveryDateString != null)
                    {
                        DateTime dateTime = Convert.ToDateTime(transaction.DesiredDeliveryDateString);
                        orderModel.DesiredDeliveryDate = (new DateTime(dateTime.Year, dateTime.Month,
                       dateTime.Day, 12, 0, 0));
                    }
                    else
                    {
                        orderModel.DesiredDeliveryDate = orderModel.OrderDate;
                    }
                   
                    
                    orderModel.CustomerPromotions = transaction.CustomerPromotions;
                    decimal promotedDiscount = 0; decimal transactionNetTotal = 0; decimal grossTotal = 0; decimal exciseTaxTotal = 0; decimal discountTotal = 0; decimal headerDiscountTotal = 0; decimal taxTotal = 0;
                    orderModel.SoldItems.ForEach(item =>
                    {
                        item = transactionManager.CalculatePackTotals(item);
                        item.SalesTransactionTypeId = SalesTransactionTypes.Sales;
                        transactionNetTotal += item.NetTotal;
                        grossTotal += item.GrossTotal;
                        taxTotal += item.CalculatedTax;
                        exciseTaxTotal += item.CalculatedRetailTax;
                        discountTotal += item.CalculatedDiscountTotal;
                        headerDiscountTotal += item.HeaderDiscount;
                        promotedDiscount += item.PromotedDiscount;
                    });

                    //orderModel.NetTotal = orderModel.NetTotal - (orderModel.NetTotal * (discount / 100));
                    orderModel.NetTotal = LocalUtilities.GetRoundedDecimal(transactionNetTotal, numberOfDigits);
                    orderModel.GrossTotal = LocalUtilities.GetRoundedDecimal(grossTotal, numberOfDigits);
                    orderModel.Tax = LocalUtilities.GetRoundedDecimal(taxTotal, numberOfDigits);
                    orderModel.Discount = LocalUtilities.GetRoundedDecimal(headerDiscountTotal, numberOfDigits);
                    orderModel.SalesMode = (_requestRepository.CurrentEchoOperator.CustomerTypeId == CustomerTypes.CashCustomer.GetHashCode() ? 1 : 2);
                    orderModel.Notes = new List<string>();
                    if (transaction.OrderNote != null && !string.IsNullOrEmpty(transaction.OrderNote.Trim()))
                    {
                        orderModel.Notes.Add(transaction.OrderNote);
                    }
                    string msg = string.Empty;
                    result = SaveOrder(orderModel, ref msg, true, dbHelper, true);

                    //if is new customer in echo
                    result = _orderManager.IsEchoFirstOrderCustomerOutlet(ref isNewCustomer);
                    if (result != GlobalErrors.Success) return result;
                    if (isNewCustomer)
                    {
                        result = _orderManager.InsertEchoFirstOrderCustomerOutlet(orderModel.OrderId, dbHelper);
                        if (result != GlobalErrors.Success) return result;
                    }
                    if (result != GlobalErrors.Success)
                    {
                        break;
                    }
                    if (result == GlobalErrors.Success && _requestRepository.Configurations.ApproveCustomerAppOrdersInFO)
                    {
                        orderModel.Notes = new List<string>();
                        if (transaction.OrderNote != null && !string.IsNullOrEmpty(transaction.OrderNote.Trim()))
                        {
                            orderModel.Notes.Add(transaction.OrderNote.Replace("'", "''"));
                        }
                        ordersToBeSent.Add(orderModel);

                    }
                    if (result == GlobalErrors.Success && transaction.EchoPaymentType == PaymentTypes.CreditCard.GetHashCode())
                    {
                        TransactionModel transaction1 = new TransactionModel();

                        transaction1.TransactionType = TransactionType.CreditNote;
                        transaction1.TransactionTypeId = TransactionType.CreditNote.GetHashCode();
                        transaction1.OutletId = transaction.OutletId;
                        transaction1.Outlet = transaction.Outlet;
                        transaction1.RouteId = transaction.RouteId;
                        transaction1.NetTotal = transaction.NetTotal;
                        transaction1.CustomerId = transaction.CustomerId;
                        transaction1.EmployeeId = transaction.EmployeeId;
                        transaction1.DocumentType = DocumentTypes.CreditNote;
                        transaction1.CreationReason = TransactionCreationReason.Normal;
                        transaction1.AllowBatchesInItems = false;
                        transaction1.TransactionDateModel = new DateModel(new DateTime(DateTime.Now.Year, DateTime.Now.Month, DateTime.Now.Day, DateTime.Now.Hour, DateTime.Now.Minute, DateTime.Now.Second));
                        transaction.Outlet.CustOutPaymentTerm = null;

                        result = _transactionService.SaveTransactionWithoutStockPosting(transaction1, false, ref errorMessage,dbHelper);

                    }
                    if (result == GlobalErrors.Success)
                    {
                       // CustomerAppSettingsModel customerAppSettingsModel = new CustomerAppSettingsModel();
                       // LoginManager.GetCustomerAppSettings(ref customerAppSettingsModel);
                        if (EchoService.CustomerAppSettings.ShowAchievements)
                        {
                            if (transaction.Achivements.Count > 0)
                            {
                                result = SaveOrderAchivements(orderModel.OrderId, transaction.Achivements, dbHelper);
                                if (result != GlobalErrors.Success) return result;
                            }
                        }
                          result = SendActionsEmail(orderModel, ref errorMessage);
                    }

                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                        if (result == GlobalErrors.Success && ordersToBeSent != null && ordersToBeSent.Count > 0
                                && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID)
                                && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                        {
                            Task.Run(() =>
                            {
                                _notificationManager.SendNotifiactionToSonicFO(ordersToBeSent);
                            });
                        }
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors SendActionsEmail(OrderModel orderModel, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            EmailModel emailModel = new EmailModel();
            List<EmailTemplateModel> emailTempList = new List<EmailTemplateModel>();
            OrganizationDetails organizationDetails = new OrganizationDetails();
            string emailTypeIDs = string.Empty;
            errorMessage = string.Empty;
            try
            {
                emailTypeIDs = "16";
                result = _emailManager.GetEmailTemplatesDetails(emailTypeIDs, ref emailTempList);
                if (result == GlobalErrors.Success && emailTempList != null && emailTempList.Count > 0)
                {
                    if (emailTempList.Any(e => (e.EmailTempId > 0)
                    && (
                    (e.EmailTypeId == 16))))
                    {
                        result = _echoManager.GetOrganizationDetails(orderModel.OrganizationId,ref organizationDetails);
                    }
                    if (result == GlobalErrors.Success && organizationDetails != null)
                    {
                        foreach (EmailTemplateModel emailTemplate in emailTempList)
                        {
                            if (emailTemplate.EmailTypeId == 16)
                            {
                                emailTemplate.Body = emailTemplate.Body.Replace("{0}", orderModel.OrderId).Replace("@@", "<br>");
                                emailTemplate.Body = emailTemplate.Body.Replace("{1}", orderModel.NetTotal.ToString()).Replace("@@", "<br>");
                                emailModel = new EmailModel();
                                emailModel.ToEmail = organizationDetails.Email;
                                emailModel.EmailTempId = emailTemplate.EmailTempId;
                                emailModel.Body = emailTemplate.Body;
                                emailModel.IsBodyHtml = false;
                                _emailService.SendEmailAsync(null, emailModel);
                            } 
                        }  
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors SaveOrderAchivements(string orderId, List<Promotion> Achivements, DBHelper<int> dbHelper)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                Achivements.ForEach(i =>
                {
                    i.PromotionOutputs.ForEach(o =>
                    {
                        result = _orderManager.SaveOrderAchivementForEcho(orderId, i.PromotionId, o.PromotionOptionId, dbHelper);
                    });

                });
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;

            }
        }
        public GlobalErrors IsEchoFirstOrderCustomerOutlet(ref bool isFirstOrderCustomerOutlet, ref bool isEchoFirstOrderHaveFreeItems)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                // if is new customer in echo
                result = _orderManager.IsEchoFirstOrderCustomerOutlet(ref isFirstOrderCustomerOutlet);
                if (result == GlobalErrors.Success)
                {
                    result = _orderManager.IsEchoFirstOrderHaveFreeItems(ref isEchoFirstOrderHaveFreeItems);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public async static Task<bool> SendOrderTest(string id, string token)
        {
            bool result = false;
            try
            {
                OrderModel order = new OrderModel();
                if (OrderManager.GetOrderHeaderStatic(id, ref order) == GlobalErrors.Success && order != null)
                {
                    NotificationManager notificationManager = new NotificationManager(null);

                    result = await notificationManager.SendNotifiactionToSonicFO(order, token, 1);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = false;
            }

            return result;

        }
        public GlobalErrors GetCustomerOrderHistoryForEcho(  ref List<EchoOrderModel> list)
        {

            return _orderManager.GetCustomerOrderHistoryForEcho(0, 0,true, ref list);
        }
        public GlobalErrors CancelOrderForEcho(EchoOrderModel order)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (order == null)
                {
                    return GlobalErrors.NoMoreRows;
                }
                return _orderManager.CancelOrderForEcho(order);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors ApproveOrderForEcho(EchoOrderModel order)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (order == null)
                {
                    return GlobalErrors.NoMoreRows;
                }
                result = _orderManager.ApproveOrderForEcho(order);
                if (result == GlobalErrors.Success)
                {
                    OrderModel orderModel = new OrderModel();
                    result = _orderManager.GetOrderHeader(order.OrderId, ref orderModel);
                    if (result == GlobalErrors.Success && orderModel != null && _requestRepository.Configurations.SplitOrderAutomatically)
                    {
                        result = _orderManager.CallProcedureSplitOrder(orderModel.CustomerId, orderModel.OutletId, orderModel.DivisionId, orderModel.OrderId);
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetOrderDetailsForEchoReorder(string orderId, ref List<ItemPackModel> orderDetails, bool fromFlutter)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                ItemManager itemManager = new ItemManager(_requestRepository);
                result = itemManager.GetEchoItems(new EchoGeneralFilter(), orderId, ref orderDetails, fromFlutter);
                if (result == GlobalErrors.Success && orderDetails != null && orderDetails.Count > 0)
                {
                    #region [Discount]

                    //if (result == GlobalErrors.Success && orderDetails != null && orderDetails.Count > 0)
                    //{
                    //    List<ItemDiscountModel> discountList = new List<ItemDiscountModel>();
                    //    result = ItemManager.GetAllDiscount(_requestRepository.CurrentEchoOperator.CustomerId, _requestRepository.CurrentEchoOperator.OutletId, ref discountList);
                    //    if (result == GlobalErrors.Success && discountList != null && discountList.Count > 0)
                    //    {
                    //        orderDetails.ForEach(i =>
                    //        {
                    //            var discount = discountList.Find(d => d.PackId == i.PackId);
                    //            if (discount != null && discount.Discount > 0)
                    //            {
                    //                if (discount.DiscountTypeId == DiscountValueTypes.Amount.GetHashCode())
                    //                {
                    //                    if (_requestRepository.Configurations.ApplyAmountDiscountPerQuantity)
                    //                    {
                    //                        i.Price = i.Price - discount.Discount;
                    //                        if (i.Price < 0)
                    //                        {
                    //                            i.Price = 0;
                    //                        }
                    //                    }
                    //                    //else Disocunt will be discarded 
                    //                }
                    //                else
                    //                {
                    //                    i.Price = i.Price - (i.Price * discount.Discount / 100);
                    //                    if (i.Price < 0)
                    //                    {
                    //                        i.Price = 0;
                    //                    }
                    //                }
                    //                i.Discount = discount.Discount;
                    //                i.DiscountTypeId = discount.DiscountTypeId;
                    //            }
                    //        });
                    //    }
                    //}

                    #endregion

                    orderDetails.ForEach(s =>
                    {
                        if (s.RequiredQty > 0)
                        {
                            s.GrossTotal = (s.RequiredQty * s.Price);
                            s.CalculatedTax = (s.RequiredQty * s.Price * s.Tax / 100);
                            s.NetTotal = LocalUtilities.GetRoundedDecimal(s.GrossTotal + s.CalculatedTax, _requestRepository.Configurations.NumberOfDigits);
                            s.NetAmountString = s.NetTotal.ToString();
                        }
                    });
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetOrderDataForEcho(string orderId, bool getStatus, ref List<ItemPackModel> soldItems, ref List<ItemPackModel> promotedItems, ref int statusId, ref string notes, ref decimal netTotal, ref decimal grossTotal, ref decimal discountTotal, ref decimal taxTotal, ref List<Promotion> achivements, ref List<Promotion> applied)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                achivements = new List<Promotion>();
                List<ItemPackModel> orderDetails = new List<ItemPackModel>();
                result = _orderManager.GetOrderDataForEcho(orderId, ref orderDetails, ref notes);
                if (result == GlobalErrors.Success)
                {
                    if (orderDetails != null && orderDetails.Count > 0)
                    {
                        orderDetails.ForEach(s =>
                        {
                            if (s.RequestedQuantity > 0)
                            {
                                s.GrossTotal = (s.RequestedQuantity * s.Price);
                                s.CalculatedDiscount = (s.GrossTotal * (s.Discount + s.PromotedDiscount) / 100);
                                s.CalculatedTax = (s.GrossTotal - s.CalculatedDiscount) * (s.Tax / 100);
                                s.NetTotal = s.GrossTotal + s.CalculatedTax - s.CalculatedDiscount;

                                //// echo flutter
                                s.ExclusivePrice = s.NetTotal / s.RequestedQuantity;
                                s.NetTotal = LocalUtilities.GetRoundedDecimal(s.NetTotal, _requestRepository.Configurations.NumberOfDigits);
                                s.NetAmountString = s.NetTotal.ToString();
                            }
                        });
                        soldItems = orderDetails.Where(i => i.SalesTransactionTypeId == SalesTransactionTypes.Sales).ToList();
                        promotedItems = orderDetails.Where(i => i.SalesTransactionTypeId != SalesTransactionTypes.Sales).ToList();
                    }
                    OrderModel order = new OrderModel();
                    result = _orderManager.GetOrderHeader(orderId, ref order);
                    if (result == GlobalErrors.Success)
                    {
                        netTotal = order.NetTotal;
                        grossTotal = order.GrossTotal;
                        discountTotal = order.Discount;
                        taxTotal = order.Tax;
                    }
                    else
                    {
                        return result;
                    }
                    if (getStatus)
                        result = _orderManager.GetOrderStatusForEcho(orderId, ref statusId);
                    if (result == GlobalErrors.Success)
                    {
                        result = _orderManager.GetOrderDataAchivemantsForEcho(orderId, ref achivements);
                    }
                    if (result == GlobalErrors.Success)
                    {
                        result = _orderManager.GetOrderDataAppliedOffersForEcho(orderId, ref applied);
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetEchoFirstOrderFreeItems(ref List<ItemPackModel> itemPacks)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                return _orderManager.GetEchoFirstOrderFreeItems(ref itemPacks);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetCRMEmployeeOrdersHistory(CustomerFilter filter, ref string uniqueIdentifier)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                List<string> selectedValues = new List<string>();
                string query = string.Empty;

                if(filter.OrderHistoryFilterType == OrderHistoryFilterType.Today)
                {
                    selectedValues.Add("From Date" + "~*~" + DateTime.Now.Date.ToString("dd/MM/yyyy"));
                    selectedValues.Add("To Date" + "~*~" + DateTime.Now.Date.ToString("dd/MM/yyyy"));
                }
                else
                {
                    if(filter.FromToDate != null && filter.FromToDate.Count == 2)
                    {
                        selectedValues.Add("From Date" + "~*~" + DateTime.Now.ToString("dd/MM/yyyy"));
                        selectedValues.Add("To Date" + "~*~" + DateTime.Now.ToString("dd/MM/yyyy"));
                    }
                }
                
                selectedValues.Add("Planned Customer" + "~*~" + (filter.ShowUnplannedCustomers ? "No" : "Yes"));
                selectedValues.Add("Customer Code" + "~*~" + (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter) ? filter.CustomListFilter.SearchFilter : "---"));
                
                

                string filtersData = string.Join("~@~", selectedValues);
                result = _orderManager.GetCRMEmployeeOrdersHistory(filter, ref query);
                if(result == GlobalErrors.Success && !string.IsNullOrEmpty(query))
                {
                    DynamicReportModel report = new DynamicReportModel();
                    report.ReportId = -1;
                    report.ReportName = "Salesperson Orders History";
                    report.QueryString = query;
                    
                    result = _dynamicReportManager.SaveTempReport(ref uniqueIdentifier, report, filtersData);
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        #endregion
    }

}