﻿using Sonic.Web.Core;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Models;
using Sonic.Web.Resources;
using Sonic.Web.SecureLibrary;
using Sonic.Web.Services;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Reflection;
using System.Text;

namespace Sonic.Web.Service
{
    public class PaymentService
    {
        private readonly IRequestRepository _requestRepository;
        private readonly PaymentManager _paymentManager;
        private readonly TransactionManager _transactionManager;
        private readonly EmployeeManager _employeeManager;
        private readonly AccountManager _accountManager;
        private readonly DocumentSequenceManager _documentSequenceManager;
        private readonly RouteManager _routeManager;
        public PaymentService(IRequestRepository requestRepository)
        {
            _requestRepository = requestRepository;
            _paymentManager = new PaymentManager(_requestRepository);
            _transactionManager = new TransactionManager(_requestRepository);
            _employeeManager = new EmployeeManager(_requestRepository);
            _accountManager = new AccountManager(_requestRepository);
            _documentSequenceManager = new DocumentSequenceManager(requestRepository);
            _routeManager = new RouteManager(requestRepository);
        }
        public GlobalErrors GetALLPaymentTypes(ref List<PaymentTypeModel> PaymentTypes)
        {
            try
            {
                return _paymentManager.GetALLPaymentTypes(ref PaymentTypes);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                PaymentTypes = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetPaymentTypes(ref List<PaymentTypeModel> paymentTypesList, bool isDownPayment,int customerID, int outletID)
        {
            try
            {
                return _paymentManager.GetPaymentTypes(ref paymentTypesList, isDownPayment, customerID,  outletID);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                paymentTypesList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetBankWithBranches(ref List<BankModel> bankList)
        {
            try
            {
                return _paymentManager.GetBankWithBranches(ref bankList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                bankList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SavePayments(SavePaymentModel savePaymentModel, ref string customerPaymentsIDsStringRef, ref string errorMessage, DBHelper<int> dbHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            //DBHelper<int> dbHelper = null;
            bool commitNow = false;
            try
            {
                
                if (savePaymentModel.CustomerId != -1 && savePaymentModel.EmployeeId != -1)
                {
                    // check for used credit notes, and return error if any used credit note become uploaded or invalid.
                    if (savePaymentModel.PaymentsList.Any(x => x.PaymentTypeId == PaymentTypes.CreditNote))
                    {
                        List<TransactionModel> availableCNs = new List<TransactionModel>();
                        result = _transactionManager.GetCustomerAvailableCreditNotes(savePaymentModel.CustomerId, savePaymentModel.OutletId, savePaymentModel.DivisionId, ref availableCNs);
                        if (result == GlobalErrors.Success)
                        {
                            List<string> usedCNIds = savePaymentModel.PaymentsList.Where(x => x.PaymentTypeId == PaymentTypes.CreditNote && x.AppliedAmount > 0).ToList().Select(x => x.CreditNoteId).ToList(); 
                            foreach(string CNId in usedCNIds)
                            {
                                if (availableCNs == null || !availableCNs.Any(AvlCN => AvlCN.TransactionId == CNId))
                                {
                                    errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_Used_CN_Not_Available, _requestRepository.LanguageId);
                                    result = GlobalErrors.Error;

                                    return result;
                                }
                            }
                        }
                    }

                    result = GlobalErrors.NotInitialized;
                    if (dbHelper == null)
                    {
                        dbHelper = new DBHelper<int>();
                        dbHelper.BeginTransaction();
                        commitNow = true;
                    }

                    string paymentIDs = string.Empty;
                    result = _paymentManager.SavePayments(savePaymentModel, dbHelper, ref paymentIDs, ref errorMessage);
                    customerPaymentsIDsStringRef = paymentIDs;

                    // Extra Payment
                    var extraPayment = savePaymentModel.PaymentsList.Where(x => x.RemainingAmount > 0).FirstOrDefault();
                    if (extraPayment != null && result == GlobalErrors.Success)
                    {
                        // Prepare data to extra payment
                        bool isExtraPaymentByCollectInvoice = true;
                        savePaymentModel.PaymentsList.Select(x =>
                        {
                            x.CustomerId = savePaymentModel.CustomerId;
                            x.OutletId = savePaymentModel.OutletId;
                            x.AppliedAmount = x.RemainingAmount;
                            x.DivisionId = savePaymentModel.DivisionId;
                            x.CurrencyId = savePaymentModel.TransactionsList.Select(x => x.CurrencyId).FirstOrDefault();
                            return x;
                        }).ToList();
                        int DivisionId = -1;
                        string paymentID = string.Empty;
                        result = SaveDownPayments(savePaymentModel.PaymentsList, ref paymentID, ref DivisionId, isExtraPaymentByCollectInvoice, dbHelper, ref errorMessage);
                    }

                }
                else
                {
                    result = GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if(commitNow && dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    if (dbHelper != null)
                    {
                        dbHelper.Dispose();
                        dbHelper = null;
                    }
                }
            }
            return result;
        }
        public GlobalErrors GetOnlineEPaymentsList(List<PaymentModel> payments)
        {
            try
            {
                return _paymentManager.GetOnlineEPaymentsList(payments);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                payments = null;
                return GlobalErrors.Error;
            }
        } 
        public GlobalErrors SaveOnlineEPaymentResponseForUpdateStatus(string ResponseDescription, string UniqueID,string  ResponseCode,PaymentModel payment,string responseContent, string chg_id)
        {
            try
            {
                return _paymentManager.SaveOnlineEPaymentResponseForUpdateStatus(ResponseDescription, UniqueID, ResponseCode, payment, responseContent, chg_id);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }        
        public GlobalErrors VoidOnlineEPayment(PaymentModel payment)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _paymentManager.VoidOnlineEPayment(payment);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            } return result;
        }
        public GlobalErrors PostOnlineEpayment(PaymentModel payment , string ApprovalCode)
        {
            DBHelper<int> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();
                result = _paymentManager.PostOnlineEpayment(payment, ApprovalCode, dbHelper);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dbHelper.CommitTransaction();
                }
                else
                {
                    dbHelper.RollBackTransaction();
                }
                if (dbHelper != null)
                {
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors SaveEPaymentNoAuthResponse(string invoiceId, string chgId)
        {
            try
            {
                return _paymentManager.SaveEPaymentNoAuthResponse(invoiceId, chgId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPaymentsList(GeneralFilter filter, ref SharedTableResult<PaymentModel> custList)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                if (LocalUtilities.IsCRM && _requestRepository.Configurations.UsePayerAccount && _requestRepository.Configurations.AllowCollectionOnPayerAccount)
                {
                    List<PayerAccountModel> payers = new List<PayerAccountModel>();

                    result = _transactionManager.GetCustomerPayerAccountIds(filter.CustomerId, filter.OutletId, ref payers);
                    if (result == GlobalErrors.Success && payers != null && payers.Count > 0)
                    {
                        filter.AccountId = payers[0].AccountId;
                        string payerCodeName = payers[0].PayerName;
                        return _paymentManager.GetPaymentsListPayerLevel(filter, payerCodeName, ref custList);
                    }
                }
                else
                {
                    return _paymentManager.GetPaymentsList(filter, ref custList);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
            return GlobalErrors.Error;
        }
        public GlobalErrors GetPaymentTransactions(PaymentModel payment,ref List<PaymentTransaction> paymentList)
        {
            try
            {
                DBHelper<PaymentTransaction> dbHelperTransactions = null;
                return _paymentManager.GetPaymentTransactions(payment, dbHelperTransactions,ref paymentList, true);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors VoidSelectedPayments(List<PaymentModel> selectedPayments, ref StringBuilder voidedMsg)
        {
            DBHelper<int> dbHelper = null;
            DBHelper<PaymentTransaction> dbHelperTransactions = null;
            List<PaymentTransaction> paymentTransactions = new List<PaymentTransaction>();
            GlobalErrors result = GlobalErrors.NotInitialized;
            string errorMessage = string.Empty;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();
                dbHelperTransactions = new DBHelper<PaymentTransaction>(dbHelper.GetConnection(), dbHelper.GetDBTransaction());

                List<TransactionModel> debitNotesList = new List<TransactionModel>();
                TransactionModel debitNoteTransaction = new TransactionModel();

                foreach (PaymentModel payment in selectedPayments)
                {
                    if (_routeManager.IsRouteUploaded(payment.RouteHistoryId, dbHelper))
                    {
                        voidedMsg.AppendLine(string.Format(" {0} {1}", payment.CustomerPaymentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Uploaded_Route, _requestRepository.LanguageId)));
                        return GlobalErrors.Error;
                    }
                    if (payment.PaymentTypeId == PaymentTypes.CreditCard && _requestRepository.Configurations.UseOnlineCreditCard)
                    {
                        voidedMsg.AppendLine(string.Format(" {0} {1}", payment.CustomerPaymentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Cannot_Void_Online_Credit_Card_Payment, _requestRepository.LanguageId)));
                        return GlobalErrors.Error;
                    }
                    if (payment.PaymentTypeId == PaymentTypes.OnlineEPayment)
                    {
                        voidedMsg.AppendLine(string.Format(" {0} {1}", payment.CustomerPaymentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Cannot_Void_Online_Credit_Card_Payment, _requestRepository.LanguageId)));
                        return GlobalErrors.Error;
                    }
                    if (_paymentManager.CheckIfPaymentVoided(payment, dbHelper))
                    {
                        voidedMsg.AppendLine(string.Format(" {0} {1}", payment.CustomerPaymentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Voided_Before, _requestRepository.LanguageId)));
                    }
                    else
                    {
                        paymentTransactions = new List<PaymentTransaction>();
                        debitNotesList = new List<TransactionModel>();
                        debitNoteTransaction = null;

                        // Cash payment transactions
                        result = _paymentManager.GetPaymentTransactions(payment, dbHelperTransactions, ref paymentTransactions);
                        if (result == GlobalErrors.Success && paymentTransactions != null && paymentTransactions.Count > 0)
                        {
                            payment.Posted = paymentTransactions[0].Posted;

                            foreach (PaymentTransaction paymentTransaction in paymentTransactions)
                            {
                                #region[Update Source Transaction Remaining When Void CN Payment]
                                if (result == GlobalErrors.Success)
                                {
                                    if (paymentTransaction.Posted && payment.PaymentTypeId == PaymentTypes.CreditNote)
                                    {
                                        if (string.IsNullOrEmpty(paymentTransaction.SourceTransactionId))
                                        {
                                            result = GlobalErrors.Error;
                                            return result;
                                        }
                                        else
                                        {
                                            result = _accountManager.UpdateTransactionRemainingAmountIncrease(dbHelper, paymentTransaction.SourceTransactionId, payment.CustomerId, payment.OutletId, payment.DivisionId, paymentTransaction.AppliedAmount, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now);
                                            if (result != GlobalErrors.Success)
                                            {
                                                return result;
                                            }
                                        }
                                    }
                                }
                                #endregion

                                #region[Update Transaction Remaninig and balances]

                                #region[Update Transaction RemainingAmount]
                                result = _accountManager.UpdateTransactionRemainingAmountIncrease(dbHelper, paymentTransaction.TransactionId, payment.CustomerId, payment.OutletId, payment.DivisionId, paymentTransaction.AppliedAmount, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now);

                                if (result == GlobalErrors.Success && paymentTransaction.PaymentTypeId == PaymentTypes.iCash.GetHashCode())
                                {
                                    result = _paymentManager.UpdateICashBalanceICashPayment(paymentTransaction.TransactionId, payment.CustomerId, payment.OutletId, dbHelper);
                                }
                                if (result != GlobalErrors.Success)
                                {
                                    return GlobalErrors.Error;
                                }
                                #endregion

                                #region[If Posted => Update Account Balances]
                                if (paymentTransaction.Posted)
                                {
                                    #region[Update Customer Account Balance]
                                    if (result == GlobalErrors.Success)
                                    {
                                        result = _accountManager.UpdateAccountBalance(payment.AccountId, paymentTransaction.AppliedAmount, paymentTransaction.TransactionDate, paymentTransaction.TransactionEmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, paymentTransaction.TransactionId, dbHelper, 5);
                                        if (result != GlobalErrors.Success)
                                        {
                                            return GlobalErrors.Error;
                                        }
                                    }
                                    #endregion

                                    #region[Update Employee Account Balance]

                                    if (result == GlobalErrors.Success)
                                    {
                                        if (paymentTransaction.SalesMode == SalesModes.CashSales || (paymentTransaction.SalesMode == SalesModes.CreditSales && _requestRepository.Configurations.UpdateEmployeeBalanceOnCreditSale))
                                        {
                                            int employeeAccountId = -1;
                                            result = _accountManager.GetEmployeeAccountID(paymentTransaction.TransactionEmployeeId, ref employeeAccountId, dbHelper);
                                            if (result == GlobalErrors.Success && employeeAccountId != -1)
                                            {
                                                result = _accountManager.UpdateAccountBalance(employeeAccountId, paymentTransaction.AppliedAmount, paymentTransaction.TransactionDate, paymentTransaction.TransactionEmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, paymentTransaction.TransactionId, dbHelper, 5);
                                                if (result == GlobalErrors.Success && _requestRepository.Configurations.ControlTransactionOnDivisionLevel && _requestRepository.Configurations.CheckCreditLimitOnEmployeeDivisionAccount)
                                                {
                                                    result = _accountManager.UpdateEmployeeDivisionAccountBalance(employeeAccountId, paymentTransaction.AppliedAmount, paymentTransaction.TransactionDate, paymentTransaction.TransactionEmployeeId, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now, paymentTransaction.TransactionId, dbHelper, 5, payment.DivisionId, ref errorMessage);
                                                }
                                            }
                                            else
                                            {
                                                // employee must have account
                                                return GlobalErrors.Error;
                                            }
                                        }
                                    }
                                    #endregion
                                }
                                #endregion

                                if (result != GlobalErrors.Success)
                                {
                                    return result;
                                }
                                #endregion
                            }

                            #region[Update Customer Payment set voided]
                            if (result == GlobalErrors.Success)
                            {
                                payment.PaymentStatusId = PaymentStatusTypes.Voided;
                                result = _paymentManager.UpdateCustomerPaymentStatus(payment, dbHelper);
                                if (result != GlobalErrors.Success)
                                {
                                    return result;
                                }
                            }
                            #endregion
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dbHelper.CommitTransaction();
                }
                else
                {
                    dbHelper.RollBackTransaction();
                }
                if (dbHelper != null)
                {
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors BounceCheques(List<PaymentModel> selectedPayments, ref StringBuilder voidedMsg)
        {
            DBHelper<int> dbHelper = null;
            DBHelper<PaymentTransaction> dbHelperTransactions = null;
            List<PaymentTransaction> paymentTransactions = new List<PaymentTransaction>();
            GlobalErrors result = GlobalErrors.NotInitialized;
            string errorMessage = string.Empty;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();
                dbHelperTransactions = new DBHelper<PaymentTransaction>(dbHelper.GetConnection(), dbHelper.GetDBTransaction());

                List<TransactionModel> debitNotesList = new List<TransactionModel>();
                TransactionModel debitNoteTransaction = new TransactionModel();

                foreach (PaymentModel payment in selectedPayments)
                {
                    if (_paymentManager.CheckIfPaymentBounced(payment, dbHelper))
                    {
                        voidedMsg.AppendLine(string.Format(" {0} {1}", payment.CustomerPaymentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Bounce_Before, _requestRepository.LanguageId)));
                    }
                    else
                    {
                        paymentTransactions = new List<PaymentTransaction>();
                        debitNotesList = new List<TransactionModel>();
                        debitNoteTransaction = null;

                        // Cash payment transactions
                        result = _paymentManager.GetPaymentTransactions(payment, dbHelperTransactions, ref paymentTransactions);
                        if (result == GlobalErrors.Success && paymentTransactions != null && paymentTransactions.Count > 0)
                        {
                            payment.Posted = paymentTransactions[0].Posted;

                            foreach (PaymentTransaction paymentTransaction in paymentTransactions)
                            {
                                #region[Prepare Debit Note transaction]
                                if (_requestRepository.Configurations.ChequeBounceMode == ChequeBounceMode.CreateDebitNote.GetHashCode() || _requestRepository.Configurations.ChequeBounceMode == ChequeBounceMode.CreateDebitNoteForPDCPostedOrNot.GetHashCode()) // Create Debit Note
                                {
                                    #region [Prepare Debit Note Transaction]
                                    if (debitNotesList.Any(debitNote => debitNote.CustomerId == payment.CustomerId && debitNote.OutletId == payment.OutletId && debitNote.AccountId == payment.AccountId && debitNote.EmployeeId == paymentTransaction.TransactionEmployeeId && debitNote.SalesMode == paymentTransaction.SalesMode))
                                    {
                                        debitNotesList.Find(debitNote => debitNote.CustomerId == payment.CustomerId && debitNote.OutletId == payment.OutletId && debitNote.AccountId == payment.AccountId && debitNote.EmployeeId == paymentTransaction.TransactionEmployeeId && debitNote.SalesMode == paymentTransaction.SalesMode).NetTotal += paymentTransaction.AppliedAmount;
                                        debitNotesList.Find(debitNote => debitNote.CustomerId == payment.CustomerId && debitNote.OutletId == payment.OutletId && debitNote.AccountId == payment.AccountId && debitNote.EmployeeId == paymentTransaction.TransactionEmployeeId && debitNote.SalesMode == paymentTransaction.SalesMode).GrossTotal += paymentTransaction.AppliedAmount;
                                        debitNotesList.Find(debitNote => debitNote.CustomerId == payment.CustomerId && debitNote.OutletId == payment.OutletId && debitNote.AccountId == payment.AccountId && debitNote.EmployeeId == paymentTransaction.TransactionEmployeeId && debitNote.SalesMode == paymentTransaction.SalesMode).RemainingAmount += paymentTransaction.AppliedAmount;
                                    }
                                    else
                                    {
                                        debitNoteTransaction = new TransactionModel();
                                        // later :  debitNoteTransaction.TransactionId = "";

                                        debitNoteTransaction.CustomerId = payment.CustomerId;
                                        debitNoteTransaction.OutletId = payment.OutletId;
                                        debitNoteTransaction.EmployeeId = paymentTransaction.TransactionEmployeeId;
                                        debitNoteTransaction.AccountId = payment.AccountId;
                                        debitNoteTransaction.DivisionId = payment.DivisionId;
                                        debitNoteTransaction.SalesMode = paymentTransaction.SalesMode;

                                        debitNoteTransaction.RemainingAmount = paymentTransaction.AppliedAmount;
                                        debitNoteTransaction.NetTotal = paymentTransaction.AppliedAmount;
                                        debitNoteTransaction.GrossTotal = paymentTransaction.AppliedAmount;

                                        debitNoteTransaction.TransactionTypeId = TransactionType.DebitNote.GetHashCode();
                                        debitNoteTransaction.TransactionType = TransactionType.DebitNote;
                                        debitNoteTransaction.TransactionDate = DateTime.Now;
                                        debitNoteTransaction.TransactionStatusId = TransactionStatues.Approved.GetHashCode();

                                        debitNoteTransaction.Synchronized = false;
                                        debitNoteTransaction.CurrencyId = paymentTransaction.CurrencyId;
                                        debitNoteTransaction.CreatedDate = DateTime.Now;
                                        debitNoteTransaction.Posted = true;
                                        debitNoteTransaction.Notes = string.Format("{0} {1}", "Debit note from voiding payment ID => ", payment.CustomerPaymentId);
                                        debitNoteTransaction.SourceTransactionId = payment.CustomerPaymentId;

                                        int helperId = -1, driverId = -1, supervisorId = -1, salesRepId = -1, salesManagerId = -1;
                                        _employeeManager.GetEmployeeRelatedStaff(paymentTransaction.TransactionEmployeeId, dbHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepId, ref salesManagerId);

                                        debitNoteTransaction.SupervisorId = supervisorId;
                                        debitNoteTransaction.HelperId = helperId;
                                        debitNoteTransaction.DriverId = driverId;
                                        debitNoteTransaction.SalesRepId = salesRepId;
                                        debitNoteTransaction.SalesManagerId = salesManagerId;
                                        debitNoteTransaction.OrganizationId = _employeeManager.GetEmployeeOrganization(paymentTransaction.TransactionEmployeeId, dbHelper);

                                        debitNotesList.Add(debitNoteTransaction);
                                    }
                                    #endregion
                                }
                                #endregion

                                #region[Update Transaction Remaninig and balances]
                                else if (_requestRepository.Configurations.ChequeBounceMode == ChequeBounceMode.UpdateTransactionRemaining.GetHashCode())
                                {
                                    #region[Update Transaction RemainingAmount]
                                    result = _accountManager.UpdateTransactionRemainingAmountIncrease(dbHelper, paymentTransaction.TransactionId, payment.CustomerId, payment.OutletId, payment.DivisionId, paymentTransaction.AppliedAmount, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now);
                                    if (result != GlobalErrors.Success)
                                    {
                                        return GlobalErrors.Error;
                                    }
                                    #endregion

                                    #region[If Posted => Update Account Balances]
                                    if (paymentTransaction.Posted)
                                    {
                                        #region[Update Customer Account Balance]
                                        if (result == GlobalErrors.Success)
                                        {
                                            result = _accountManager.UpdateAccountBalance(payment.AccountId, paymentTransaction.AppliedAmount, paymentTransaction.TransactionDate, paymentTransaction.TransactionEmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, paymentTransaction.TransactionId, dbHelper, 5);
                                            if (result != GlobalErrors.Success)
                                            {
                                                return GlobalErrors.Error;
                                            }
                                        }
                                        #endregion

                                        #region[Update Employee Account Balance]

                                        if (result == GlobalErrors.Success)
                                        {
                                            if (paymentTransaction.SalesMode == SalesModes.CashSales || (paymentTransaction.SalesMode == SalesModes.CreditSales && _requestRepository.Configurations.UpdateEmployeeBalanceOnCreditSale))
                                            {
                                                int employeeAccountId = -1;
                                                result = _accountManager.GetEmployeeAccountID(paymentTransaction.TransactionEmployeeId, ref employeeAccountId, dbHelper);
                                                if (result == GlobalErrors.Success && employeeAccountId != -1)
                                                {
                                                    result = _accountManager.UpdateAccountBalance(employeeAccountId, paymentTransaction.AppliedAmount, paymentTransaction.TransactionDate, paymentTransaction.TransactionEmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, paymentTransaction.TransactionId, dbHelper, 5);
                                                    if (result == GlobalErrors.Success && _requestRepository.Configurations.ControlTransactionOnDivisionLevel && _requestRepository.Configurations.CheckCreditLimitOnEmployeeDivisionAccount)
                                                    {
                                                        result = _accountManager.UpdateEmployeeDivisionAccountBalance(employeeAccountId, paymentTransaction.AppliedAmount, paymentTransaction.TransactionDate, paymentTransaction.TransactionEmployeeId, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now, paymentTransaction.TransactionId, dbHelper, 5, payment.DivisionId, ref errorMessage);
                                                    }
                                                }
                                                else
                                                {
                                                    // employee must have account
                                                    return GlobalErrors.Error;
                                                }
                                            }
                                        }
                                        #endregion
                                    }
                                    #endregion
                                }
                                if (result != GlobalErrors.Success)
                                {
                                    return result;
                                }
                                #endregion
                            }

                            #region[Save generated debit notes]
                            if (_requestRepository.Configurations.ChequeBounceMode == ChequeBounceMode.CreateDebitNote.GetHashCode() || _requestRepository.Configurations.ChequeBounceMode == ChequeBounceMode.CreateDebitNoteForPDCPostedOrNot.GetHashCode())
                            {
                                foreach (TransactionModel debitNote in debitNotesList)
                                {
                                    string debitNoteID = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, debitNote.EmployeeId, DocumentTypes.DebitNote, debitNote.DivisionId);
                                    if (string.IsNullOrEmpty(debitNoteID))
                                    {
                                        result = GlobalErrors.Error;
                                        return result;
                                    }
                                    debitNote.TransactionId = debitNoteID;
                                    result = _transactionManager.SaveTransactionHeader(debitNote, dbHelper);
                                    if (result != GlobalErrors.Success)
                                    {
                                        return result;
                                    }
                                    if (result == GlobalErrors.Success)
                                    {
                                        result = _documentSequenceManager.UpdateMaxTransactionID(dbHelper, DocumentTypes.DebitNote, debitNote.TransactionId, debitNote.EmployeeId, debitNote.DivisionId);
                                        if (result == GlobalErrors.Success)
                                        {
                                            #region[If Posted => Update Account Balances]
                                            if (payment.Posted)
                                            {
                                                #region[Update Customer Account Balance]
                                                if (result == GlobalErrors.Success)
                                                {
                                                    result = _accountManager.UpdateAccountBalance(debitNote.AccountId, debitNote.NetTotal, debitNote.TransactionDate, debitNote.EmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, debitNote.TransactionId, dbHelper, 5);
                                                    if (result != GlobalErrors.Success)
                                                    {
                                                        return GlobalErrors.Error;
                                                    }
                                                }
                                                #endregion

                                                #region[Update Employee Account Balance]

                                                if (result == GlobalErrors.Success)
                                                {
                                                    if (debitNote.SalesMode == SalesModes.CashSales || (debitNote.SalesMode == SalesModes.CreditSales && _requestRepository.Configurations.UpdateEmployeeBalanceOnCreditSale))
                                                    {
                                                        int employeeAccountId = -1;
                                                        result = _accountManager.GetEmployeeAccountID(debitNote.EmployeeId, ref employeeAccountId, dbHelper);
                                                        if (result == GlobalErrors.Success && employeeAccountId != -1)
                                                        {
                                                            result = _accountManager.UpdateAccountBalance(employeeAccountId, debitNote.NetTotal, debitNote.TransactionDate, debitNote.EmployeeId, Convert.ToInt32(_requestRepository.CurrentOperator.EmployeeId), DateTime.Now, debitNote.TransactionId, dbHelper, 5);
                                                            if (result == GlobalErrors.Success && _requestRepository.Configurations.ControlTransactionOnDivisionLevel && _requestRepository.Configurations.CheckCreditLimitOnEmployeeDivisionAccount)
                                                            {
                                                                result = _accountManager.UpdateEmployeeDivisionAccountBalance(employeeAccountId, debitNote.NetTotal, debitNote.TransactionDate, debitNote.EmployeeId, _requestRepository.CurrentOperator.EmployeeId, DateTime.Now, debitNote.TransactionId, dbHelper, 5, debitNote.DivisionId, ref errorMessage);
                                                            }
                                                        }
                                                        else
                                                        {
                                                            // employee must have account
                                                            return GlobalErrors.Error;
                                                        }
                                                    }
                                                }
                                                #endregion
                                            }
                                            #endregion
                                        }
                                    }
                                }
                            }
                            #endregion

                            #region[Update Customer Payment set Bounced]
                            if (result == GlobalErrors.Success)
                            {
                                payment.PaymentStatusId = PaymentStatusTypes.Bounced;
                                result = _paymentManager.UpdateCustomerPaymentStatus(payment, dbHelper);
                                if (result != GlobalErrors.Success)
                                {
                                    return result;
                                }
                            }
                            #endregion
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dbHelper.CommitTransaction();
                }
                else
                {
                    dbHelper.RollBackTransaction();
                }
                if (dbHelper != null)
                {
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }

        #region [Down Payments]
        public GlobalErrors GetDownPaymentsList(GeneralFilter filter, ref SharedTableResult<PaymentModel> custList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _paymentManager.GetDownPaymentsList(filter, ref custList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors SaveDownPayments(List<PaymentModel> paymentsList, ref string paymentIDs, ref int divisionId, bool isExtraPaymentByCollectInvoice, ref string errorMessage)
        {
            return SaveDownPayments(paymentsList, ref paymentIDs, ref divisionId, isExtraPaymentByCollectInvoice, null, ref errorMessage);
        }

        public GlobalErrors SaveDownPayments(List<PaymentModel> paymentsList, ref string paymentIDs, ref int divisionId, bool isExtraPaymentByCollectInvoice, DBHelper<int> dbHelper, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            bool commitNow = false;
            try
            {
                if (dbHelper == null)
                {
                    dbHelper = new DBHelper<int>();
                    dbHelper.BeginTransaction();
                    commitNow = true;
                }

                foreach (PaymentModel payment in paymentsList)
                {
                    if (!string.IsNullOrEmpty(payment.Notes))
                    {
                        payment.Notes = payment.Notes.Replace("'", "''");
                    }
                    payment.EmployeeId = _requestRepository.CurrentOperator.EmployeeId;
                    payment.OrganizationId = _requestRepository.CurrentOperator.OrganizationId;
                    payment.HelperId = _requestRepository.CurrentOperator.HelperId;
                    payment.DriverId = _requestRepository.CurrentOperator.DriverId;
                    payment.SupervisorId = _requestRepository.CurrentOperator.SupervisorId;
                    payment.SalesRepId = _requestRepository.CurrentOperator.SalesRepId;
                    payment.SalesManagerId = _requestRepository.CurrentOperator.SalesManagerId;

                    payment.CustomerPaymentId = _documentSequenceManager.GetMaxDocumentSequence(dbHelper, payment.EmployeeId, DocumentTypes.Collection, payment.DivisionId);
                    paymentIDs += "'" + payment.CustomerPaymentId + "'" + ",";
                    divisionId = payment.DivisionId;
                    if (string.IsNullOrEmpty(payment.CustomerPaymentId))
                    {
                        errorMessage = MessagesConstants.Desc_No_Document_Sequence;
                        return GlobalErrors.Error;
                    }
                    result = _paymentManager.SaveUnallocatedPayment(payment, dbHelper);

                    if (result == GlobalErrors.Success)
                    {
                        if (result == GlobalErrors.Success)
                        {
                            result = _documentSequenceManager.UpdateMaxTransactionID(dbHelper, DocumentTypes.Collection, payment.CustomerPaymentId, payment.EmployeeId, payment.DivisionId);
                        }
                        if(result == GlobalErrors.Success)
                        {
                            if (_requestRepository.Configurations.CreateCreditNoteOnDownPayment && payment.PaymentTypeId != PaymentTypes.PostedDatedCheque)
                            {
                                result = _transactionManager.CreateCreditNoteFromDownPayment(payment, dbHelper);
                            }// for Posted Dated Cheque in extra payment
                            else if ((_requestRepository.Configurations.CreateCreditNoteOnDownPayment && isExtraPaymentByCollectInvoice))
                            {
                                result = _transactionManager.CreateCreditNoteFromDownPayment(payment, dbHelper);
                            }
                        }
                    }
                }
                paymentIDs = paymentIDs.TrimEnd(',');
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (commitNow)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                }
            }
            return result;
        }
        public GlobalErrors VoidDownPayment(PaymentModel downPayment, ref string voidedMsg)
        {
            DBHelper<int> dbHelper = null;
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();

                #region[Check if voided before]
                if (_paymentManager.IsDownPaymentVoided(downPayment, dbHelper))
                {
                    voidedMsg = ResourcesManager.TranslateKey(MessagesConstants.Desc_Selected_Payment_Voided, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                #endregion

                #region[Check if payment synchronized]
                if (_paymentManager.IsDownPaymentSynchronized(downPayment, dbHelper))
                {
                    voidedMsg = ResourcesManager.TranslateKey(MessagesConstants.MsgPaymentSynchronized, _requestRepository.LanguageId);
                    return GlobalErrors.Error;
                }
                #endregion

                #region[Check for CN generated from payment]
                DBHelper<TransactionModel> dBHelperCN = new DBHelper<TransactionModel>(dbHelper.GetConnection(), dbHelper.GetDBTransaction());
                TransactionModel CNTransaction = null;
                result = _transactionManager.GetCreditNoteGeneratedFromDownPayment(downPayment.CustomerPaymentId, downPayment.CustomerId, downPayment.OutletId, downPayment.DivisionId, ref CNTransaction, dBHelperCN);

                if (result == GlobalErrors.Success && CNTransaction != null)
                {
                    if (_paymentManager.CheckUploadedCreditNote(CNTransaction.CustomerId, CNTransaction.OutletId, CNTransaction.TransactionDate, dbHelper))
                    {
                        voidedMsg = ResourcesManager.TranslateKey(MessagesConstants.MsgCannotVoidPaymentForUploadedCN, _requestRepository.LanguageId);
                        return GlobalErrors.Error;
                    }
                    if (_paymentManager.CheckIfCreditNoteUsedInPayment(CNTransaction.TransactionId, CNTransaction.DivisionId, dbHelper))
                    {
                        voidedMsg = ResourcesManager.TranslateKey(MessagesConstants.MsgCannotVoidPaymentForUsedCreditNote, _requestRepository.LanguageId);
                        return GlobalErrors.Error;
                    }
                }
                #endregion

                #region[Void Payment]
                if (result == GlobalErrors.Success)
                {
                    result = _paymentManager.VoidDownPayment(downPayment, dbHelper);
                }
                #endregion

                #region[Void Related CN]
                if (result == GlobalErrors.Success && _requestRepository.Configurations.CreateCreditNoteOnDownPayment && downPayment.PaymentTypeId != PaymentTypes.PostedDatedCheque)
                {
                    result = _transactionManager.SetTransactionAsVoided(CNTransaction.TransactionId, downPayment.CustomerId, downPayment.OutletId, downPayment.DivisionId, dbHelper);
                }
                #endregion
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        #endregion
        public SavePaymentModel PrepareSavePaymentObjectForReturnOrder(TransactionModel debitTransaction, PaymentModel payment,int customerId, int outletId)
        {
            SavePaymentModel savePaymentObj = null;
            try
            {
                savePaymentObj = new SavePaymentModel();
                savePaymentObj.CustomerId = customerId;
                savePaymentObj.OutletId = outletId;
                savePaymentObj.DivisionId = _requestRepository.Configurations.ControlTransactionOnDivisionLevel ? debitTransaction.DivisionId : -1;
                savePaymentObj.EmployeeId = _requestRepository.CurrentOperator.EmployeeId;
                savePaymentObj.TransactionsList.Add(debitTransaction);
                savePaymentObj.PaymentsList.Add(payment);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return null;
            }
            return savePaymentObj;
        }
        public GlobalErrors SaveCRMEPayments(SavePaymentModel savePaymentModel, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = SaveOnlineEPayments(savePaymentModel, ref errorMessage); // save payment to DB , save payment group && reflect transaction remaning value
                if (result == GlobalErrors.Success)
                {
                    result =_paymentManager.SendPaymentLink(savePaymentModel); // send payment link, save third part transaction, save online response
                    if (result != GlobalErrors.Success)
                    {
                        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, "Failed to save Online e-payment respone" + "PaymentID => [ " + savePaymentModel.PaymentsList[0].CustomerPaymentId + " ] UUID => [ " + savePaymentModel.PaymentsList[0].UUID + " ]", null, null, 0);
                        errorMessage = "Error happened while sending E-Payment link !";

                        GlobalErrors voidingResult = _paymentManager.VoidOnlineEPayment(savePaymentModel.PaymentsList[0]);
                        if(voidingResult != GlobalErrors.Success)
                        {
                            result = GlobalErrors.Error;
                            errorMessage = "Error happened while voiding E-Payment related to error in payment link response!";
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveOnlineEPayments(SavePaymentModel savePaymentModel, ref string errorMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                savePaymentModel.EmployeeId = _requestRepository.CurrentOperator.EmployeeId;

                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();

                result = _paymentManager.SaveOnlineEPayments(savePaymentModel, ref errorMessage, dbHelper);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    if (dbHelper != null)
                    {
                        dbHelper.Dispose();
                        dbHelper = null;
                    }
                }
            }
            return result;
        }
        
        #region[Post PDC]
        public GlobalErrors GetPDCSharedTable(GeneralFilter filter, ref List<PaymentModel> pdcSharedTable)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _paymentManager.GetPDCSharedTable(filter, ref pdcSharedTable);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors PostPDC(List<PaymentModel> chequesList, ref int numberOfPosted, ref int numberOfBounced)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dBHelper = null;
            DBHelper<PaymentModel> dBHelper_PaymentModel = null;
            numberOfPosted = 0;
            numberOfBounced = 0;
            try
            {
                if(chequesList != null && chequesList.Count > 0)
                {
                    dBHelper = new DBHelper<int>();
                    dBHelper.BeginTransaction();
                    foreach (PaymentModel cheque in chequesList)
                    {
                        int customerId = cheque.CustomerId;
                        int employeeId = cheque.EmployeeId;
                        bool chequeBounced = false;
                        result = CheckIfBouncedCheque(cheque.EmployeeId, cheque.CustomerId, cheque.VoucherDateModel.Date, cheque.VoucherNumber, ref chequeBounced, dBHelper);
                        if (result != GlobalErrors.Success) break;
                        if (chequeBounced)
                        {
                            numberOfBounced++;
                            continue;
                        }
                        else
                        {
                            if (dBHelper != null)
                            {
                                dBHelper_PaymentModel = new DBHelper<PaymentModel>(dBHelper.GetConnection(), dBHelper.GetDBTransaction());
                            }
                            List<PaymentModel> payments = new List<PaymentModel>();
                            result = GetAllPostedPayments(cheque.EmployeeId, cheque.CustomerId, cheque.VoucherDateModel.Date, cheque.VoucherNumber, ref payments, dBHelper_PaymentModel);
                            if(result == GlobalErrors.Success)
                            {
                                if(payments != null && payments.Count > 0)
                                {
                                    foreach (PaymentModel payment in payments)
                                    {
                                        int outletId = payment.OutletId;
                                        string errorMessage = string.Empty;
                                        result = _paymentManager.PostCustomerPayments(employeeId, payment.CustomerPaymentId, customerId, outletId, true, dBHelper, ref errorMessage);
                                        if (result != GlobalErrors.Success) break;
                                    }
                                    if (result == GlobalErrors.Success)
                                    {
                                        numberOfPosted++;
                                    }
                                }
                                else
                                {
                                    numberOfPosted++;
                                    continue;
                                }
                            }
                            else
                            {
                                result = GlobalErrors.Error;
                                break;
                            }
                        }
                    }
                }
                else
                {
                    result =  GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result =  GlobalErrors.Error;
            }
            finally
            {
                if (dBHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dBHelper.CommitTransaction();
                    }
                    else
                    {
                        dBHelper.RollBackTransaction();
                    }
                    dBHelper.Dispose();
                    dBHelper = null;
                }
            }
            return result;
        }

        public GlobalErrors CheckIfBouncedCheque(int employeeId, int customerId,DateTime voucherDate, string voucherNo, ref bool chequeBouched ,DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                object field = null;
                string query = string.Format(@"Select Top 1 PaymentStatusID From CustomerPayment Where EmployeeID = {0} And CustomerID = {1} And VoucherDate = {2} AND VoucherNumber='{3}' And PaymentStatusID In({4},{5})", employeeId, customerId, LocalUtilities.ParseDateAndTimeToSQL(voucherDate), voucherNo, PaymentStatusTypes.Voided.GetHashCode(), PaymentStatusTypes.Bounced.GetHashCode());
                result = dBHelper.ExecuteScalar(query, ref field);
                if (result == GlobalErrors.Success && field != null && !string.IsNullOrEmpty(field.ToString()))
                    chequeBouched =  true;
                else
                    chequeBouched = false;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetAllPostedPayments(int employeeId, int customerId, DateTime voucherDate, string voucherNo, ref List<PaymentModel> payments, DBHelper<PaymentModel> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string query = string.Format("Select Distinct CustomerPayment.CustomerPaymentID,CustomerPayment.OutletID From CustomerPayment Where EmployeeID = {0} And CustomerID = {1} And VoucherDate = {2} And VoucherNumber = '{3}' And Posted = 0", employeeId, customerId, LocalUtilities.ParseDateAndTimeToSQL(voucherDate), voucherNo);
                result = dBHelper.GetQueryList(query, ref payments);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                payments = null;
                return GlobalErrors.Error;
            }
            return result;
        }

        #endregion
        public GlobalErrors CancelOnlinePayments(List<PaymentModel> selectedPayments, ref List<string> errorMsglist)
        {
            DBHelper<int> dbHelper = new DBHelper<int>();
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                foreach (PaymentModel payment in selectedPayments)
                {
                    payment.HasError = true;
                    if (_routeManager.IsRouteUploaded(payment.RouteHistoryId, dbHelper))
                    {
                        errorMsglist.Add(string.Format("{0} : {1}", payment.CustomerPaymentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Uploaded_Route, _requestRepository.LanguageId)));
                        continue;
                    }
                    if (_paymentManager.CheckIfMainPaymentVoided(payment, dbHelper))
                    {
                        errorMsglist.Add(string.Format("{0} : {1}", payment.CustomerPaymentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Voided_Before, _requestRepository.LanguageId)));
                        continue;
                    }
                    if (_paymentManager.CancelOnlineEPayments(payment) != GlobalErrors.Success)
                    {
                        errorMsglist.Add(string.Format("{0} : {1}", payment.CustomerPaymentId, ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_Update_Data, _requestRepository.LanguageId)));
                        continue;
                    }
                    payment.HasError = false;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
    }
}