﻿using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.SignalR;
using Sonic.Web.Core;
using Sonic.Web.DAL;
using Sonic.Web.Resources;
using Sonic.Web.Model;
using System;
using System.Collections.Generic;
using System.IO;
using System.Net.Mail;
using System.Reflection;
using Newtonsoft.Json;

namespace Sonic.Web.Service
{
    public class PricesAndOffersService
    {
        private readonly IRequestRepository _requestRepository;
        private readonly PricesAndOffersManager _pricesAndOffersManager;
        private readonly EmailService _emailService;
        private readonly CustomerService _customerService;
        private readonly EmployeeManager _employeeManager;
        private readonly NotificationService _notificationService;
        private IHostingEnvironment _hostingEnvironment;
        private readonly CustomerManager _customerManager;

        public PricesAndOffersService(IHubContext<NotifyHub, ITypedHubClient> hubContext, IRequestRepository requestRepository, IHostingEnvironment? hostingEnvironment )
        {
            _requestRepository = requestRepository;
            _emailService = new EmailService(_requestRepository);
            _pricesAndOffersManager = new PricesAndOffersManager(requestRepository);
            _customerService = new CustomerService(requestRepository);
            _employeeManager = new EmployeeManager(requestRepository);
            _notificationService = new NotificationService(hubContext, _requestRepository);
            _customerManager = new CustomerManager(_requestRepository);
            _hostingEnvironment = hostingEnvironment;
        }
        #region Price

        public GlobalErrors GetPriceLists(GeneralFilter filter, ref SharedTableResult<PriceListModel> priceLists)
        {
            try
            {

                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _pricesAndOffersManager.GetPriceLists(filter, ref priceLists);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                priceLists = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors UpdatePendingPriceLists(PendingPriceListModel priceLists)
        {
            try
            {

              
                return _pricesAndOffersManager.UpdatePendingPriceLists(priceLists);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                priceLists = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetPendingPriceLists(GeneralFilter filter, ref SharedTableResult<PendingPriceListModel> priceLists)
        {
            try
            {

                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _pricesAndOffersManager.GetPendingPriceLists(filter, ref priceLists);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                priceLists = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetSalesOrderStatusTypes(ref List<OrderModel> salesOrderTypes, List<int> orderStatuses)
        {
            try
            {
                return _pricesAndOffersManager.GetSalesOrderStatusTypes(ref salesOrderTypes, orderStatuses);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                salesOrderTypes = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SaveCustomerPriceList(SimpleCustomerModel customerPricList)
        {
                GlobalErrors result = GlobalErrors.NotInitialized;
                DBHelper<int> dBHelper = new DBHelper<int>();
            try
            {

                dBHelper.BeginTransaction();

                result = _pricesAndOffersManager.InsertcustomerPriceList(customerPricList, dBHelper);

                if (result == GlobalErrors.Success)
                {
                    foreach (TextLanguage textLanguage in customerPricList.DescriptionLanguages)
                    {
                        if (string.IsNullOrEmpty(textLanguage.Description))
                        {
                            textLanguage.Description = customerPricList.CustomerName;
                        }
                        result = _customerManager.UpdateCustomerLanguage(customerPricList.CustomerId, textLanguage.LanguageId, textLanguage.Description, dBHelper);
                        if (result != GlobalErrors.Success)
                        {
                            return result;
                        }
                    }

                }
                if (result == GlobalErrors.Success)
                {
                    result = _customerManager.InsertCustomerOrganization(customerPricList.CustomerId, _requestRepository.CurrentOperator.OrganizationId, dBHelper);
                }

                    if (result == GlobalErrors.Success && customerPricList.CustomerOutletContacts != null && customerPricList.CustomerOutletContacts.Count > 0)
                    {
                        result = _customerManager.SaveCustomerOutletContacts(customerPricList.CustomerOutletContacts, customerPricList.CustomerId, customerPricList.OutletId, dBHelper);
                    }
                

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dBHelper.CommitTransaction();
                }
                else
                {
                    dBHelper.RollBackTransaction();
                }
            }


        }
        
        public GlobalErrors DeletePriceList(PriceListModel price, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dBHelper = new DBHelper<int>();
                dBHelper.BeginTransaction();

                if (_requestRepository.Configurations.ApplyApprovalCycleInWeb.Contains("4"))
                {
                    int currentPriceStatus = 0;
                    bool isDeleted = false;
                    int originalStatus = _pricesAndOffersManager.GetCurrentStatus();
                    result = _pricesAndOffersManager.GetPriceListStatus(price.PriceListId, ref currentPriceStatus, ref isDeleted);
                    if (_requestRepository.Configurations.ForcePreviousLevelApproval)
                    {
                        if (currentPriceStatus != 0)// delete from orginal 
                        {
                            if (isDeleted)
                            {
                                if (originalStatus == -1)// max level 
                                {
                                    int maxLevel = _pricesAndOffersManager.GetMaxApprovalCycleLevels(ApprovalCycleOperations.Price.GetHashCode());
                                    if (currentPriceStatus != -1 && currentPriceStatus != maxLevel)
                                    {
                                        result = GlobalErrors.ValueIsMoreThanRequested;
                                        return result;
                                    }
                                }
                                else
                                {
                                    if(originalStatus != price.Status)
                                    {
                                        result = GlobalErrors.ValueIsMoreThanRequested;
                                        return result;
                                    }
                                    if (originalStatus != currentPriceStatus)
                                    {
                                        result = GlobalErrors.ValueIsMoreThanRequested;
                                        return result;
                                    }
                                }
                            }
                            else
                            {
                                if(originalStatus == -1)
                                {
                                    originalStatus = _pricesAndOffersManager.GetMaxApprovalCycleLevels(ApprovalCycleOperations.Price.GetHashCode());
                                }
                                if (originalStatus != currentPriceStatus && currentPriceStatus != -1)//when delete pending and get active // and isDeleted
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                                if (originalStatus != price.Status && price.Status != -1)
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                            }
                        }
                        else
                        {
                            result = GlobalErrors.ValueIsMoreThanRequested;
                            return result;
                        }
                    }
                    else
                    {
                        if (currentPriceStatus == 0)
                        {
                            result = GlobalErrors.ValueIsMoreThanRequested;
                            return result;
                        }
                    }

                    PriceListModel priceListModel = new PriceListModel();
                    PriceListModel priceListModelPending = new PriceListModel();
                    
                    if (_pricesAndOffersManager.CheckIsPriceListPendingExisit(price.PriceListId))
                    {
                        result = _pricesAndOffersManager.DeletePriceListPending(price.PriceListId, dBHelper);
                        if (result == GlobalErrors.Success)
                        {
                            result = _pricesAndOffersManager.InsertPriceCycleHistory(price.PriceListId, 3, _requestRepository.CurrentOperator.EmployeeId, price.Status, price.Status, -1, price.RejectNote, dBHelper);
                        }
                    }
                    else
                    {
                        result = _pricesAndOffersManager.DeleteActivePriceList(price, dBHelper);
                        if (result == GlobalErrors.Success)
                        {
                            result = _pricesAndOffersManager.InsertPriceCycleHistory(price.PriceListId, 3, _requestRepository.CurrentOperator.EmployeeId, price.Status, price.Status, originalStatus, price.RejectNote, dBHelper);

                            //result = _pricesAndOffersManager.DeletePriceList(price.PriceListId);
                        }
                    }
                    return result;
                }
                else
                {
                    result = _pricesAndOffersManager.DeletePriceList(price.PriceListId, dBHelper);
                    if(result == GlobalErrors.Success)
                    {
                        result = _pricesAndOffersManager.DeletePriceListPending(price.PriceListId, dBHelper);
                    }
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dBHelper.CommitTransaction();
                }
                else
                {
                    dBHelper.RollBackTransaction();
                }
            }
        }
        public GlobalErrors DeletePriceListPending(int priceListId)
        {
      
            try
            {
                return _pricesAndOffersManager.DeletePriceListPending(priceListId, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors DeleteCustomerApplied(ref List<SimpleCustomerModel>  appliedCustomers)
        {

            try
            {
                return _pricesAndOffersManager.DeleteAppliedCustomer(ref appliedCustomers);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListTypes(ref List<PriceListTypesModel> priceListTypes)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListTypes( ref priceListTypes);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                priceListTypes = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListHeader(int priceListID, ref PriceListModel priceList)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListHeader( priceListID, ref priceList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                priceList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListHeaderPending(int priceListID, ref PriceListModel priceList)
        {
            try
            {
                return _pricesAndOffersManager.GetPriceListHeaderPending(priceListID, ref priceList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                priceList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListDescriptions(int priceListId, ref List<TextLanguage> descriptions)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListDescriptions(priceListId, ref descriptions);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                descriptions = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListDescriptionsPending(int priceListId, ref List<TextLanguage> descriptions)
        {
            try
            {
                return _pricesAndOffersManager.GetPriceListDescriptionsPending(priceListId, ref descriptions);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                descriptions = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListAppliedGroups(int priceListId, ref List<CustomerGroupModel> appliedGroups)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListAppliedGroups(priceListId, ref appliedGroups);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedGroups = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListAppliedGroupsPending(int priceListId, ref List<CustomerGroupModel> appliedGroups)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListAppliedGroupsPending(priceListId, ref appliedGroups, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedGroups = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListAppliedSDCs(int priceListId, ref List<SalesDistributionChannelModel> appliedSDCs)
        {
            try
            {
                return _pricesAndOffersManager.GetPriceListAppliedSDCs(priceListId, ref appliedSDCs);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedSDCs = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetPriceListAppliedSDCsPending(int priceListId, ref List<SalesDistributionChannelModel> appliedSDCs)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try 
            {
                result = _pricesAndOffersManager.GetPriceListAppliedSDCsPending(priceListId, ref appliedSDCs, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedSDCs = null;
                return GlobalErrors.Error;
            }
            return result;
        }

        public GlobalErrors GetPriceListAppliedChannels(int priceListId, ref List<CustomerChannelModel> appliedChannels)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListAppliedChannels(priceListId, ref appliedChannels);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedChannels = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListAppliedChannelsPending(int priceListId, ref List<CustomerChannelModel> appliedChannels)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListAppliedChannelsPending(priceListId, ref appliedChannels, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedChannels = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListItems(int priceListId, ref List<PriceListItemsModel> priceListItems)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListItems(priceListId,ref priceListItems);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                priceListItems = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListItemsPending(int priceListId, ref List<PriceListItemsModel> priceListItems)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListItemsPending(priceListId, ref priceListItems, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                priceListItems = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListAppliedCustomers(int priceListId, ref List<SimpleCustomerModel> appliedCustomers)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListAppliedCustomers(priceListId, ref appliedCustomers);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedCustomers = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPriceListAppliedCustomersPending(int priceListId, ref List<SimpleCustomerModel> appliedCustomers)
        {
            try
            {

                return _pricesAndOffersManager.GetPriceListAppliedCustomersPending(priceListId, ref appliedCustomers, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedCustomers = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SavePriceList(PriceListModel priceListObj)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dBHelper = new DBHelper<int>();
            dBHelper.BeginTransaction();
            try
            {
                if (_requestRepository.Configurations.ApplyApprovalCycleInWeb.Contains("4"))
                {
                    int employeeStatus = _pricesAndOffersManager.GetCurrentStatus();
                    if (priceListObj.PriceListId != -1)
                    {
                        int priceListStatus = 0;
                        bool isDeleted = false;
                        result = _pricesAndOffersManager.GetPriceListStatus(priceListObj.PriceListId, ref priceListStatus, ref isDeleted);

                        if (_requestRepository.Configurations.ForcePreviousLevelApproval)
                        {
                            if (employeeStatus == -1)//max level
                            {
                                int maxLevel = _pricesAndOffersManager.GetMaxApprovalCycleLevels(ApprovalCycleOperations.Price.GetHashCode());
                                if (priceListStatus != maxLevel && priceListStatus != -1)//priceListStatus != -1 for Active Price
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                            }
                            else
                            {
                                if ((isDeleted && employeeStatus != priceListStatus) || employeeStatus != priceListStatus && priceListStatus != -1)//priceListStatus != -1 for Active Price // priceListStatus != 0 found on ABC_20200207 DB
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                                else
                                if (employeeStatus != priceListObj.Status && priceListStatus != -1)//priceListStatus != -1 for Active Price // priceListStatus != 0 found on ABC_20200207 DB
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                            }

                        }
                    }
                   
                    
                    SettingManager settingManager = new SettingManager(_requestRepository);
                   
                    if (priceListObj.PriceListId <= 0)
                    {
                        if (employeeStatus == -1)//check tax type delete
                        {
                            result = _pricesAndOffersManager.InsertPriceList(priceListObj,dBHelper);
                            int level = priceListObj.Status;
                            if (employeeStatus != priceListObj.Status)
                                level = employeeStatus;
                            result = _pricesAndOffersManager.InsertPriceCycleHistory(priceListObj.PriceListId, 1, _requestRepository.CurrentOperator.EmployeeId, level, priceListObj.Status, employeeStatus, priceListObj.RejectNote, dBHelper);
                        }
                        else
                        {
                            result = _pricesAndOffersManager.InsertPriceListPending(priceListObj, dBHelper);
                            if (result == GlobalErrors.Success)
                            {
                                int level = priceListObj.Status;
                                if (employeeStatus != priceListObj.Status)
                                    level = employeeStatus;
                                result = _pricesAndOffersManager.InsertPriceCycleHistory(priceListObj.PriceListId, 1, _requestRepository.CurrentOperator.EmployeeId, level, priceListObj.Status, employeeStatus, priceListObj.RejectNote, dBHelper);
                            }
                        }
                    }

                    else
                    {
                        if (priceListObj.Status != 0 && priceListObj.Status != -1)// priceListObj.Status != -1 for Active Price
                        {

                            result = _pricesAndOffersManager.UpdatePriceListPending(priceListObj, dBHelper);
                            if (!priceListObj.IsCreationLevel)
                            {
                                int level = -1;
                                bool isValid = false;
                                result = IsValidToApprove(priceListObj.Status, ApprovalCycleOperations.Price.GetHashCode(), -1,ref isValid);
                                if (isValid)
                                {

                                    result = ApproveOperation(priceListObj.PriceListId, priceListObj.Status, ApprovalCycleOperations.Price.GetHashCode(), ref level, dBHelper);
                                    if (result != GlobalErrors.Success) return result;
                                  

                                }
                                
                            }
                            if (priceListObj.IsDeleted && employeeStatus == -1)//for max employee level 
                            {
                                result = _pricesAndOffersManager.DeletePriceListPending(priceListObj.PriceListId, dBHelper);
                                if (result == GlobalErrors.Success)
                                {
                                    result = _pricesAndOffersManager.DeletePriceList(priceListObj.PriceListId, dBHelper);
                                }
                            }

                        }
                        else
                        {
                            bool isMaxLevel = false;
                            result = settingManager.IsMaxOperationLevel(ApprovalCycleOperations.Price.GetHashCode(), ref isMaxLevel, dBHelper);
                            if (isMaxLevel) // note #1
                            {
                                result = _pricesAndOffersManager.UpdatePriceList(priceListObj, dBHelper);
                            }
                            else
                            {
                                priceListObj.Status = _pricesAndOffersManager.GetCurrentStatus();
                                result = _pricesAndOffersManager.InsertPriceListPending(priceListObj, dBHelper);
                                if (result == GlobalErrors.Success)
                                {
                                    //result = _pricesAndOffersManager.UpdatePriceListPending(priceListObj, dBHelper);
                                    int level = -1;
                                    result = _pricesAndOffersManager.InsertPriceCycleHistory(priceListObj.PriceListId, 2, _requestRepository.CurrentOperator.EmployeeId, level, priceListObj.Status, employeeStatus, priceListObj.RejectNote, dBHelper);
                                }
                            }

                        }
                    }

                }
                else
                {
                    if (priceListObj.PriceListId <= 0)
                    {
                        result = _pricesAndOffersManager.InsertPriceList(priceListObj, dBHelper);
                    }
                    else
                    {
                        result = _pricesAndOffersManager.UpdatePriceList(priceListObj, dBHelper);
                    }
                }

                if (result == GlobalErrors.SuccessWithZeroRowAffected) result = GlobalErrors.Success;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dBHelper.CommitTransaction();
                }
                else
                {
                    dBHelper.RollBackTransaction();
                }
            }
            return result;
        }
        public bool CheckIsPriceListPendingExisit(int priceListId)
        {
            try
            {
                return _pricesAndOffersManager.CheckIsPriceListPendingExisit(priceListId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return false;
            }
        }

        public GlobalErrors CheckAllowEditPromotionByOrgAccess(int CreatedBy, ref bool isValid)
        {
            try
            {
                return _pricesAndOffersManager.CheckAllowEditPromotionByOrgAccess(CreatedBy, ref isValid);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }


        public GlobalErrors GetPriceListStatus(int priceListID, ref int status, ref bool isDeleted)
        {
            try
            {
                return _pricesAndOffersManager.GetPriceListStatus(priceListID, ref status, ref isDeleted);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error; ;
            }
        }

        public GlobalErrors GetTaxTypes(ref List<TaxTypesModel> TaxTypes)
        {
            try
            {
                return _pricesAndOffersManager.GetTaxTypes(ref TaxTypes);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

       
        # endregion

        #region Discount

        public GlobalErrors GetDiscounts(GeneralFilter filter, ref SharedTableResult<DiscountModel> discounts)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _pricesAndOffersManager.GetDiscounts(filter, ref discounts);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                discounts = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors DeleteDiscount(int discountId)
        {
            try
            {
                return _pricesAndOffersManager.DeleteDiscount(discountId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetDiscountTypes(ref List<DiscountTypesModel> discountTypes)
        {
            try
            {
                return _pricesAndOffersManager.GetDiscountTypes(ref discountTypes);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                discountTypes = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetSalesOrderTypes(ref List<DiscountModel> salesOrderTypes)
        {
            try
            {
                return _pricesAndOffersManager.GetSalesOrderTypes(ref salesOrderTypes);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                salesOrderTypes = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetDiscountDescriptions(int discountId, bool isClone, ref List<TextLanguage> descriptions)
        {
            try
            {
                return _pricesAndOffersManager.GetDiscountDescriptions(discountId, isClone, ref descriptions);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                descriptions = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetDiscountAppliedGroups(int discountId, ref List<CustomerGroupModel> appliedGroups)
        {
            try
            {
                return _pricesAndOffersManager.GetDiscountAppliedGroups(discountId, ref appliedGroups);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedGroups = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetDiscountAppliedChannels(int discountId, ref List<CustomerChannelModel> appliedChannels)
        {
            try
            {
                return _pricesAndOffersManager.GetDiscountAppliedChannels(discountId, ref appliedChannels);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedChannels = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetDiscountAppliedCustomers(int discountId, ref List<SimpleCustomerModel> appliedCustomers)
        {
            try
            {
                return _pricesAndOffersManager.GetDiscountAppliedCustomers(discountId, ref appliedCustomers);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedCustomers = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetDiscountAppliedSDCs(int discountId, ref List<SalesDistributionChannelModel> sdcsList)
        {
            try
            {
                return _pricesAndOffersManager.GetDiscountAppliedSDCs(discountId, ref sdcsList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SaveDiscount(DiscountModel discountObj)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (discountObj.DiscountId <= 0)
                {
                    result = _pricesAndOffersManager.InsertDiscount(discountObj);
                }
                else
                {
                    result = _pricesAndOffersManager.UpdateDiscount(discountObj);
                }
                if (result == GlobalErrors.SuccessWithZeroRowAffected) result = GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors UpdateDiscountActiveStatus(int discountId, bool isActive)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _pricesAndOffersManager.UpdateDiscountActiveStatus(discountId, isActive);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        #endregion

        #region[Promotions]
        public GlobalErrors GetPromotionsListSharedTable(PromotionFilterModel filter, ref SharedTableResult<Promotion> promotionsSharedList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _pricesAndOffersManager.GetPromotionsListSharedTable(filter, ref promotionsSharedList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                promotionsSharedList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SavePromotion(Promotion promotion, IFormFileCollection Files)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();

                result = _pricesAndOffersManager.SavePromotionProcess(promotion, dbHelper);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        #region [Promotion Notification]
        public GlobalErrors SendNotification(Promotion promotion, IHostingEnvironment _hostingEnvironment, ref string errorMsg)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            NotificaitonsModel notification = new NotificaitonsModel();
            try
            {
                if (promotion.Resend)
                {
                    bool SendNotificationToFO = promotion.SendNotificationToFO;
                    result = _pricesAndOffersManager.GetPromotionData(promotion.PromotionId, false, ref promotion);
                    promotion.SendNotification = promotion.SendNotification;
                    promotion.SendNotificationToFO = SendNotificationToFO;
                }
                else
                {
                    result = GlobalErrors.Success;
                }
                if (result == GlobalErrors.Success && promotion.SendNotification)
                {
                    List<string> imagesNames = new List<string>();
                    List<string> attachementNames = new List<string>();
                    FileManagerModel model = new FileManagerModel();
                    string folderName = Path.Combine("Resources\\promotions-img\\", promotion.PromotionId.ToString());
                    string rootPath = _hostingEnvironment.ContentRootPath;
                    var userImagesPath = Path.Combine(rootPath, folderName);
                    DirectoryInfo dir = new DirectoryInfo(userImagesPath);
                    if (dir.Exists)
                    {
                        FileInfo[] files = dir.GetFiles();
                        model.Files = files;
                        int maxId = -1;
                        result = _customerService.GetMaxNotificationId(ref maxId);
                        if (result == GlobalErrors.Success)
                        {
                            notification.NotificationId = maxId;
                        }
                        if (files != null && files.Length > 0)
                        {
                            foreach (FileInfo file in files)
                            {
                                imagesNames.Add(file.Name);
                                result = UploadNotificationImage(file, ref notification);

                            }
                        }
                    }
                    if (imagesNames != null && imagesNames.Count > 0)
                    {
                        promotion.ImageList = imagesNames;
                    }
                    result = _pricesAndOffersManager.SendNotification(promotion);
                    if (result == GlobalErrors.Error || result == GlobalErrors.NotInitialized)
                    {
                        errorMsg = ResourcesManager.TranslateKey(MessagesConstants.Desc_Notification_Sent_Failed_To_Echo, _requestRepository.LanguageId);
                        promotion.FailedSentToEcho = true;
                    }
                }
                if (promotion.SendNotificationToFO)
                {
                    result = SendPromotionNotificationToFo(promotion);
                    if (result == GlobalErrors.Error || result == GlobalErrors.NotInitialized)
                    {
                        errorMsg = ResourcesManager.TranslateKey(MessagesConstants.Desc_Notification_Sent_Failed_To_Android, _requestRepository.LanguageId);
                        promotion.FailedSentToAndroid = true;
                    }
                }
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
        public GlobalErrors SendPromotionNotificationToFo(Promotion promotion)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            NotificationModel notification = new NotificationModel();
            List<EmployeeModel> employeesList;
            List<int> promotionEmployees = new List<int>();
            List<string> deviceTokenList = new List<string>();
            List<NotificationTextLanguage> descriptionList = new List<NotificationTextLanguage>();
            string body = "";
            string arBody = "";
            NotificationTextLanguage t1 = new NotificationTextLanguage();
            NotificationTextLanguage t2 = new NotificationTextLanguage();
            try
            {
                employeesList = new List<EmployeeModel>();
                result = _employeeManager.GetAllEmployeePromotion(promotion,ref employeesList);
                if (result != GlobalErrors.Success)
                {
                    return result;
                }
                if (employeesList != null && employeesList.Count > 0)
                {
                    foreach (var employee in employeesList)
                    {
                        promotionEmployees.Add(employee.EmployeeId);
                        deviceTokenList.Add(employee.DeviceToken); 
                    }
                    if (promotionEmployees != null && promotionEmployees.Count > 0 && deviceTokenList != null && deviceTokenList.Count > 0)
                    {
                        if (!string.IsNullOrEmpty(LocalUtilities.SonicFONotificationID) && !string.IsNullOrEmpty(LocalUtilities.SonicFONotificationSenderID))
                        {
                            notification.SenderId = _requestRepository.CurrentOperator.EmployeeId;
                            notification.NotificationDate = DateTime.Now;
                            notification.NotificationTypeId = NotificationTypes.Promotion;
                            notification.IsBOUser = false;
                            notification.ReceiversList = promotionEmployees;
                            notification.DeviceTokenList = deviceTokenList;
                            body = ResourcesManager.TranslateKey(MessagesConstants.Desc_Promotion_Updated, 1);
                            arBody = ResourcesManager.TranslateKey(MessagesConstants.Desc_Promotion_Updated, 2);
                            t1.BodyDescription = body;
                            t1.LanguageId = 1;
                            t2.BodyDescription = arBody;
                            t2.LanguageId = 2;
                            descriptionList.Add(t1);
                            descriptionList.Add(t2);
                            descriptionList.ForEach(a =>
                            {
                                    a.HeaderDescription = string.Empty;
                                    a.BodyDescription = string.Format(ResourcesManager.TranslateKey(MessagesConstants.Desc_Promotion_Updated, a.LanguageId));
                            });
                            notification.DescriptionsList = descriptionList;

                            var data = new Object();
                            data = new
                            {
                                registration_ids = notification.DeviceTokenList,
                                data = new
                                {
                                    type = notification.NotificationTypeId,
                                    notification
                                }
                            };

                            string jsonBody = JsonConvert.SerializeObject(data);
                            notification.jsonBody = jsonBody;
                            result = _notificationService.SendNotification(notification, null);
                            if (result != GlobalErrors.Success)
                            {
                                return result;
                            }
                        }
                    }
                }
            }

            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        #endregion
        public GlobalErrors UploadNotificationImage(FileInfo image, ref NotificaitonsModel CustomerNotificaitons)
        {
            try
            {
                if (image == null)
                {
                    return GlobalErrors.Error;
                }

                string path = string.Format(@"{0}", LocalUtilities.CustomerAppServicePath);
                string folderName = "wwwroot\\notifications\\images";
                string newPath = Path.Combine(path, folderName);
                var file = image;

                if (!Directory.Exists(newPath))
                {
                    Directory.CreateDirectory(newPath);
                }

                if (file.Length > 0)
                {
                    //string fileName = file.FileName;// ContentDispositionHeaderValue.Parse(file.ContentDisposition).FileName.Trim('"');
                    string extension = Path.GetExtension(file.Name).ToLowerInvariant();
                    if (!string.IsNullOrEmpty(extension))
                    {
                        string fileName = CustomerNotificaitons.NotificationId + extension;
                        string fullPath = Path.Combine(newPath, fileName);
                        file.CopyTo(fullPath,true);
                        CustomerNotificaitons.ImageName = fileName;
                    }
                    else
                    {
                        return GlobalErrors.Error;
                    }

                }
                return GlobalErrors.Success;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetMaxNotificationId(ref int maxId)
        {
            try
            {
                maxId = CustomerManager.GetMaxID(CoreDataBaseConstants.TableNames.EchoNotifications, CoreDataBaseConstants.QueryColumnsNames.NotificationId, null);
                if (maxId == -1)
                {
                    return GlobalErrors.Error;
                }
                else
                {
                    return GlobalErrors.Success;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetPromotionData(int promotionId, bool isClone, ref Promotion selectedPromotion)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _pricesAndOffersManager.GetPromotionData(promotionId, isClone, ref selectedPromotion);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors DeletePromotion(Promotion promotion, ref bool isValidToDelete, ref string validationMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string errorMsg = string.Empty;
             try
            {
                result = _pricesAndOffersManager.DeletePromotion(promotion, ref isValidToDelete, ref validationMessage);
                if (result == GlobalErrors.Success && promotion.SendNotificationToFO)
                {
                     SendNotification(promotion, _hostingEnvironment, ref errorMsg);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors UpdatePromotionActiveStatus(Promotion promotion, bool setInactive)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string errorMsg = string.Empty;
            try
            {
                result = _pricesAndOffersManager.UpdatePromotionActiveStatus(promotion.PromotionId, setInactive);
                if (result == GlobalErrors.Success && promotion.SendNotificationToFO)
                {
                    SendNotification(promotion, _hostingEnvironment, ref errorMsg);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetPromotionReasonList(ref List<PromotionReasonModel> promotionReasonList)
        {
            try
            {
                return _pricesAndOffersManager.GetPromotionReasonList(ref promotionReasonList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                promotionReasonList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors ApproveOperation(int Id, int currentStatusId, int operationTypeId, ref int actualNewLevel, DBHelper<int> dBHelper)
        {
            int newStatus = 0;
            return ApproveOperation( Id,  currentStatusId,  operationTypeId, ref actualNewLevel, ref newStatus , dBHelper);
        }
        public GlobalErrors ApproveOperation(int Id, int currentStatusId, int operationTypeId ,ref int actualNewLevel,ref int newStatus, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (dBHelper == null)
                {
                    dBHelper = new DBHelper<int>();
                }
                result = _pricesAndOffersManager.ApproveOperation(Id, operationTypeId, ref actualNewLevel, ref newStatus,string.Empty, dBHelper);
                if (result == GlobalErrors.Success)
                {
                    result = _pricesAndOffersManager.SaveInApprovalCycleHistory(Id, currentStatusId, operationTypeId, newStatus, dBHelper);
                    if (result == GlobalErrors.Success)
                    {
                        result = _pricesAndOffersManager.InsertPriceCycleHistory(Id, 7, _requestRepository.CurrentOperator.EmployeeId, currentStatusId, currentStatusId, newStatus, "Null", dBHelper);
                    }

                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
          
            return result;
        }
        public GlobalErrors SendApprovalEmail(int operationTypeId,int Id, ref int actualLevelID,string operationCode , bool isApprove, ref string errorMessage ,IHostingEnvironment _hostingEnvironment) {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<string> dBHelper = new DBHelper<string>();
            DBHelper<int> dBHelper2 = new DBHelper<int>();
            int EmailTemplateId = 0;
            List<string> recieversEmails = new List<string>();
            EmailModel newEmailModel = new EmailModel();
            string q = "";
            errorMessage = "";
            try {
                //send by email
                int targettedLevel = 0;
                if (isApprove)
                    targettedLevel = actualLevelID + 2;
                else
                    targettedLevel = actualLevelID + 1;
                if(operationTypeId == ApprovalCycleOperations.Price.GetHashCode() && _requestRepository.Configurations.ForcePreviousLevelApproval || operationTypeId == ApprovalCycleOperations.ContractedFOC.GetHashCode() && _requestRepository.Configurations.ForcePreviousLevelApproval)
                {
                    targettedLevel = targettedLevel - 1;
                    bool isMaxLevel = false;
                    SettingManager settingManager = new SettingManager(_requestRepository);
                    if(operationTypeId == ApprovalCycleOperations.Price.GetHashCode())
                    {
                        result = settingManager.IsMaxOperationLevel(ApprovalCycleOperations.Price.GetHashCode(), ref isMaxLevel, dBHelper2);
                    }
                    else
                    {
                        result = settingManager.IsMaxOperationLevel(ApprovalCycleOperations.ContractedFOC.GetHashCode(), ref isMaxLevel, dBHelper2);
                    }
                    if (isMaxLevel)
                    {
                        return GlobalErrors.ValueIsMoreThanRequested;
                    }


                }
                q = String.Format(@"select Email FROM Employee INNER JOIN EmployeeOperator ON Employee.EmployeeID = EmployeeOperator.EmployeeID
                        INNER JOIN OperatorSecurityGroup ON EmployeeOperator.OperatorID = OperatorSecurityGroup.OperatorID INNER JOIN
                        ApprovalCycleLevels ON ApprovalCycleLevels.SecurityGroupID = OperatorSecurityGroup.SecurityGroupID Where
                        ApprovalCycleLevels.LevelID = {0}
                        AND ApprovalCycleLevels.OperationID = {1} AND Email != ''
                        ", targettedLevel, operationTypeId);

                result = dBHelper.GetQueryList(q, ref recieversEmails);

                if (result == GlobalErrors.Success) {
                    q = String.Format("Select EmailTemplateID from EmailTemplate Where EmailTypeId = 11  and InActive != 1");
                    result = dBHelper2.GetQuerySingle(q, ref EmailTemplateId);
                    if (result == GlobalErrors.Success) {
                        if (EmailTemplateId <= 0)
                        {
                            errorMessage = "Desc_No_Email_Template_Found_For_Document_Type";
                            if(operationTypeId == ApprovalCycleOperations.Price.GetHashCode() || operationTypeId == ApprovalCycleOperations.ContractedFOC.GetHashCode())
                            {
                                return GlobalErrors.ValueIsMoreThanRequested;

                            }
                            return GlobalErrors.NoMoreRows;
                        }
                        if (recieversEmails == null || recieversEmails.Count <= 0)
                        {
                            errorMessage = "Desc_No_Email_Found";
                            return GlobalErrors.NoMoreRows;
                        }
                        foreach (string email in recieversEmails)
                        {
                            //MemoryStream fff = _templateGenerator.CreateProofsPDF(DocumentId, pdf, proofTypeData);
                            MemoryStream fff = new MemoryStream();
                            newEmailModel.ToEmail = email;
                            newEmailModel.EmailTempId = EmailTemplateId;
                            ICollection<IFormFile> file = null;

                            //file.Add(fff);
                            if (newEmailModel != null)
                            {
                                string rootPath = _hostingEnvironment.ContentRootPath;
                                string operationName = "";
                                switch ((ApprovalCycleOperations)operationTypeId)
                                {
                                    case ApprovalCycleOperations.Promotions:
                                        operationName = "promotions";
                                        break;
                                    case ApprovalCycleOperations.Discounts:
                                        operationName = "discounts";
                                        break;
                                    case ApprovalCycleOperations.CreditRelease:
                                        operationName = "credit Releases";
                                        break;
                                    case ApprovalCycleOperations.Price:
                                        operationName = "price";
                                        break;
                                    case ApprovalCycleOperations.ContractedFOC:
                                        operationName = "contracted Foc";
                                        break;
                                }
                                List<Attachment> attachments = new List<Attachment>();
                                string folderName = Path.Combine("Resources\\attachements\\");
                                string fullFolderName = Path.Combine(folderName, operationName, Id.ToString());
                                var userAttachementPath = Path.Combine(rootPath, fullFolderName);
                                DirectoryInfo Attachementdir = new DirectoryInfo(userAttachementPath);
                                if (Attachementdir.Exists)
                                {
                                    foreach (var file1 in Attachementdir.GetFiles())
                                    {
                                        //FileStream stream = new FileStream(userAttachementPath, FileAccess.Read);
                                        Attachment t = new Attachment(file1.FullName);
                                        attachments.Add(t);
                                    }
                                }

                                result = _emailService.SendEmail(file, newEmailModel, attachments , (ApprovalCycleOperations)operationTypeId ,operationCode,Id);
                            }

                            if (result != GlobalErrors.Success)
                            {
                                break;
                            }
                        }

                    }
                }
            }
            catch (Exception ex) {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors IsValidToApprove(int status , int OperationTypeId, int operationDetailId,ref bool isValid)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                List<int> approvalCycleValidations = new List<int>();
                int currentLevel = -1;
                result = _pricesAndOffersManager.IsValidToApprove(status, OperationTypeId, operationDetailId, string.Empty,ref isValid, ref approvalCycleValidations, ref currentLevel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors IsValidToAffect(int status, int OperationTypeId, ref bool isValid)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _pricesAndOffersManager.IsValidToAffect(status, OperationTypeId, ref isValid);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors IsValidToEditFoc(ContractedFOCModel contractedFOC, ref bool isValid,ref string editingMsg)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _pricesAndOffersManager.IsValidToEditFoc(contractedFOC, ref isValid, ref editingMsg);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetAchievementTypesForTargetPromotions(ref List<AchievementTypeModel> achievementList)
        {
            try
            {
                return _pricesAndOffersManager.GetAchievementTypesForTargetPromotions(ref achievementList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                achievementList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors IsValidToEditTargetPromotion(int promotionId, bool isRelatedToTarget, ref bool isValidToEdit, ref List<CustomerOutletModel> customerList)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                bool isPromoted = false;
                bool isPromotionTaken = false;
                if (!isRelatedToTarget)
                {
                    result = _pricesAndOffersManager.IsTargetPromotionPromoted(promotionId, ref isPromoted, null);
                    if (result == GlobalErrors.Success)
                    {
                        if (isPromoted)
                        {
                            isValidToEdit = false;
                            result = _pricesAndOffersManager.GetAchievementCustomer(promotionId, ref customerList, null);
                        }
                    }
                }
                else
                {
                    result = _pricesAndOffersManager.IsPromotionTakenForRelatedToTarget(promotionId,ref isPromotionTaken, null);
                    if (isPromotionTaken)
                    {
                        isValidToEdit = false;
                        result = _pricesAndOffersManager.GetAchievementCustomer(promotionId, ref customerList, null);
                    }
                }
        }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        #endregion
            
        public GlobalErrors GetQuantityRangeList(ref List<QuantityRange> quantityRanges)    
        {   
            try    
            {     
                return _pricesAndOffersManager.GetQuantityRangeList(ref quantityRanges);   
            }    
            catch (Exception ex)   
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            
        }

        public GlobalErrors InsertQuantityRange(QuantityRange quantityRange, ref int quantityRangeID)    
        {   
            try    
            {     
                return _pricesAndOffersManager.InsertQuantityRange(quantityRange,ref quantityRangeID);   
            }    
            catch (Exception ex)   
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            
        }
        #region [Promotion Groups]
        public GlobalErrors GetPromotionGroupsList(GeneralFilter filter, ref SharedTableResult<PromotionGroupModel> PromotionGroupListTable)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _pricesAndOffersManager.GetPromotionGroupsList(filter, ref PromotionGroupListTable);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                PromotionGroupListTable = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors InsertPromotionGroup(PromotionGroupModel PromotionGroup)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                PromotionGroup.PromotionGroupId = GetMaxID("PromotionGroupID", "PromotionGroup", null);
                result = _pricesAndOffersManager.InsertPromotionGroup(PromotionGroup);
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
        public GlobalErrors UpdatePromotionGroup(PromotionGroupModel PromotionGroup)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                result = _pricesAndOffersManager.UpdatePromotionGroup(PromotionGroup);
            }
            catch (Exception ex)
            {
                result = GlobalErrors.Error;
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return result;
        }
        public static int GetMaxID(string columnName, string tableName, DBHelper<int> dBHelper)
        {
            try
            {
                if (dBHelper == null)
                    dBHelper = new DBHelper<int>();
                object max = null;
                string query = string.Format(@"select isnull(max({0}),0)+1 from {1} ", columnName, tableName);
                GlobalErrors result = dBHelper.ExecuteScalar(query, ref max);
                if (result == GlobalErrors.Success && max != null && !string.IsNullOrEmpty(max.ToString()))
                {
                    return int.Parse(max.ToString());
                }

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return -1;
            }
            return 1;
        }
        public GlobalErrors GetPromotionGroupData(int PromotionGroupId, ref PromotionGroupModel promotionGroup)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                var descriptions = new List<TextLanguage>();
                var promotions = new List<PromotionGroupModel>();

                result = _pricesAndOffersManager.GetAssignedPromotions(PromotionGroupId, ref promotions);
                if (result == GlobalErrors.Success)
                    result = _pricesAndOffersManager.GetPromotionGroupDescriptions(PromotionGroupId, ref descriptions);
                if (result == GlobalErrors.Success)
                {
                    promotionGroup.languageDescriptions = descriptions;
                    promotionGroup.PromotionIdsList = promotions;
                    promotionGroup.PromotionGroupId = PromotionGroupId;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetUnAssignedPromotions(PromotionFilterModel filter, ref SharedTableResult<PromotionGroupModel> promotionsSharedList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _pricesAndOffersManager.GetUnAssignedPromotions(filter, ref promotionsSharedList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                promotionsSharedList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors DeletePromotionGroupData(int promotionGroupId, ref string errorMessage)
        {
            try
            {
                return _pricesAndOffersManager.DeletePromotionGroup(promotionGroupId, ref errorMessage);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors DeletePromotionAssignment(int promotionId, ref string errorMessage)
        {
            try
            {
                return _pricesAndOffersManager.DeletePromotionAssignment(promotionId, ref errorMessage);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        #endregion

        #region [Contracted Foc]
        public GlobalErrors GetContractedFocList(ContractedFOCFilterModel filter, ref SharedTableResult<ContractedFOCModel> custList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _pricesAndOffersManager.GetContractedFocList(filter, ref custList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractTypes(ref List<ContractTypeModel> contractTypesObj)
        {
            return _pricesAndOffersManager.GetContractTypes(ref contractTypesObj);
        }
        public GlobalErrors SaveContractedFOCList(ContractedFOCModel contractedFOC)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dBHelper = new DBHelper<int>();
            dBHelper.BeginTransaction();
            try
            {
                if (_requestRepository.Configurations.ApplyApprovalCycleInWeb.Contains("5"))
                {
                    int employeeStatus = _pricesAndOffersManager.GetCurrentStatus();
                    if (contractedFOC.ContractId != -1)
                    {
                        int contractListStatus = 0;
                        bool isDeleted = false;
                        result = _pricesAndOffersManager.GetContractedFocListStatus(contractedFOC.ContractId, ref contractListStatus, ref isDeleted);

                        if (_requestRepository.Configurations.ForcePreviousLevelApproval)
                        {
                            if (employeeStatus == -1)//max level
                            {
                                int maxLevel = _pricesAndOffersManager.GetMaxApprovalCycleLevels(ApprovalCycleOperations.ContractedFOC.GetHashCode());
                                if (contractListStatus != maxLevel && contractListStatus != -1)//priceListStatus != -1 for Active Price
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                            }
                            else
                            {
                                if ((isDeleted && employeeStatus != contractListStatus) || employeeStatus != contractListStatus && contractListStatus != -1)//priceListStatus != -1 for Active Price // priceListStatus != 0 found on ABC_20200207 DB
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                                else
                                if (employeeStatus != contractedFOC.Status && contractListStatus != -1)//priceListStatus != -1 for Active Price // priceListStatus != 0 found on ABC_20200207 DB
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                            }

                        }
                    }


                    SettingManager settingManager = new SettingManager(_requestRepository);

                    if (contractedFOC.ContractId <= 0)
                    {
                        if (employeeStatus == -1)//check tax type delete
                        {
                            result = _pricesAndOffersManager.InsertContractedFOC(contractedFOC, dBHelper);
                            int level = contractedFOC.Status;
                            if (employeeStatus != contractedFOC.Status)
                                level = employeeStatus;
                            result = _pricesAndOffersManager.InsertContractedFOCCycleHistory(contractedFOC.ContractId, 1, _requestRepository.CurrentOperator.EmployeeId, level, contractedFOC.Status, employeeStatus, contractedFOC.RejectNote, dBHelper);
                        }
                        else
                        {
                            result = _pricesAndOffersManager.InsertContractedFOCPending(contractedFOC, dBHelper);
                            if (result == GlobalErrors.Success)
                            {
                                int level = contractedFOC.Status;
                                if (employeeStatus != contractedFOC.Status)
                                    level = employeeStatus;
                                result = _pricesAndOffersManager.InsertContractedFOCCycleHistory(contractedFOC.ContractId, 1, _requestRepository.CurrentOperator.EmployeeId, level, contractedFOC.Status, employeeStatus, contractedFOC.RejectNote, dBHelper);
                            }
                        }
                    }

                    else
                    {
                        if (contractedFOC.Status != 0 && contractedFOC.Status != -1)// priceListObj.Status != -1 for Active Price
                        {

                            result = _pricesAndOffersManager.UpdateContractedFOCPending(contractedFOC, dBHelper);
                            if (!contractedFOC.IsCreationLevel)
                            {
                                int level = -1;
                                bool isValid = false;
                                result = IsValidToApprove(contractedFOC.Status, ApprovalCycleOperations.ContractedFOC.GetHashCode(), -1,ref isValid);
                                if (isValid)
                                {

                                    result = ApproveContractOperation(contractedFOC.ContractId, contractedFOC.Status, ApprovalCycleOperations.ContractedFOC.GetHashCode(), ref level, dBHelper);
                                    if (result != GlobalErrors.Success) return result;


                                }

                            }
                            if (contractedFOC.IsDeleted && employeeStatus == -1)//for max employee level 
                            {
                                string Msg = string.Empty;
                                result = _pricesAndOffersManager.DeleteContractedFocPending(contractedFOC, dBHelper);
                                if (result == GlobalErrors.Success)
                                {
                                    result = _pricesAndOffersManager.DeleteContractedFOC(contractedFOC, ref Msg, dBHelper);
                                }
                            }

                        }
                        else
                        {
                            bool isMaxLevel = false;
                            result = settingManager.IsMaxOperationLevel(ApprovalCycleOperations.ContractedFOC.GetHashCode(), ref isMaxLevel, dBHelper);
                            if (isMaxLevel) // note #1
                            {
                                result = _pricesAndOffersManager.UpdateContractedFoc(contractedFOC, dBHelper);
                            }
                            else
                            {
                                contractedFOC.Status = _pricesAndOffersManager.GetCurrentStatus();
                                result = _pricesAndOffersManager.InsertContractedFOCPending(contractedFOC, dBHelper);
                                if (result == GlobalErrors.Success)
                                {
                                    //result = _pricesAndOffersManager.UpdatePriceListPending(priceListObj, dBHelper);
                                    int level = -1;
                                    result = _pricesAndOffersManager.InsertContractedFOCCycleHistory(contractedFOC.ContractId, 2, _requestRepository.CurrentOperator.EmployeeId, level, contractedFOC.Status, employeeStatus, contractedFOC.RejectNote, dBHelper);
                                }
                            }

                        }
                    }

                }
                else
                {
                    if (contractedFOC.ContractId <= 0)
                    {
                        result = _pricesAndOffersManager.InsertContractedFOC(contractedFOC, dBHelper);
                    }
                    else
                    {
                        result = _pricesAndOffersManager.UpdateContractedFoc(contractedFOC, dBHelper);
                    }
                }

                if (result == GlobalErrors.SuccessWithZeroRowAffected) result = GlobalErrors.Success;

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dBHelper.CommitTransaction();
                }
                else
                {
                    dBHelper.RollBackTransaction();
                }
            }
            return result;
        }
        public bool CheckIsContractedFocPendingExist(int contractId)
        {
            try
            {
                return _pricesAndOffersManager.CheckIsContractedFocPendingExist(contractId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return false;
            }
        }
        public GlobalErrors GetContractedFocHeaderPending(int contractId, ref ContractedFOCModel contractedFoc)
        {
            try
            {
                return _pricesAndOffersManager.GetContractedFocHeaderPending(contractId, ref contractedFoc);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                contractedFoc = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocHeader(int contractID, ref ContractedFOCModel contractedFoc)
        {
            try
            {

                return _pricesAndOffersManager.GetContractedFocHeader(contractID, ref contractedFoc);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                contractedFoc = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocItems(ContractedFOCModel contractedFocObject, ref List<ContractedFOCModel> contractedFoc)
        {
            try
            {

                return _pricesAndOffersManager.GetContractedFocItems(contractedFocObject, ref contractedFoc);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                contractedFoc = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocAppliedCustomers(ContractedFOCModel contractedFocObject, ref List<SimpleCustomerModel> appliedCustomers)
        {
            try
            {

                return _pricesAndOffersManager.GetContractedFocAppliedCustomers(contractedFocObject, ref appliedCustomers);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedCustomers = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocAppliedCustomersPending(ContractedFOCModel contractedFOC, ref List<SimpleCustomerModel> appliedCustomers)
        {
            try
            {

                return _pricesAndOffersManager.GetContractedFocAppliedCustomersPending(contractedFOC, ref appliedCustomers, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedCustomers = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocItemsPending(ContractedFOCModel contractedFocObject, ref List<ContractedFOCModel> contractedListItems)
        {
            try
            {

                return _pricesAndOffersManager.GetContractedFocItemsPending(contractedFocObject, ref contractedListItems, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                contractedListItems = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocAmountPending(ContractedFOCModel contractedFocObject, ref ContractedFOCModel contractedFoc)
        {
            try
            {

                return _pricesAndOffersManager.GetContractedFocAmountPending(contractedFocObject, ref contractedFoc, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                contractedFoc = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocDescriptionsPending(int contractId, ref List<TextLanguage> descriptions)
        {
            try
            {
                return _pricesAndOffersManager.GetContractedFocDescriptionsPending(contractId, ref descriptions);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                descriptions = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocDescriptions(int contractId, ref List<TextLanguage> descriptions)
        {
            try
            {

                return _pricesAndOffersManager.GetContractedFocDescriptions(contractId, ref descriptions);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                descriptions = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetContractedFocAmount(ContractedFOCModel contractedFOC, ref ContractedFOCModel contractedFoc)
        {
            try
            {

                return _pricesAndOffersManager.GetContractedFocAmount(contractedFOC, ref contractedFoc);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                contractedFoc = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors DeleteContractedFoc(ContractedFOCModel contractedFOC,ref string deletingMsg, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                dBHelper = new DBHelper<int>();
                dBHelper.BeginTransaction();

                if (_requestRepository.Configurations.ApplyApprovalCycleInWeb.Contains("5"))
                {
                    int currentContractStatus = 0;
                    bool isDeleted = false;
                    int originalStatus = _pricesAndOffersManager.GetCurrentStatus();
                    result = _pricesAndOffersManager.GetContractedFocListStatus(contractedFOC.ContractId, ref currentContractStatus, ref isDeleted);
                    if (_requestRepository.Configurations.ForcePreviousLevelApproval)
                    {
                        if (currentContractStatus != 0)// delete from orginal 
                        {
                            if (isDeleted)
                            {
                                if (originalStatus == -1)// max level 
                                {
                                    int maxLevel = _pricesAndOffersManager.GetMaxApprovalCycleLevels(ApprovalCycleOperations.ContractedFOC.GetHashCode());
                                    if (currentContractStatus != -1 && currentContractStatus != maxLevel)
                                    {
                                        result = GlobalErrors.ValueIsMoreThanRequested;
                                        return result;
                                    }
                                }
                                else
                                {
                                    if (originalStatus != contractedFOC.Status)
                                    {
                                        result = GlobalErrors.ValueIsMoreThanRequested;
                                        return result;
                                    }
                                    if (originalStatus != currentContractStatus)
                                    {
                                        result = GlobalErrors.ValueIsMoreThanRequested;
                                        return result;
                                    }
                                }
                            }
                            else
                            {
                                if (originalStatus == -1)
                                {
                                    originalStatus = _pricesAndOffersManager.GetMaxApprovalCycleLevels(ApprovalCycleOperations.Price.GetHashCode());
                                }
                                if (originalStatus != currentContractStatus && currentContractStatus != -1)//when delete pending and get active // and isDeleted
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                                if (originalStatus != contractedFOC.Status && contractedFOC.Status != -1)
                                {
                                    result = GlobalErrors.ValueIsMoreThanRequested;
                                    return result;
                                }
                            }
                        }
                        else
                        {
                            result = GlobalErrors.ValueIsMoreThanRequested;
                            return result;
                        }
                    }
                    else
                    {
                        if (currentContractStatus == 0)
                        {
                            result = GlobalErrors.ValueIsMoreThanRequested;
                            return result;
                        }
                    }

                    PriceListModel priceListModel = new PriceListModel();
                    PriceListModel priceListModelPending = new PriceListModel();

                    if (_pricesAndOffersManager.CheckIsContractedFocPendingExist(contractedFOC.ContractId))
                    {
                        result = _pricesAndOffersManager.DeleteContractedFocPending(contractedFOC, dBHelper);
                        if (result == GlobalErrors.Success)
                        {
                            result = _pricesAndOffersManager.InsertContractedFOCCycleHistory(contractedFOC.ContractId, 3, _requestRepository.CurrentOperator.EmployeeId, contractedFOC.Status, contractedFOC.Status, -1, contractedFOC.RejectNote, dBHelper);
                        }
                    }
                    else
                    {
                        result = _pricesAndOffersManager.DeleteActiveContractedFoc(contractedFOC, dBHelper);
                        if (result == GlobalErrors.Success)
                        {
                            result = _pricesAndOffersManager.InsertContractedFOCCycleHistory(contractedFOC.ContractId, 3, _requestRepository.CurrentOperator.EmployeeId, contractedFOC.Status, contractedFOC.Status, originalStatus, contractedFOC.RejectNote, dBHelper);

                            //result = _pricesAndOffersManager.DeletePriceList(price.PriceListId);
                        }
                    }
                    return result;
                }
                else
                {
                    result = _pricesAndOffersManager.DeleteContractedFOC(contractedFOC, ref deletingMsg, dBHelper);
                    if (result == GlobalErrors.Success)
                    {
                        result = _pricesAndOffersManager.DeleteContractedFocPending(contractedFOC, dBHelper);
                    }
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dBHelper.CommitTransaction();
                }
                else
                {
                    dBHelper.RollBackTransaction();
                }
            }
        }
        public GlobalErrors GetContractedFocListStatus(int contractId, ref int status, ref bool isDeleted)
        {
            try
            {
                return _pricesAndOffersManager.GetContractedFocListStatus(contractId, ref status, ref isDeleted);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error; ;
            }
        }
        public GlobalErrors ApproveContractOperation(int Id, int currentStatusId, int operationTypeId, ref int actualNewLevel, DBHelper<int> dBHelper)
        {
            int newStatus = 0;
            return ApproveContractOperation(Id, currentStatusId, operationTypeId, ref actualNewLevel, ref newStatus, dBHelper);
        }
        public GlobalErrors ApproveContractOperation(int Id, int currentStatusId, int operationTypeId, ref int actualNewLevel, ref int newStatus, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                if (dBHelper == null)
                {
                    dBHelper = new DBHelper<int>();
                }
                result = _pricesAndOffersManager.ApproveOperation(Id, operationTypeId, ref actualNewLevel, ref newStatus, string.Empty, dBHelper);
                if (result == GlobalErrors.Success)
                {
                    result = _pricesAndOffersManager.SaveInApprovalCycleHistory(Id, currentStatusId, operationTypeId, newStatus, dBHelper);
                    if (result == GlobalErrors.Success)
                    {
                        result = _pricesAndOffersManager.InsertContractedFOCCycleHistory(Id, 7, _requestRepository.CurrentOperator.EmployeeId, currentStatusId, currentStatusId, newStatus, "Null", dBHelper);
                    }

                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }

            return result;
        }

        #endregion

        #region Header Discount

        public GlobalErrors GetHeaderDiscountAppliedCustomers(int headerdiscountId, ref List<SimpleCustomerModel> appliedCustomers)
        {
            try
            {
                return _pricesAndOffersManager.GetHeaderDiscountAppliedCustomers(headerdiscountId, ref appliedCustomers);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedCustomers = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetHeaderDiscountAppliedGroups(int headerdiscountId, ref List<CustomerGroupModel> appliedGroups)
        {
            try
            {
                return _pricesAndOffersManager.GetHeaderDiscountAppliedGroups(headerdiscountId, ref appliedGroups);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedGroups = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetHeaderDiscountAppliedChannels(int headerdiscountId, ref List<CustomerChannelModel> appliedChannels)
        {
            try
            {
                return _pricesAndOffersManager.GetHeaderDiscountAppliedChannels(headerdiscountId, ref appliedChannels);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                appliedChannels = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors DeleteHeaderDiscount(int HeaderdiscountId)
        {
            try
            {
                return _pricesAndOffersManager.DeleteHeaderDiscount(HeaderdiscountId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetHeaderDiscounts(GeneralFilter filter, ref SharedTableResult<HeaderDiscountModel> Headerdiscounts)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _pricesAndOffersManager.GetHeaderDiscounts(filter, ref Headerdiscounts);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                Headerdiscounts = null;
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors SaveHeaderDiscount(HeaderDiscountModel headerDiscountobj)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DAL.DBHelper<int> dBHelper = new DAL.DBHelper<int>();
            dBHelper.BeginTransaction();
            try
            {
                if (headerDiscountobj != null && headerDiscountobj.DateModelRange != null && headerDiscountobj.DateModelRange.Count == 2)
                {
                    headerDiscountobj.StartDate = new DateTime(headerDiscountobj.DateModelRange[0].Year, headerDiscountobj.DateModelRange[0].Month, headerDiscountobj.DateModelRange[0].Day);
                    headerDiscountobj.EndDate = new DateTime(headerDiscountobj.DateModelRange[1].Year, headerDiscountobj.DateModelRange[1].Month, headerDiscountobj.DateModelRange[1].Day);
                }

                if (headerDiscountobj.HeaderDiscountId <= 0)
                {
                    result = _pricesAndOffersManager.InsertHeaderDiscount(headerDiscountobj, dBHelper);
                }
                else
                {
                    result = _pricesAndOffersManager.UpdateHeaderDiscount(headerDiscountobj, dBHelper);
                    if (result == GlobalErrors.Success)
                    {
                        result = _pricesAndOffersManager.DeleteCustomerHeaderDiscount(headerDiscountobj, dBHelper);
                    }
                }
                if (result == GlobalErrors.Success)
                {
                    //Save applied customer
                    result = _pricesAndOffersManager.InsertCustomerHeaderDiscount(headerDiscountobj, dBHelper);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (result == GlobalErrors.Success)
                {
                    dBHelper.CommitTransaction();
                }
                else
                {
                    dBHelper.RollBackTransaction();
                }
            }
            return result;
        }
        public GlobalErrors GetSalesMode(ref List<SalesModeModel> salesModes)
        {
            try
            {

                return _pricesAndOffersManager.GetSalesMode(ref salesModes);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                salesModes = null;
                return GlobalErrors.Error;
            }
        }
        #endregion

    }
}
