﻿using Sonic.Web.Core;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Models;
using Sonic.Web.Resources;
using Sonic.Web.SecureLibrary;
using Sonic.Web.Services;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Reflection;

namespace Sonic.Web.Service
{
    public class TransactionService
    {
        private readonly IRequestRepository _requestRepository;
        private readonly OrderManager _orderManager;
        private readonly TransactionManager _transactionManager;
        private readonly DocumentSequenceService _documentSequenceService;
        private readonly DocumentSequenceManager _documentSequenceManager;
        private readonly EmployeeManager _employeeManager;
        private readonly CurrencyManager _currencyManager;
        private readonly CustomerManager _customerManager;
        public TransactionService(IRequestRepository requestRepository)
        {
            _requestRepository = requestRepository;
            _orderManager = new OrderManager(_requestRepository);
            _transactionManager = new TransactionManager(_requestRepository);
            _documentSequenceService = new DocumentSequenceService(requestRepository);
            _documentSequenceManager = new DocumentSequenceManager(requestRepository);
            _employeeManager = new EmployeeManager(requestRepository);
            _currencyManager = new CurrencyManager(requestRepository);
            _customerManager = new CustomerManager(requestRepository);
        }
        public GlobalErrors GetTransactions(GeneralFilter filter, ref SharedTableResult<TransactionModel> transList, bool getUnpaidTransactions)
        {
            return GetTransactions(filter, ref transList, getUnpaidTransactions, TransactionType.Sales,false);
        }
        public GlobalErrors GetTransactions(GeneralFilter filter, ref SharedTableResult<TransactionModel> custList, bool getUnpaidTransactions, TransactionType transactionType,bool isTestTransaction)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _transactionManager.GetTransactions(filter, ref custList, getUnpaidTransactions, transactionType, isTestTransaction);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetICashTransactions(GeneralFilter filter, ref SharedTableResult<TransactionModel> custList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _transactionManager.GetICashTransactions(filter, ref custList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetCustomerAvailableCreditNotes(int customerId, int outletId, int firstoutletId, int divisionID, ref List<TransactionModel> creditNote)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string customerTypeConfiguration = string.Empty;
            try
            {
                var customerTypeId = (int)_customerManager.GetCustomerType(customerId, firstoutletId);
                if (customerTypeId == (int)CustomerTypes.CashCustomer)
                {
                    customerTypeConfiguration = _requestRepository.Configurations.PaymentTypesForCashCustomer;
                }
                else if (customerTypeId == (int)CustomerTypes.CreditCustomer || customerTypeId == (int)CustomerTypes.BillToBillCustomer)
                {
                    customerTypeConfiguration = _requestRepository.Configurations.PaymentTypesForCreditCustomer;
                }
                result = GlobalErrors.Success;
                if (customerTypeConfiguration.Contains(PaymentTypes.CreditNote.GetHashCode().ToString()))
                {
                    return _transactionManager.GetCustomerAvailableCreditNotes(customerId, outletId, divisionID, ref creditNote);
                }                                              
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetDocumentReasonDataSource(int documentTypeId, ref List<TransactionDocumentReasonModel> documentReason)
        {
            try
            {
                return _transactionManager.GetDocumentReasonDataSource(documentTypeId, ref documentReason);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetAllTaxesForGeneralPrice( ref List<TaxModel> taxes, int employeeId)
        {
          
            try
            {
                return _transactionManager.GetAllTaxesForGeneralPrice(ref taxes , employeeId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        
        public GlobalErrors CheckIfCustomreRouteUploaded(int customerId, int outletId, ref bool isRouteUploaded)
        {
            try
            {
                return _transactionManager.CheckIfCustomreRouteUploaded(customerId, outletId, ref isRouteUploaded);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetTransactionData(ref TransactionModel transactionModel, bool isTestTransaction)
        {
            try
            {
                return _transactionManager.GetTransactionData(ref transactionModel, isTestTransaction);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        private bool AutoCollectBOInvoices(int customerTypeId, int salesMode)
        {
            return (_requestRepository.Configurations.AutoCollectBOInvoices && !_transactionManager.AllowSalesModeSelectionForCustomerInSales(customerTypeId)) ||
                (_transactionManager.AllowSalesModeSelectionForCustomerInSales(customerTypeId) && salesMode == CustomerTypes.CashCustomer.GetHashCode());
                
        }
        public GlobalErrors SaveReturnTransaction(TransactionModel transaction, DBHelper<int> dBHelper,bool isExchange, decimal salesNetTotal, ref string errorMessage)
        {
            var result = GlobalErrors.NotInitialized;
            bool commitNow = false;
            try
            {
                if (dBHelper == null)
                {
                    dBHelper = new DBHelper<int>();
                    dBHelper.BeginTransaction();
                    commitNow = true;
                }
                #region [Fill employee related data]

                var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;
                if (!isExchange)
                {
                    int helperId = -1, driverId = -1, supervisorId = -1, salesRepID = -1, SalesManagerId = -1, currency = -1;
                    result = _employeeManager.GetEmployeeRelatedStaff(transaction.EmployeeId, dBHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepID, ref SalesManagerId);

                    if (result == GlobalErrors.Success)
                    {
                        transaction.HelperId = helperId;
                        transaction.DriverId = driverId;
                        transaction.SupervisorId = supervisorId;
                        transaction.SalesRepId = salesRepID;
                        transaction.SalesManagerId = SalesManagerId;
                    }
                    transaction.OrganizationId = _employeeManager.GetEmployeeOrganization(transaction.EmployeeId, dBHelper);
                    if (string.IsNullOrEmpty(transaction.TransactionId))
                    {
                        return GlobalErrors.Error;
                    }
                }
                #endregion

                if (_requestRepository.Configurations.AllowSellReturns) // if this configuration false then no stock reflection
                {
                    if (GiveCreditOnReturn((CustomerTypes)transaction.Outlet.CustomerTypeId))
                    {
                        transaction.TransactionStatusId = TransactionStatues.Done.GetHashCode();
                    }
                    else
                        transaction.TransactionStatusId = TransactionStatues.Processed.GetHashCode();
                }
                else
                {
                    if (GiveCreditOnReturn((CustomerTypes)transaction.Outlet.CustomerTypeId))
                    {
                        transaction.TransactionStatusId = TransactionStatues.Approved.GetHashCode();
                    }
                    else
                        transaction.TransactionStatusId = TransactionStatues.None.GetHashCode();
                }


                if (!isExchange)
                {
                    transaction.TransactionType = TransactionType.Return;
                    transaction.Posted = true;
                    transaction.NetTotal = LocalUtilities.GetRoundedDecimal(transaction.NetTotal, numberOfDigits);
                    transaction.GrossTotal = LocalUtilities.GetRoundedDecimal(transaction.GrossTotal, numberOfDigits);
                    //transaction.RemainingAmount = LocalUtilities.GetRoundedDecimal(transaction.RemainingAmount, numberOfDigits);
                    transaction.RemainingAmount = 0;
                    transaction.Discount = LocalUtilities.GetRoundedDecimal(transaction.Discount, numberOfDigits);
                    transaction.Tax = LocalUtilities.GetRoundedDecimal(transaction.Tax, numberOfDigits);
                    transaction.CalculatedRetailTax = LocalUtilities.GetRoundedDecimal(transaction.CalculatedRetailTax, numberOfDigits);
                    transaction.PromotedDiscount = LocalUtilities.GetRoundedDecimal(transaction.PromotedDiscount, numberOfDigits);
                    transaction.TransactionDate = new DateTime(transaction.TransactionDateModel.Year, transaction.TransactionDateModel.Month, transaction.TransactionDateModel.Day, DateTime.Now.Hour, DateTime.Now.Minute, DateTime.Now.Second);
                    transaction.CreatedDate = DateTime.Now;
                    transaction.UpdatedDate = transaction.CreatedDate;
                    transaction.UpdatedBy = transaction.CreatedBy;
                    transaction.Voided = false;
                    transaction.SalesMode = SalesModes.None;
                    transaction.CreationReason = TransactionCreationReason.Normal;
                    transaction.AccountId = transaction.Outlet.Account.AccountId;
                    transaction.CurrencyId = transaction.Outlet.Account.CurrencyId;
                    transaction.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
                    //transaction.DueDate = transaction.CreatedDate;
                    transaction.FinalDiscount = transaction.FinalDiscount;
                    transaction.Tempreture = transaction.Tempreture;
                }
                result = _transactionManager.SaveTransactionHeader(transaction, dBHelper);
                if (result == GlobalErrors.Success)
                {
                    result = _transactionManager.GetDefaultPriceList(transaction, dBHelper);
                }
                //    result = _transactionManager.SaveTransactionDetails(transaction, true, dBHelper);
                if (result == GlobalErrors.Success)
                {
                    int packSequence = 0;
                    AppliedBatch appliedBatch = new AppliedBatch();
                    List<ItemPackModel> packBatchList = new List<ItemPackModel>();
                    ItemPackModel packBatchItem = null;
                    if (isExchange)
                        transaction.AllItems = transaction.AllItems.Where(a => a.SalesTransactionTypeId == SalesTransactionTypes.None).ToList();
                    foreach (var pack in transaction.AllItems)
                    {
                        if (pack.ReturnDetailsBatches != null && pack.ReturnDetailsBatches.Count > 0)
                        {
                            foreach(var batch in pack.ReturnDetailsBatches)
                            {
                                packSequence++;
                                packBatchItem = new ItemPackModel(pack);
                                packBatchItem.RequiredQty = batch.RequiredQty;
                                packBatchItem.Sequence = packSequence;
                                appliedBatch.BatchNumber = batch.BatchNo;
                                appliedBatch.ExpiryDate = new DateTime(batch.ExpiryDateModel.Year, batch.ExpiryDateModel.Month, batch.ExpiryDateModel.Day);
                                appliedBatch.SalesTransactionTypeId = batch.SalesTransactionTypeId;
                                result = _transactionManager.SaveTransactionDetailsOnBatchLevel(transaction, packBatchItem, dBHelper, appliedBatch);
                                if (result != GlobalErrors.Success)
                                {
                                    return GlobalErrors.Error;
                                }
                            }
                        }
                        else
                        {
                            packSequence++;
                            pack.Sequence = packSequence;
                            if (transaction.IsReturn)
                            {
                                if (pack.ExpiryDateModel != null)
                                    pack.ExpiryDate = new DateTime(pack.ExpiryDateModel.Year, pack.ExpiryDateModel.Month, pack.ExpiryDateModel.Day);
                                else
                                    pack.ExpiryDate = LocalUtilities.DefaultExpiryDate;
                                appliedBatch.ExpiryDate = pack.ExpiryDate;
                            }
                            else
                            {
                                appliedBatch.ExpiryDate = pack.ExpiryDate;
                            }
                            appliedBatch.BatchNumber = pack.BatchNo;
                            appliedBatch.PackId = pack.PackId;
                            appliedBatch.QuantityChange = pack.RequiredQty;
                            appliedBatch.Price = pack.Price;
                            appliedBatch.Tax = pack.CalculatedTax;
                            result = _transactionManager.SaveTransactionDetailsOnBatchLevel(transaction, pack, dBHelper, appliedBatch);
                            if (result != GlobalErrors.Success)
                            {
                                return GlobalErrors.Error;
                            }
                        }
                    }
                }
                #region[Check for resellable pack statuses]
                List<int> selectedstatuses = new List<int>();
                if (result == GlobalErrors.Success)
                {
                    result = GetSelectedPackStatues(transaction.AllItems, ref selectedstatuses, dBHelper); // will retrive sellable only or all statuses based on configuration
                }
                #endregion


                if (result == GlobalErrors.Success && _requestRepository.Configurations.AllowGivingCreditNoteOnReturn && GiveCreditOnReturn((CustomerTypes)transaction.Outlet.CustomerTypeId))
                {
                    bool givePartialCredit = GivePartialCredit((CustomerTypes)transaction.Outlet.CustomerTypeId);
                    bool needToCreateCN = false;
                    decimal CNValue = 0;

                    if (!isExchange)
                    {
                        needToCreateCN = !givePartialCredit;
                        CNValue = transaction.NetTotal;
                    }
                    else
                    {
                        needToCreateCN = true; // exchange will create credit note regadless giveCredit is true or false
                        if (givePartialCredit)
                        {
                            CNValue = Math.Min(salesNetTotal, transaction.NetTotal); //find the minimub between sales and return
                        }
                        else
                        {
                            CNValue = transaction.NetTotal;
                        }
                    }
                    if (needToCreateCN && (transaction.TransactionStatusId == TransactionStatues.Approved.GetHashCode() || transaction.TransactionStatusId == TransactionStatues.Done.GetHashCode()) && transaction.NetTotal > 0)
                    {
                        var maxCreditNoteTransID = _documentSequenceService.GetMaxDocumentSequence(transaction.EmployeeId, DocumentTypes.CreditNote, transaction.DivisionId, dBHelper);
                        if (!string.IsNullOrEmpty(maxCreditNoteTransID))
                        {
                            string notes = "Generated Credit Note " + transaction.Notes;
                            result = _customerManager.CreateCreditNote(transaction.CustomerId.ToString(), transaction.OutletId.ToString(), transaction.DivisionId, transaction.TransactionId, CNValue, maxCreditNoteTransID, null, notes, transaction.AccountId, dBHelper);
                        }
                        else
                        {
                            errorMessage = MessagesConstants.Desc_No_Document_Sequence_Credit_Note;
                            return GlobalErrors.Error;
                        }
                    }

                }
                if (result == GlobalErrors.Success)
                    result = _documentSequenceManager.UpdateMaxTransactionID(dBHelper, DocumentTypes.Return, transaction.TransactionId, _requestRepository.CurrentOperator.EmployeeId, transaction.DivisionId);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            finally
            {
                if (dBHelper != null && commitNow)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dBHelper.CommitTransaction();
                    }
                    else
                    {
                        dBHelper.RollBackTransaction();
                    }
                    dBHelper.Dispose();
                    dBHelper = null;
                }
            }
            return result;
        }
        //public GlobalErrors SaveExchangeTransaction(TransactionModel transaction, DBHelper<int> dBHelper, ref string errorMessage)
        //{
        //    var result = GlobalErrors.NotInitialized;
        //    bool commitNow = false;
        //    string customertype = string.Empty;
        //    string NoEnoughtQtyFromItemString = string.Empty;
        //    try
        //    {
        //        if (dBHelper == null)
        //        {
        //            dBHelper = new DBHelper<int>();
        //            dBHelper.BeginTransaction();
        //            commitNow = true;
        //        }
        //        #region [Fill employee related data]

        //        int helperId = -1, driverId = -1, supervisorId = -1, salesRepID = -1, SalesManagerId = -1, currency = -1;
        //        result = _employeeManager.GetEmployeeRelatedStaff(transaction.EmployeeId, dBHelper, ref helperId, ref driverId, ref supervisorId, ref salesRepID, ref SalesManagerId);

        //        if (result == GlobalErrors.Success)
        //        {
        //            transaction.HelperId = helperId;
        //            transaction.DriverId = driverId;
        //            transaction.SupervisorId = supervisorId;
        //            transaction.SalesRepId = salesRepID;
        //            transaction.SalesManagerId = SalesManagerId;
        //        }
        //        transaction.OrganizationId = _employeeManager.GetEmployeeOrganization(transaction.EmployeeId, dBHelper);
        //        currency = _currencyManager.GetCustomerOuteltCurrency(transaction.CustomerId, transaction.OutletId, dBHelper);
        //        if (currency == 0) currency = -1;
        //        #endregion
        //        var SalesItem = new List<ItemPackModel>();
        //        var ReturnsItem = new List<ItemPackModel>();
        //        TransactionModel SalesTransaction = new TransactionModel();
        //        var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;

        //        SalesItem = transaction.AllItems.Where(a => a.SalesTransactionTypeId == SalesTransactionTypes.Sales || a.SalesTransactionTypeId == SalesTransactionTypes.Promotion).ToList();
        //        ReturnsItem = transaction.AllItems.Where(a => a.SalesTransactionTypeId == SalesTransactionTypes.None).ToList();
        //        if (SalesItem != null && SalesItem.Count > 0)
        //        {
        //            SalesTransaction.SoldItems = SalesItem;
        //            SalesTransaction.PromotedItems = transaction.PromotedItems;
        //            transaction.TransactionType = TransactionType.Sales;
        //            SalesTransaction.TransactionId = transaction.TransactionId;
        //            SalesTransaction.TransactionStatusId = TransactionStatues.None.GetHashCode();
        //            SalesTransaction.TransactionDate = transaction.TransactionDate;
        //            SalesTransaction.CustomerId = transaction.CustomerId;
        //            SalesTransaction.OutletId = transaction.OutletId;
        //            SalesTransaction.EmployeeId = transaction.EmployeeId;
        //            SalesTransaction.WarehouseId = transaction.WarehouseId;
        //            SalesTransaction.DivisionId = transaction.DivisionId;
        //            SalesTransaction.TransactionDate = new DateTime(transaction.TransactionDateModel.Year, transaction.TransactionDateModel.Month, transaction.TransactionDateModel.Day);
        //            SalesTransaction.TransactionDate = SalesTransaction.TransactionDate;
        //            foreach (ItemPackModel salesInvoice in SalesItem)
        //            {
        //                SalesTransaction.GrossTotal += decimal.Round(salesInvoice.GrossTotal, numberOfDigits);
        //                SalesTransaction.NetTotal += decimal.Round(salesInvoice.NetTotal, numberOfDigits);
        //                SalesTransaction.Tax += decimal.Round(salesInvoice.Tax, numberOfDigits);
        //                SalesTransaction.RemainingAmount += decimal.Round(salesInvoice.NetTotal, numberOfDigits);
        //                SalesTransaction.Discount += decimal.Round(salesInvoice.Discount, numberOfDigits);
        //                salesInvoice.ExpiryDate = LocalUtilities.DefaultExpiryDate;
        //                salesInvoice.BatchNo = LocalUtilities.DefaultBatchNo;
        //            }
        //            if (ReturnsItem != null && ReturnsItem.Count > 0)
        //            {
        //                SalesTransaction.TransactionTypeId = TransactionType.SalesExchange.GetHashCode();
        //                SalesTransaction.SourceTransactionId = transaction.ReturnId;
        //                SalesTransaction.TransactionType = TransactionType.SalesExchange;
        //            }
        //            else
        //            {
        //                SalesTransaction.TransactionTypeId = TransactionType.Sales.GetHashCode();
        //            }
        //            SalesTransaction.Voided = false;
        //            SalesTransaction.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
        //            SalesTransaction.UpdatedBy = _requestRepository.CurrentOperator.EmployeeId;
        //            SalesTransaction.CreatedDate = DateTime.Now;
        //            SalesTransaction.UpdatedDate = DateTime.Now;
        //            SalesTransaction.RouteId = transaction.RouteId;
        //            SalesTransaction.SupervisorId = supervisorId;
        //            SalesTransaction.HelperId = helperId;
        //            SalesTransaction.DriverId = driverId;
        //            SalesTransaction.SalesManagerId = SalesManagerId;
        //            SalesTransaction.SalesRepId = salesRepID;
        //            SalesTransaction.OrganizationId = transaction.OrganizationId;
        //            SalesTransaction.CurrencyId = currency;
        //            customertype = _customerManager.GetCustomerType(SalesTransaction.CustomerId, SalesTransaction.OutletId, dBHelper).ToString();
        //            SalesTransaction.SalesMode = (SalesModes)int.Parse(customertype);
        //            SalesTransaction.CreationReason = TransactionCreationReason.Normal;
        //            SalesTransaction.DivisionId = transaction.DivisionId;
        //        }
        //        TransactionModel ReturnsTransaction = new TransactionModel();
        //        ReturnsTransaction.SoldItems = ReturnsItem;
        //        if (ReturnsItem != null && ReturnsItem.Count > 0)
        //        {
        //            transaction.TransactionType = TransactionType.Return;
        //            ReturnsTransaction.TransactionId = transaction.ReturnId;
        //            ReturnsTransaction.TransactionDate = transaction.TransactionDate;
        //            ReturnsTransaction.CustomerId = transaction.CustomerId;
        //            ReturnsTransaction.OutletId = transaction.OutletId;
        //            ReturnsTransaction.EmployeeId = transaction.EmployeeId;
        //            ReturnsTransaction.RouteId = transaction.RouteId;
        //            ReturnsTransaction.TransactionDate = new DateTime(transaction.TransactionDateModel.Year, transaction.TransactionDateModel.Month, transaction.TransactionDateModel.Day);
        //            ReturnsTransaction.TransactionDate = ReturnsTransaction.TransactionDate;
        //            ReturnsTransaction.WarehouseId = transaction.RefWarehouseId;
        //            ReturnsTransaction.DivisionId = transaction.DivisionId;
        //            ReturnsTransaction.IsReturn = true;
        //            ReturnsTransaction.TransactionStatusId = TransactionStatues.None.GetHashCode();
        //            foreach (ItemPackModel returnInvoice in ReturnsItem)
        //            {
        //                ReturnsTransaction.GrossTotal += decimal.Round(returnInvoice.GrossTotal, numberOfDigits);
        //                ReturnsTransaction.NetTotal += decimal.Round(returnInvoice.NetTotal, numberOfDigits);
        //                ReturnsTransaction.RemainingAmount = 0;
        //                ReturnsTransaction.Tax += decimal.Round(returnInvoice.Tax, numberOfDigits);
        //                ReturnsTransaction.Discount += decimal.Round(returnInvoice.Discount, numberOfDigits);
        //                if (returnInvoice.ExpiryDateModel != null)
        //                {
        //                    returnInvoice.ExpiryDate = new DateTime(returnInvoice.ExpiryDateModel.Year, returnInvoice.ExpiryDateModel.Month, returnInvoice.ExpiryDateModel.Day);
        //                    returnInvoice.ExpiryDate = returnInvoice.ExpiryDate;
        //                }
        //            }
        //            if (SalesItem != null && SalesItem.Count > 0)
        //            {
        //                ReturnsTransaction.TransactionTypeId = TransactionType.ReturnExchange.GetHashCode();
        //                ReturnsTransaction.SourceTransactionId = transaction.TransactionId;
        //                ReturnsTransaction.TransactionType = TransactionType.ReturnExchange;
        //            }
        //            else
        //            {
        //                ReturnsTransaction.TransactionTypeId = TransactionType.Return.GetHashCode();
        //            }
        //            ReturnsTransaction.Voided = false;
        //            ReturnsTransaction.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
        //            ReturnsTransaction.UpdatedBy = _requestRepository.CurrentOperator.EmployeeId;
        //            ReturnsTransaction.CreatedDate = DateTime.Now;
        //            ReturnsTransaction.UpdatedDate = DateTime.Now;
        //            ReturnsTransaction.RouteId = transaction.RouteId;
        //            ReturnsTransaction.SupervisorId = supervisorId;
        //            ReturnsTransaction.HelperId = helperId;
        //            ReturnsTransaction.DriverId = driverId;
        //            ReturnsTransaction.SalesManagerId = SalesManagerId;
        //            ReturnsTransaction.SalesRepId = salesRepID;
        //            ReturnsTransaction.OrganizationId = transaction.OrganizationId;
        //            ReturnsTransaction.CurrencyId = currency;
        //            ReturnsTransaction.CreationReason = TransactionCreationReason.Normal;
        //            ReturnsTransaction.SalesMode = SalesModes.None;
        //        }
        //        CustomerOutletModel outlet = new CustomerOutletModel();
        //        outlet.CustomerId = transaction.CustomerId;
        //        outlet.OutletId = transaction.OutletId;
        //        outlet.CustomerTypeId = int.Parse(customertype);
        //        _transactionManager.ReadCustomerAccount(ref outlet, transaction.DivisionId, dBHelper);
        //        if (outlet.Account != null)
        //        {
        //            SalesTransaction.AccountId = outlet.Account.AccountId;
        //            ReturnsTransaction.AccountId = outlet.Account.AccountId;
        //            SalesTransaction.Outlet.Account.AccountId = outlet.Account.AccountId;
        //            ReturnsTransaction.Outlet.Account.AccountId = outlet.Account.AccountId;
        //        }
        //        else
        //        {
        //            errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Account_For_Outlet, _requestRepository.LanguageId);
        //            SalesTransaction = null;
        //            ReturnsTransaction = null;
        //            return GlobalErrors.Error;
        //        }
        //        int packSequence = 0;
        //        AppliedBatch appliedBatch = new AppliedBatch();
        //        foreach (var pack in SalesItem)
        //        {
        //            packSequence++;
        //            pack.Sequence = packSequence;
        //            if (pack.ExpiryDateModel != null && pack.SalesTransactionTypeId != SalesTransactionTypes.Promotion)
        //                pack.ExpiryDate = new DateTime(pack.ExpiryDateModel.Year, pack.ExpiryDateModel.Month, pack.ExpiryDateModel.Day);
        //            appliedBatch.ExpiryDate = pack.ExpiryDate;
        //            appliedBatch.BatchNumber = pack.BatchNo;
        //            appliedBatch.PackId = pack.PackId;
        //            appliedBatch.QuantityChange = pack.RequiredQty;
        //            appliedBatch.Price = pack.Price;
        //            appliedBatch.Tax = pack.CalculatedTax;
        //            result = _transactionManager.SaveTransactionDetailsOnBatchLevel(SalesTransaction, pack, dBHelper, appliedBatch);
        //            if (result != GlobalErrors.Success)
        //            {
        //                return GlobalErrors.Error;
        //            }
        //        }

        //        foreach (var pack in ReturnsItem)
        //        {
        //            packSequence++;
        //            pack.Sequence = packSequence;
        //            if (pack.ExpiryDateModel != null && pack.SalesTransactionTypeId != SalesTransactionTypes.Promotion)
        //                pack.ExpiryDate = new DateTime(pack.ExpiryDateModel.Year, pack.ExpiryDateModel.Month, pack.ExpiryDateModel.Day);
        //            appliedBatch.ExpiryDate = pack.ExpiryDate;
        //            appliedBatch.BatchNumber = pack.BatchNo;
        //            appliedBatch.PackId = pack.PackId;
        //            appliedBatch.QuantityChange = pack.RequiredQty;
        //            appliedBatch.Price = pack.Price;
        //            appliedBatch.Tax = pack.CalculatedTax;
        //            result = _transactionManager.SaveTransactionDetailsOnBatchLevel(ReturnsTransaction, pack, dBHelper, appliedBatch);
        //            if (result != GlobalErrors.Success)
        //            {
        //                return GlobalErrors.Error;
        //            }
        //        }
        //        if (result == GlobalErrors.Success && SalesTransaction != null && ReturnsTransaction != null)
        //        {
        //            result = _transactionManager.SaveExchange(SalesTransaction, ReturnsTransaction, outlet, dBHelper, ref errorMessage);
        //            if (result != GlobalErrors.Success)
        //            {
        //                return GlobalErrors.Error;
        //            }

        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
        //        result = GlobalErrors.Error;
        //    }
        //    finally
        //    {
        //        if (dBHelper != null && commitNow)
        //        {
        //            if (result == GlobalErrors.Success)
        //            {
        //                dBHelper.CommitTransaction();
        //            }
        //            else
        //            {
        //                dBHelper.RollBackTransaction();
        //            }
        //            dBHelper.Dispose();
        //            dBHelper = null;
        //        }
        //    }
        //    return result;
        //}

        public GlobalErrors GetSelectedPackStatues(List<ItemPackModel> orderItems, ref List<int> selectedStatuses, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                var orderPackStatuses = string.Join(',', orderItems.Select(i => i.PackStatusId).Distinct().ToList());
                if (!string.IsNullOrEmpty(orderPackStatuses))
                {
                    string resallableCondition = _requestRepository.Configurations.RestoreNonSellableReturns ? string.Empty : $" And PackStatus.ReSellable = {PackStatus.ReSellable.GetHashCode()} ";
                    string query = $@"select PackStatus.StatusID  from PackStatus where PackStatus.StatusID in ({orderPackStatuses}) {resallableCondition}";
                    result = dBHelper.GetQueryList(query, ref selectedStatuses);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error; ;
            }
            return result;
        }

        public bool GiveCreditOnReturn(CustomerTypes outletType)
        {
            try
            {
                if (outletType == CustomerTypes.CashCustomer)
                {
                    return _requestRepository.Configurations.GiveCreditOnReturnForCashCustomers;
                }
                else
                {
                    return _requestRepository.Configurations.GiveCreditOnReturnForCreditCustomers;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return false;
            }
        }

        public bool GivePartialCredit(CustomerTypes outletType)
        {
            try
            {
                if (outletType == CustomerTypes.CashCustomer)
                {
                    return _requestRepository.Configurations.GivePartialCreditForCashCustomer;
                }
                if (outletType == CustomerTypes.CreditCustomer || outletType == CustomerTypes.BillToBillCustomer)
                {
                    return _requestRepository.Configurations.GivePartialCreditForCreditCustomer;
                }
                return false;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return false;
            }
        }


        public GlobalErrors GetSourceTransactionData(string transactionID, ref TransactionModel transactionModel)
        {
            try
            {
                return _transactionManager.GetSourceTransactionData(transactionID, ref transactionModel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SaveICashReversal(TransactionModel transaction, ref bool noDocSeq)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                transaction.DocumentType = DocumentTypes.ICashReversal;
                transaction.TransactionId = _documentSequenceService.GetMaxDocumentSequence(_requestRepository.CurrentOperator.EmployeeId, DocumentTypes.ICashReversal, transaction.DivisionId);
                if (string.IsNullOrEmpty(transaction.TransactionId))
                {
                    noDocSeq = true;
                    return GlobalErrors.Error;
                }

                // Fill Data
                transaction.RouteHistoryId = -1;
                transaction.Posted = false;
                transaction.RemainingAmount = 0;

                transaction.NetTotal = LocalUtilities.GetRoundedDecimal(transaction.NetTotal, _requestRepository.Configurations.NumberOfDigits);
                transaction.GrossTotal = LocalUtilities.GetRoundedDecimal(transaction.GrossTotal, _requestRepository.Configurations.NumberOfDigits);
                transaction.TransactionType = TransactionType.Sales;
                transaction.TransactionStatusId = TransactionStatues.None.GetHashCode();

                transaction.TransactionDate = DateTime.Now;
                transaction.CreatedDate = transaction.TransactionDate;
                transaction.UpdatedDate = transaction.TransactionDate;
                transaction.UpdatedBy = _requestRepository.CurrentOperator.EmployeeId;
                transaction.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
                transaction.Voided = false;
                transaction.SalesMode = SalesModes.CreditSales;
                transaction.RouteId = -1;
                transaction.TransactionStatusId = TransactionStatues.None.GetHashCode();
                transaction.CreationReason = TransactionCreationReason.ICashReversal;
                transaction.SourceTransactionId = transaction.SourceTransactionId;

                // Saving 
                result = _transactionManager.SaveICashReversalTransaction(transaction);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                result = GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetTransactionsList(GeneralFilter filter, ref SharedTableResult<TransactionModel> custList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _transactionManager.GetTransactionsList(filter, ref custList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors SaveTransactionWithoutStockPosting(TransactionModel transaction, bool saveDetails, ref string errorMessage)
        {
            return SaveTransactionWithoutStockPosting(transaction, saveDetails, ref errorMessage, null);
        }
        public GlobalErrors SaveTransactionWithoutStockPosting(TransactionModel transaction, bool saveDetails, ref string errorMessage,DBHelper<int> dBHelper)
        {
            var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;
            transaction.TransactionId = _documentSequenceService.GetMaxDocumentSequence(transaction.EmployeeId, transaction.DocumentType, transaction.DivisionId);
            if (string.IsNullOrEmpty(transaction.TransactionId))
            {
                errorMessage = ResourcesManager.TranslateKey(MessagesConstants.Desc_No_Document_Sequence, _requestRepository.LanguageId);
                return GlobalErrors.Error;
            }

            // Fill Data
            //TransactionService.DecryptedIDs(ref transaction);
            transaction.RouteHistoryId = -1;
            transaction.Posted = false;
            transaction.RemainingAmount = transaction.NetTotal;
            transaction.NetTotal = LocalUtilities.GetRoundedDecimal(transaction.NetTotal, numberOfDigits);
            transaction.GrossTotal = LocalUtilities.GetRoundedDecimal(transaction.GrossTotal, numberOfDigits);
            transaction.RemainingAmount = LocalUtilities.GetRoundedDecimal(transaction.RemainingAmount, numberOfDigits);
            transaction.Discount = LocalUtilities.GetRoundedDecimal(transaction.Discount, numberOfDigits);
            transaction.Tax = LocalUtilities.GetRoundedDecimal(transaction.Tax, numberOfDigits);
            transaction.CalculatedRetailTax = LocalUtilities.GetRoundedDecimal(transaction.CalculatedRetailTax, numberOfDigits);
            transaction.PromotedDiscount = LocalUtilities.GetRoundedDecimal(transaction.PromotedDiscount, numberOfDigits);
            transaction.TransactionDate = new DateTime(transaction.TransactionDateModel.Year, transaction.TransactionDateModel.Month, transaction.TransactionDateModel.Day, DateTime.Now.Hour, DateTime.Now.Minute, DateTime.Now.Second);
            transaction.CreatedDate = DateTime.Now;
            transaction.UpdatedDate = DateTime.Now;
            transaction.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
            transaction.UpdatedBy = transaction.CreatedBy;
            transaction.Voided = false;
            transaction.AccountId = transaction.Outlet.Account.AccountId;
            if (_requestRepository.Configurations.ApplyCurrencyPerOrganization)
                transaction.CurrencyId = _transactionManager.getOrganizationBaseCurrency(transaction.OrganizationId);
            else
                transaction.CurrencyId = transaction.Outlet.Account.CurrencyId;
            transaction.SalesMode = transaction.Outlet.CustomerTypeId == CustomerTypes.CashCustomer.GetHashCode() ? SalesModes.CashSales : SalesModes.CreditSales;
            transaction.TransactionStatusId = TransactionStatues.None.GetHashCode();
            transaction.CreationReason = TransactionCreationReason.Normal;
            if (transaction.Outlet != null && transaction.Outlet.CustOutPaymentTerm != null)
            {
                PaymentTermModel paymentTerm = new PaymentTermModel();
                paymentTerm = transaction.Outlet.CustOutPaymentTerm;
                transaction.DueDate = DateTime.Now.AddDays(paymentTerm.GetDays());
            }
            return _transactionManager.SaveTransactionWithoutStockPosting(transaction, saveDetails, dBHelper, ref errorMessage);
        }
        public string GetStringQRCode(int customerId, int outletId, string transactionId, ReportMode reportMode)
        {
            try
            {
                return _transactionManager.GetStringQRCode(customerId, outletId, transactionId, reportMode);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return string.Empty;
        }
        #region[CRM]
        public GlobalErrors GetTransactionsHistory(CustomerFilter filter, ref SharedTableResult<TransactionHistory> transactionsList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;

                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 20;

                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }

                return _transactionManager.GetTransactionsHistory(filter, ref transactionsList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetOrdersHistory(CustomerFilter filter, ref List<CRMOrderModel> ordersList)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {

                List<CRMOrderDetailsModel> ordersDetails = new List<CRMOrderDetailsModel>();
                result = _transactionManager.GetOrdersHistory(filter, ref ordersDetails);
                if (result == GlobalErrors.Success && ordersDetails != null && ordersDetails.Count > 0)
                {
                    ordersList = ordersDetails.GroupBy(i => new { i.OrderId, i.DivisionId }).Select(grp => new CRMOrderModel
                    {
                        OrderId = grp.FirstOrDefault().OrderId,
                        CustomerId = grp.FirstOrDefault().CustomerId,
                        OutletId = grp.FirstOrDefault().OutletId,
                        DivisionId = grp.FirstOrDefault().DivisionId,
                        OrganizationId = grp.FirstOrDefault().OrderOrganizationId,
                        OrderNetTotal = grp.FirstOrDefault().OrderNetTotal,
                        OrderStatusId = grp.FirstOrDefault().OrderStatusId,
                        OrderStatus = grp.FirstOrDefault().OrderStatus,
                        OrderSubStatusId = grp.FirstOrDefault().OrderSubStatusId,
                        OrderSubStatus = grp.FirstOrDefault().OrderSubStatus,
                        OrderBy = grp.FirstOrDefault().OrderBy,
                        OrderDate = grp.FirstOrDefault().OrderDate,
                        OrderStatusColor = grp.FirstOrDefault().OrderStatusColor,
                        Synchronized = grp.FirstOrDefault().Synchronized,
                        AllowEdit = grp.FirstOrDefault().AllowEdit,
                        OrderDetails = ordersDetails.Where(i => i.OrderId == grp.FirstOrDefault().OrderId && i.DivisionId == grp.FirstOrDefault().DivisionId).ToList()
                    }).ToList();
                }
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetTransactionDetails(TransactionHistory transaction, ref List<ItemPackModel> transactionDetails, ref bool allowEdit)
        {
            var result = GlobalErrors.NotInitialized;
            allowEdit = false;
            try
            {

                if (transaction.IsTransaction || (transaction.IsOrder && (int)transaction.OrderStatusId == (int)OrderStatus.Delivered))
                {
                    result = _transactionManager.GetTransactionDetailsAsReport(transaction, ref transactionDetails);
                }
                else if (transaction.IsOrder)
                {
                    result = _transactionManager.GetOrderDetailsAsReport(transaction, ref transactionDetails);
                    if (result == GlobalErrors.Success)
                    {
                        result = _orderManager.CheckOrderCanbeEdited(transaction.OrderId, transaction.CustomerId, transaction.OutletId, ref allowEdit);
                    }
                }
                //else if (transaction.IsTopUp)
                //{
                //    result = _iCashManager.GetTopupDetails(transaction, ref transactionDetails);
                //}
                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }

        public GlobalErrors GetCustomerSalesSummary(CustomerFilter filter, ref SalesSummaryModel summary)
        {
            var result = GlobalErrors.NotInitialized;
            var salesValueList = new List<MonthlySummaryModel>();
            var salesVolumeList = new List<MonthlySummaryModel>();
            try
            {
                result = _transactionManager.GetCustomerSalesSummaryHeader(filter, ref summary);

                if (result == GlobalErrors.Success)
                {
                    result = _transactionManager.GetMonthlySalesValue(filter, ref salesValueList);

                    if (result == GlobalErrors.Success && salesValueList != null && salesValueList.Count > 0)
                    {
                        summary.SalesValue.CurrentMonth = salesValueList.Where(a => a.Month == DateTime.Now.Month && a.Year == DateTime.Now.Year).ToList().Sum(a => a.Value);

                        summary.SalesValue.CurrentQuarter = salesValueList.Where(a => a.Quarter == (DateTime.Now.Month + 2) / 3 && a.Year == DateTime.Now.Year).ToList().Sum(a => a.Value);

                        summary.SalesValue.CurrentYear = salesValueList.Where(a => a.Year == DateTime.Now.Year).ToList().Sum(a => a.Value);

                        summary.SalesValue.LastYear = salesValueList.Where(a => a.Year == (DateTime.Now.Year - 1)).ToList().Sum(a => a.Value);

                        summary.SalesValue.Overall = salesValueList.Sum(a => a.Value);


                    }
                }

                if (result == GlobalErrors.Success)
                {
                    result = _transactionManager.GetMonthlySalesVolume(filter, ref salesVolumeList);

                    summary.SalesVolume.CurrentMonth = salesVolumeList.Where(a => a.Month == DateTime.Now.Month && a.Year == DateTime.Now.Year).ToList().Sum(a => a.Value);

                    summary.SalesVolume.CurrentQuarter = salesVolumeList.Where(a => a.Quarter == (DateTime.Now.Month + 2) / 3 && a.Year == DateTime.Now.Year).ToList().Sum(a => a.Value);

                    summary.SalesVolume.CurrentYear = salesVolumeList.Where(a => a.Year == DateTime.Now.Year).ToList().Sum(a => a.Value);

                    summary.SalesVolume.LastYear = salesVolumeList.Where(a => a.Year == (DateTime.Now.Year - 1)).ToList().Sum(a => a.Value);

                    summary.SalesVolume.Overall = salesVolumeList.Sum(a => a.Value);

                }

                return result;
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        #endregion
        public GlobalErrors PrepareAndSaveDebitNote(ref TransactionModel debitTransaction, TransactionModel orderTransaction, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            string errorMessage = string.Empty;
            try
            {
                var numberOfDigits = _requestRepository.Configurations.NumberOfDigits;

                if (debitTransaction == null)
                {
                    debitTransaction = new TransactionModel();
                }
                debitTransaction.EmployeeId = _requestRepository.CurrentOperator.EmployeeId;
                debitTransaction.DocumentType = DocumentTypes.DebitNote;
                debitTransaction.DivisionId = _requestRepository.Configurations.ControlTransactionOnDivisionLevel ? orderTransaction.DivisionId : -1;
                debitTransaction.TransactionId = _documentSequenceService.GetMaxDocumentSequence(debitTransaction.EmployeeId, debitTransaction.DocumentType, debitTransaction.DivisionId);
                debitTransaction.RemainingAmount = debitTransaction.AppliedAmount;
                debitTransaction.RouteHistoryId = -1;
                debitTransaction.Posted = false;
                debitTransaction.TransactionDate = new DateTime(orderTransaction.TransactionDateModel.Year, orderTransaction.TransactionDateModel.Month, orderTransaction.TransactionDateModel.Day, DateTime.Now.Hour, DateTime.Now.Minute, DateTime.Now.Second);
                debitTransaction.CreatedDate = DateTime.Now;
                debitTransaction.UpdatedDate = DateTime.Now;
                debitTransaction.CreatedBy = _requestRepository.CurrentOperator.EmployeeId;
                debitTransaction.UpdatedBy = orderTransaction.CreatedBy;
                debitTransaction.Voided = false;
                debitTransaction.AccountId = orderTransaction.Outlet.Account.AccountId;
                debitTransaction.Outlet = orderTransaction.Outlet;
                debitTransaction.TransactionType = TransactionType.DebitNote;
                debitTransaction.TransactionTypeId = TransactionType.DebitNote.GetHashCode();
                debitTransaction.SourceTransactionId = orderTransaction.TransactionId;
                if (_requestRepository.Configurations.ApplyCurrencyPerOrganization)
                    debitTransaction.CurrencyId = _transactionManager.getOrganizationBaseCurrency(debitTransaction.OrganizationId);
                else
                    debitTransaction.CurrencyId = debitTransaction.Outlet.Account.CurrencyId;
                //debitTransaction.SalesMode = debitTransaction.Outlet.CustomerTypeId == CustomerTypes.CashCustomer.GetHashCode() ? SalesModes.CashSales : SalesModes.CreditSales;
                debitTransaction.SalesMode = SalesModes.CashSales;
                debitTransaction.TransactionStatusId = TransactionStatues.None.GetHashCode();
                debitTransaction.CreationReason = TransactionCreationReason.Normal;

                result = _transactionManager.SaveTransactionWithoutStockPosting(debitTransaction, false, dBHelper, ref errorMessage);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetVMTransactionsForSelectedCustomers(GeneralFilter filter, ref SharedTableResult<TransactionModel> custList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _transactionManager.GetVMTransactionsForSelectedCustomers(filter, ref custList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
        }
        public GlobalErrors GetVMTransactionsDetails(GeneralFilter filter, ref List<ItemPackModel> itemList)
        {
            try
            {
                return _transactionManager.GetVMTransactionsDetails(filter, ref itemList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                itemList = null;
                return GlobalErrors.Error;
            }
        }
        #region [CRM]
        public GlobalErrors GetCRMUnpaidTransactionsForCustomers(CustomerFilter filter, ref SharedTableResult<TransactionHistory> transactionsList, ref List<PayerAccountModel> accountList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;

                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 20;

                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }

                return _transactionManager.GetCRMUnpaidTransactionsForCustomers(filter, ref transactionsList,ref accountList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        #endregion
        #region [Return]
        public GlobalErrors GetReturnTransactions(GeneralFilter filter, ref SharedTableResult<TransactionModel> custList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _transactionManager.GetReturnTransactions(filter, ref custList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
        }
        #endregion

        #region [Exchange]
        public GlobalErrors GetExchangeTransactions(GeneralFilter filter, ref SharedTableResult<TransactionModel> custList)
        {
            try
            {
                if (filter.CustomListFilter.Page < 0)
                    filter.CustomListFilter.Page = 0;
                if (filter.CustomListFilter.PageSize <= 0)
                    filter.CustomListFilter.PageSize = 100;
                if (!string.IsNullOrEmpty(filter.CustomListFilter.SearchFilter))
                {
                    filter.CustomListFilter.SearchFilter = filter.CustomListFilter.SearchFilter.Replace("'", "''");
                }
                return _transactionManager.GetExchangeTransactions(filter, ref custList);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                custList = null;
                return GlobalErrors.Error;
            }
        }
        #endregion
    }
}