﻿using Microsoft.AspNetCore.Hosting;
using Sonic.Web.Core;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Models;
using Sonic.Web.Resources;
using Sonic.Web.Services;
using System;
using System.Collections.Generic;
using System.Data;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.IO;
using Microsoft.AspNetCore.Http;
using System.Collections.ObjectModel;

namespace Sonic.Web.Service
{
    public class WebFormService
    {
        private readonly IRequestRepository _requestRepository;
        private readonly WebFormManager _webFormManager;
        private readonly MasterDataService _masterDataService;
        private readonly SharedMethod _sharedMethod;
        private readonly DynamicReportManager _dynamicReportManager;
        public WebFormService(IRequestRepository requestRepository)
        {
            _requestRepository = requestRepository;
            _webFormManager = new WebFormManager(_requestRepository);
            _masterDataService = new MasterDataService(_requestRepository);
            _sharedMethod = new SharedMethod(_requestRepository);
            _dynamicReportManager = new DynamicReportManager(_requestRepository);
        }
        public GlobalErrors InitializeWebFormList(InitializeWebFormList retriveWebFormModel, ref WebFormModel formModel)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                formModel = new WebFormModel();
                List<WebFormColumnModel> columns = new List<WebFormColumnModel>();
                List<WebFormFilterModel> filters = new List<WebFormFilterModel>();
                List<WebFormActionProcedure> procedures = new List<WebFormActionProcedure>();
                SharedTableResult<dynamic> sharedTableDataSource = null;
                bool isFiltered = false;

                #region[From]
                result = _webFormManager.GetWebForm(retriveWebFormModel.URL, -1, WebFormType.List, ref formModel);
                if (result != GlobalErrors.Success || formModel == null || formModel.WebFormId <= 0)
                {
                    return GlobalErrors.Error;
                }
                #endregion

                #region[Filters]
                result = _webFormManager.FillWebFormFilters(formModel.WebFormId, ref filters);
                if (result == GlobalErrors.Success)
                {
                    formModel.Filters = filters;
                }
                #endregion

                #region[Columns]
                if (result == GlobalErrors.Success)
                {
                    result = _webFormManager.FillWebFormColumns(formModel.WebFormId, ref columns);
                    if (result == GlobalErrors.Success)
                    {
                        formModel.Columns = columns;
                    }
                }
                #endregion

                #region[Procedures]
                if (result == GlobalErrors.Success)
                {
                    result = _webFormManager.FillWebFormProcedures(formModel.WebFormId, ref procedures);
                    if (result == GlobalErrors.Success)
                    {
                        formModel.ActionProcedures = procedures;
                    }
                }
                #endregion

                #region[Data]
                if (result == GlobalErrors.Success)
                {
                    result = _webFormManager.FillWebFormData(formModel, retriveWebFormModel.SharedTableFilter, isFiltered, ref sharedTableDataSource);
                    if (result == GlobalErrors.Success)
                    {
                        formModel.SharedTableDataSource = sharedTableDataSource;
                    }
                }
                #endregion

            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors GetWebFormListData(InitializeWebFormList retriveWebFormModel, ref WebFormModel formModel)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            bool isFiltered = true;
            try
            {
                //formModel = new WebFormModel();
                SharedTableResult<dynamic> sharedTableDataSource = new SharedTableResult<dynamic>();
                //formModel.Columns = retriveWebFormModel.Columns;
                //formModel.Filters = retriveWebFormModel.Filters;
                //formModel.WebFormId = retriveWebFormModel.WebFormId;
                //formModel.WebMenuId = retriveWebFormModel.WebMenuId;
                //formModel.DefaultSortingPattern = retriveWebFormModel.DefaultSortingPattern;

                result = _webFormManager.FillWebFormData(retriveWebFormModel.Form, retriveWebFormModel.SharedTableFilter, isFiltered, ref sharedTableDataSource);
                if (result == GlobalErrors.Success)
                {
                    formModel.SharedTableDataSource = sharedTableDataSource;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors InitializeWebFormEntry(InitializeWebFormEntry retriveWebFormModel, ref WebFormModel formModel, IHostingEnvironment hostingEnvironment)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                formModel = new WebFormModel();
                List<WebFormControlModel> controls = new List<WebFormControlModel>();
                List<WebFormTable> tables = new List<WebFormTable>();
                List<WebFormActionProcedure> procedures = new List<WebFormActionProcedure>();

                #region[From]
                result = _webFormManager.GetWebForm(string.Empty, retriveWebFormModel.WebFormId, WebFormType.Entry, ref formModel);
                if (result != GlobalErrors.Success || formModel == null || formModel.WebFormId <= 0)
                {
                    return GlobalErrors.Error;
                }
                #endregion

                #region[Controls]
                result = _webFormManager.FillEntryControls(formModel.WebFormId, ref controls);
                if (result == GlobalErrors.Success)
                {
                    formModel.Controls = controls;
                }
                #endregion

                #region[Tables]
                result = _webFormManager.FillEntryTables(formModel.WebFormId, ref tables);
                if (result == GlobalErrors.Success)
                {
                    formModel.FormTables = tables;
                }
                #endregion

                #region[Procedures]
                if (result == GlobalErrors.Success)
                {
                    result = _webFormManager.FillWebFormProcedures(formModel.WebFormId, ref procedures);
                    if (result == GlobalErrors.Success)
                    {
                        formModel.ActionProcedures = procedures;
                    }
                }
                #endregion

                #region[Data]
                if (retriveWebFormModel.GetSavedDetails)
                {
                    result = _webFormManager.FillEntryWebFormDataOnEdit(formModel, retriveWebFormModel.PrimaryKeyColumnList);
                }
                #endregion

                #region[Fill Childs Controls Data Source In Edit Mode]
                if (retriveWebFormModel.GetSavedDetails && formModel.Controls.Any(control => control.FilterBySpecificControl && !control.BuiltIn && (control.ControlTypeId == WebFormControlType.DropDown || control.ControlTypeId == WebFormControlType.MultiSelectDropDown) && !string.IsNullOrEmpty(control.ParentFilterControlParameter)))
                {
                    foreach (WebFormControlModel childControl in formModel.Controls.Where(control => control.FilterBySpecificControl && !control.BuiltIn && (control.ControlTypeId == WebFormControlType.DropDown || control.ControlTypeId == WebFormControlType.MultiSelectDropDown) && !string.IsNullOrEmpty(control.ParentFilterControlParameter)))
                    {
                        if (formModel.Controls.Any(parentControl => parentControl.ControlIndex == childControl.ParentFilterControlIndex && !string.IsNullOrEmpty(parentControl.SelectedValue)))
                        {
                            childControl.ParentFilterControlValue = formModel.Controls.Find(parentControl => parentControl.ControlIndex == childControl.ParentFilterControlIndex && !string.IsNullOrEmpty(parentControl.SelectedValue)).SelectedValue;

                            List<ControlDataSourceModel> dataSourceList = new List<ControlDataSourceModel>();
                            string disabledControlValue = string.Empty;
                            result = _webFormManager.FillControlDataSourceBasedOnParentControlValue(childControl, ref dataSourceList, ref disabledControlValue);
                            if (result == GlobalErrors.Success)
                            {
                                formModel.Controls.Find(x => x.ControlIndex == childControl.ControlIndex).DataSourceList = dataSourceList;
                            }
                            if (result != GlobalErrors.Success)
                            {
                                return GlobalErrors.Error;
                            }
                        }
                    }
                }
                #endregion

                #region [Fill Selected Values For Multi Select]
                if (retriveWebFormModel.GetSavedDetails && formModel.Controls.Any(control => control.ControlTypeId == WebFormControlType.MultiSelectDropDown))
                {
                    foreach (WebFormControlModel control in formModel.Controls.Where(control => control.ControlTypeId == WebFormControlType.MultiSelectDropDown))
                    {
                        List<string> selectedValues = new List<string>();
                        result = _webFormManager.FillSelectedValuesForMultiSelectControl(control, retriveWebFormModel.PrimaryKeyColumnList, ref selectedValues);
                        if (result == GlobalErrors.Success && selectedValues.Count > 0)
                        {
                            control.SelectedValuesList = new List<string>();
                            foreach (string valueId in selectedValues)
                            {
                                if (control.DataSourceList.Any(x => x.ValueId == valueId))
                                {
                                    control.SelectedValuesList.Add(valueId);
                                }
                            }
                        }
                        if (result != GlobalErrors.Success)
                        {
                            return result;
                        }
                    }
                }
                #endregion

                #region Fill Attachments
                // if view or edit mode
                if (result == GlobalErrors.Success && formModel.HasAttachments && retriveWebFormModel.GetSavedDetails)
                {
                    List<string> attachementNames = new List<string>();
                   result =  FillAttachments(formModel, hostingEnvironment, retriveWebFormModel.WebMenuURL, ref attachementNames);
                    if(result == GlobalErrors.Success  && attachementNames.Count > 0)
                    {
                        formModel.AttachedFiles = attachementNames;
                    }
                }
                #endregion
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors FillAttachments(WebFormModel form, IHostingEnvironment hostingEnvironment, string WebMenuURL, ref List<string> attachementNames)
        {
            try
            {
                var primaryKeys = form.Controls.Where(x => x.IsPrimary).ToList();
                var attachmentFolder = string.Empty;
                foreach (var controlModel in primaryKeys)
                {
                    if (controlModel.ControlTypeId == WebFormControlType.Date)
                    {
                        var date = controlModel.SelectedDateModelValue.Year.ToString() + '-' + controlModel.SelectedDateModelValue.Month.ToString() + '-' + controlModel.SelectedDateModelValue.Day.ToString() + '-' + controlModel.SelectedDateModelValue.Hour.ToString() + '-' + controlModel.SelectedDateModelValue.Minute.ToString() + '-' + controlModel.SelectedDateModelValue.Second.ToString();
                        attachmentFolder += date + "_";
                    }
                    else
                    {
                        attachmentFolder += controlModel.SelectedValue + "_";
                    }
                }
                if (!string.IsNullOrEmpty(attachmentFolder))
                {
                    attachmentFolder = attachmentFolder.Remove(attachmentFolder.Length - 1);
                    form.AttachmentFolder = attachmentFolder;
                    FileManagerModel model = new FileManagerModel();

                    string rootPath = hostingEnvironment.ContentRootPath;
                    string folderName = Path.Combine("Resources\\attachements\\", WebMenuURL, attachmentFolder);
                    var userAttachmentPath = Path.Combine(rootPath, folderName);

                    DirectoryInfo dir = new DirectoryInfo(userAttachmentPath);
                    if (dir.Exists)
                    {
                        FileInfo[] files = dir.GetFiles();
                        model.Files = files;
                        if (files != null && files.Length > 0)
                        {
                            foreach (FileInfo file in files)
                            {
                                attachementNames.Add(file.Name);
                            }
                        }

                        if (attachementNames != null && attachementNames.Count > 0)
                        {
                            form.AttachedFiles = attachementNames;
                        }
                    }
                }
               
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);

            }

            return GlobalErrors.Success;
        }

        public GlobalErrors AddAttachments(WebFormModel form, IHostingEnvironment hostingEnvironment, string WebMenuURL, Collection<IFormFile> attachementNames)
        {
            try
            {

                var primaryKeys = form.Controls.Where(x => x.IsPrimary).ToList();
                var attachmentFolder = string.Empty;
                foreach (var controlModel in primaryKeys)
                {
                    if (controlModel.ControlTypeId == WebFormControlType.Date)
                    {
                        var date =
                            controlModel.SelectedDateModelValue.Year.ToString() + '-' +
                            controlModel.SelectedDateModelValue.Month.ToString() + '-' +
                            controlModel.SelectedDateModelValue.Day.ToString() + '-' +
                            controlModel.SelectedDateModelValue.Hour.ToString() + '-' +
                            controlModel.SelectedDateModelValue.Minute.ToString() + '-' +
                            controlModel.SelectedDateModelValue.Second.ToString();
                        attachmentFolder += date + "_";
                    }
                    else
                    {
                        attachmentFolder += controlModel.SelectedValue + "_";
                    }
                }
                if (!string.IsNullOrEmpty(attachmentFolder))
                {

                    attachmentFolder = attachmentFolder.Remove(attachmentFolder.Length - 1);
                    form.AttachmentFolder = attachmentFolder;
                    string rootPath = hostingEnvironment.ContentRootPath;
                    string folderName = Path.Combine("Resources\\attachements\\", form.WebMenuURL);
                    string fullPath = Path.Combine(folderName, form.AttachmentFolder);

                    if (!Directory.Exists(fullPath))
                    {
                        Directory.CreateDirectory(fullPath);
                    }
                    else
                    {
                        Directory.Delete(fullPath, true);
                        Directory.CreateDirectory(fullPath);
                    }

                    for (int i = 0; i < attachementNames.Count; i++)
                    {
                        var file = attachementNames[i];
                        if (file.Length > 0)
                        {
                            string fileName = file.FileName;
                            string filePath = Path.Combine(fullPath, fileName);
                            using (var stream = new FileStream(filePath, FileMode.Create))
                            {
                                file.CopyTo(stream);
                            }

                        }
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);

            }

            return GlobalErrors.Success;
        }

        public GlobalErrors FillControlsDataSourceBasedOnParentControlValue(ref List<WebFormControlModel> controls)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                foreach (WebFormControlModel control in controls)
                {
                    List<ControlDataSourceModel> dataSourceList = new List<ControlDataSourceModel>();
                    string disabledControlValue = string.Empty;
                    result = _webFormManager.FillControlDataSourceBasedOnParentControlValue(control, ref dataSourceList, ref disabledControlValue);
                    if (result == GlobalErrors.Success)
                    {
                        if (control.ControlTypeId == WebFormControlType.DropDown || control.ControlTypeId == WebFormControlType.MultiSelectDropDown)
                        {
                            control.DataSourceList = dataSourceList;
                        }
                        else if (control.ControlTypeId == WebFormControlType.DisabledText)
                        {
                            control.SelectedValue = disabledControlValue;
                        }
                    }
                    else
                    {
                        return result;
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors SaveEntryForm(WebFormModel entryForm, ref string validationMessage, IHostingEnvironment hostingEnvironment, Collection<IFormFile> attachedFiles)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            int attachedCount = 0;
            try
            {
                dbHelper = new DBHelper<int>();
                dbHelper.BeginTransaction();
                List<WebFormControlModel> primaryKeys = new List<WebFormControlModel>();

                if (entryForm.IsAddMode)
                {
                    foreach (WebFormControlModel control in entryForm.Controls.Where(x => x.IsPrimary).ToList())
                    {
                        if (string.IsNullOrEmpty(control.SelectedValue))
                        {
                            if (control.ControlTypeId == WebFormControlType.Integer)
                            {
                                control.SelectedValue = _webFormManager.GetMaxID(dbHelper, control.DBColumnName, control.TableName).ToString();
                                if (control.SelectedValue == "-1")
                                {
                                    return GlobalErrors.Error;
                                }
                            }
                            else if (control.ControlTypeId == WebFormControlType.SCOPE_IDENTITY)
                            {
                                // Handle Scop Identity primary key value later
                            }
                        }
                    }
                }
                primaryKeys = entryForm.Controls.Where(x => x.IsPrimary).ToList();

                foreach (WebFormTable table in entryForm.FormTables)
                {
                    #region[Save Main Table]
                    if (!table.IsAssignmentTable && !table.TableName.ToLower().Contains("Language".ToLower()))
                    {
                        attachedCount = attachedFiles.Count;
                        if (entryForm.IsAddMode)
                        {
                            result = _webFormManager.SaveNewEntryFormData(entryForm, table, attachedCount, dbHelper);                            
                        }
                        else if (entryForm.IsEditMode)
                        {
                            //if (entryForm.WebFormName.ToLower() == "branchentry" && table.TableName == "bankbranch") // no update over this table
                            if (entryForm.Controls.Any(x => x.TableName == table.TableName && x.AllowEdit)) // if theres is any editable clm then update the table
                            {
                                result = _webFormManager.EditEntryFormData(entryForm, table, attachedCount, dbHelper);
                            } 
                            else if (table.HasAttachmentColumn)
                            {
                                result = _webFormManager.EditEntryFormData(entryForm, table, attachedCount, dbHelper);
                            }
                            else
                            {
                                result = GlobalErrors.Success;
                            }
                        }
                        if (result != GlobalErrors.Success)
                        {
                            return result;
                        }
                    }
                    #endregion

                    #region[Save Language Table]
                    else if (table.TableName.ToLower().Contains("Language".ToLower())) // Save into language table
                    {
                        if (entryForm.IsEditMode)
                        {
                            result = _webFormManager.DeleteDescriptions(primaryKeys, entryForm, table, dbHelper);
                            if (result != GlobalErrors.Success)
                            {
                                return result;
                            }
                        }
                        if (entryForm.Controls.Any(x => x.ControlTypeId == WebFormControlType.DescriptionText && x.TableName.ToLower() == table.TableName.ToLower() && x.DescriptionLanguages != null && x.DescriptionLanguages.Count > 0))
                        {
                            result = _webFormManager.SaveWebFormEntryLanguage(primaryKeys, entryForm, table, dbHelper);
                        }
                        if (result != GlobalErrors.Success)
                        {
                            return result;
                        }
                    }
                    #endregion

                    #region[Save In Assignemnt Table]
                    else if (table.IsAssignmentTable) // Save into Assignment table
                    {
                        if (entryForm.IsEditMode)
                        {
                            result = _webFormManager.DeleteFromAssignemntTable(primaryKeys, entryForm, table, dbHelper);
                            if (result != GlobalErrors.Success)
                            {
                                return result;
                            }
                        }
                        if (entryForm.Controls.Any(x => x.ControlTypeId == WebFormControlType.MultiSelectDropDown && x.TableName.ToLower() == table.TableName.ToLower() && x.SelectedValuesList != null && x.SelectedValuesList.Count > 0))
                        {
                            result = _webFormManager.SaveWebFormEntryAssignemnt(primaryKeys, entryForm, table, dbHelper);
                        }
                        if (result != GlobalErrors.Success)
                        {
                            return result;
                        }
                    }
                    #endregion
                }

                #region [Report Scedual]
                if (entryForm.IsAddMode && entryForm.WebFormName.ToLower() == "reportscheduleentry")
                {
                    ReportScheduleModel reportScheduleModel = new ReportScheduleModel();
                    result = _dynamicReportManager.GetInsertedScheduleProperties(ref reportScheduleModel , dbHelper, entryForm.FilledPrimaryKeyCondition);
                    if (result == GlobalErrors.Success && reportScheduleModel.IsEndOfDaySubmission == false)
                    {
                        result = _sharedMethod.GetReportScheduleNextRunTime(reportScheduleModel);
                        result = _dynamicReportManager.UpdateNewTimeToRun(ref reportScheduleModel, dbHelper, false , entryForm.FilledPrimaryKeyCondition);
                    }                   
                }
                #endregion

                #region[Run Save Procesures]
                bool isValid = false;
                if (result == GlobalErrors.Success && entryForm.IsAddMode && entryForm.ActionProcedures != null && entryForm.ActionProcedures.Any(x => x.ActionId == 1))
                {
                    result = PerformActionProcedure(entryForm.FilledPrimaryKeyCondition, entryForm.ActionProcedures.Find(x => x.ActionId == 1).ProcedureName, ref isValid, ref validationMessage, dbHelper);
                    if (!isValid)
                    {
                        result = GlobalErrors.Error;
                    }
                }
                else if (result == GlobalErrors.Success && entryForm.IsEditMode && entryForm.ActionProcedures != null && entryForm.ActionProcedures.Any(x => x.ActionId == 2))
                {
                    result = PerformActionProcedure(entryForm.FilledPrimaryKeyCondition, entryForm.ActionProcedures.Find(x => x.ActionId == 2).ProcedureName, ref isValid, ref validationMessage, dbHelper);
                    if (!isValid)
                    {
                        result = GlobalErrors.Error;
                    }
                }
                else
                {
                    isValid = true;
                }
                if (isValid)
                {
                    validationMessage = MessagesConstants.Desc_Saved_Successfully;
                }
                if (result != GlobalErrors.Success)
                {
                    return result;
                }
                #endregion

                #region  Save Attachments
                if (entryForm.HasAttachments)
                {
                    result =  AddAttachments(entryForm, hostingEnvironment, entryForm.WebMenuURL, attachedFiles);
                }
                #endregion
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dbHelper != null)
                {
                    if (result == GlobalErrors.Success)
                    {
                        dbHelper.CommitTransaction();
                    }
                    else
                    {
                        dbHelper.RollBackTransaction();
                    }
                    dbHelper.Dispose();
                    dbHelper = null;
                }
            }
            return result;
        }
        public GlobalErrors CheckForUniqueValidation(WebFormModel entryForm, ref bool isValid, ref string validationMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            DBHelper<int> dbHelper = null;
            try
            {
                List<WebFormControlModel> uniqueControls = new List<WebFormControlModel>();
                uniqueControls = entryForm.Controls.Where(c => c.ValidationId == WebFormValidation.Unique || c.ValidationId == WebFormValidation.RequiredUnique).ToList();
                foreach (WebFormControlModel control in uniqueControls)
                {
                    string excludedString = string.Empty;
                    bool isUniqueValue = false;
                    if (entryForm.IsEditMode && entryForm.Controls.Any(x => x.IsPrimary && x.TableName == control.TableName))
                    {
                        List<WebFormControlModel> primaryKeys = entryForm.Controls.Where(x => x.IsPrimary && x.TableName == control.TableName).ToList();
                        if (primaryKeys != null && primaryKeys.Count > 0)
                        {
                            foreach (WebFormControlModel primary in primaryKeys)
                            {
                                excludedString += string.Format(" And ({0} != {1}) ", primary.DBColumnName, primary.SelectedValue);
                                excludedString += ",";
                            }
                        }
                        excludedString = excludedString.TrimEnd(',');
                    }
                    result = _masterDataService.CheckIfUniqueValue(control.TableName, control.DBColumnName, control.SelectedValue, excludedString, ref isUniqueValue);
                    if (!isUniqueValue)
                    {
                        isValid = false;
                        validationMessage = ResourcesManager.TranslateKey(control.Description, _requestRepository.LanguageId) + " : " + ResourcesManager.TranslateKey(MessagesConstants.Desc_Msg_Must_Be_Unique, _requestRepository.LanguageId);
                        return result;
                    }
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors PerformActionProcedure(string primaryKeyCondition, string procedureName, ref bool isValid, ref string validationMessage, DBHelper<int> dBHelper)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            bool disposeNow = false;

            try
            {
                if (dBHelper == null) // for delete action db transaction must open in procedure, for save action db transaction already opened before so use it. 
                {
                    dBHelper = new DBHelper<int>();
                    disposeNow = true;
                }
                Dictionary<string, object> inputParamters = new Dictionary<string, object>();
                Dictionary<string, object> outputParamters = new Dictionary<string, object>();
                Dictionary<string, object> resultParameters = null;

                inputParamters.Add("@PrimaryKeyCondition", primaryKeyCondition);
                inputParamters.Add("@EmployeeId", _requestRepository.CurrentOperator.EmployeeId);

                outputParamters.Add("@IsValid", 0);
                outputParamters.Add("@Result", 0);
                outputParamters.Add("@Msg", string.Empty);

                if (DBManager.CheckIfProcedureExisted(procedureName))
                {
                    result = dBHelper.ExecuteProcedureWithParameters(procedureName, inputParamters, outputParamters, ref resultParameters, 0, disposeNow);
                    if (result == GlobalErrors.Success)
                    {
                        if (resultParameters.ContainsKey("@IsValid"))
                        {
                            isValid = resultParameters["@IsValid"].ToString().Trim() == "1" ? true : false;
                        }
                        if (resultParameters.ContainsKey("@Msg"))
                        {
                            validationMessage = resultParameters["@Msg"].ToString();
                        }
                        if (resultParameters.ContainsKey("@Result"))
                        {
                            result = (GlobalErrors)(int.Parse(resultParameters["@Result"].ToString()));
                        }
                    }
                }
                else
                {
                    result = GlobalErrors.Error;
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            finally
            {
                if (dBHelper != null && disposeNow)
                {
                    dBHelper.Dispose();
                    dBHelper = null;
                }
            }
            return result;
        }

        public GlobalErrors PerformActionProcedure(PerformActionProcedureModel filter, ref bool isValid, ref string validationMessage)
        {
            GlobalErrors result = GlobalErrors.NotInitialized;
            try
            {
                string primaryKeyCondition = string.Empty;
                isValid = true;
                if (filter.PrimaryKeyColumnList != null && filter.PrimaryKeyColumnList.Count > 0)
                {
                    int index = 0;
                    foreach (PrimaryKeyModel primary in filter.PrimaryKeyColumnList)
                    {
                        if (index > 0)
                        {
                            primaryKeyCondition += " And ";
                        }
                        if (primary.ColumnTypeId == WebFormColumnType.Date)
                        {
                            DateTime d = DateTime.Parse(primary.Value);
                            primaryKeyCondition += primary.ColumnName + '=' + LocalUtilities.ParseDateAndTimeToSQL(d);
                        }
                        else if (!string.IsNullOrEmpty(primary.Value))
                        {
                            if (primary.ColumnTypeId == WebFormColumnType.Text)
                            {
                                primaryKeyCondition += primary.ColumnName + '=' + "'" + primary.Value.Replace("'", "''") + "'";
                            }
                            else
                            {
                                primaryKeyCondition += primary.ColumnName + '=' + primary.Value;
                            }
                        }
                        index++;
                    }
                }
                result = PerformActionProcedure(primaryKeyCondition, filter.ProcedureName, ref isValid, ref validationMessage, null);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
        public GlobalErrors getOrganizationDetailsForParent(int OrganizationId , ref OrganizationModel organizationModel)
        {
            try
            {
                return _webFormManager.getOrganizationDetailsForParent(OrganizationId , ref organizationModel);
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
        }
        public string GetWebFormPath(ref string path, ref WebFormModel printModel)
        {
            try
            {
                if (printModel.IsFromExport || string.IsNullOrEmpty(printModel.FileName))
                {
                    path = LocalUtilities.WebFormDefaultFolderPath;

                    printModel.Columns = printModel.Columns.Where(x => x.IsVisible == true).ToList();

                    printModel.Columns.ForEach(x =>
                    {
                        x.Title = ResourcesManager.TranslateKey(x.Title, _requestRepository.LanguageId);
                    });
                }
                else
                {
                    path = LocalUtilities.WebFormCustomizedFolderPath + _requestRepository.LanguageId + "/";
                    path = Path.Combine(path, printModel.FileName + ".cshtml");
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
            }
            return path;
        }
        public GlobalErrors PreparePrintingQuery(ref WebFormModel printModel, CustomListFilters SharedTableFilter, ref string path)
        {
            var result = GlobalErrors.NotInitialized;
            try
            {
                string query = _webFormManager.GetWebFormPrintingQuery(printModel.WebFormId, printModel.IsFromExport);
                query = query.ToLower();

                result = _webFormManager.ReplaceSessionParameters(ref query);
                if (result != GlobalErrors.Success)
                {
                    return result;
                }

                printModel.Query = query;
                result = _webFormManager.FillPrinitingQueryData(ref printModel, SharedTableFilter);

                if (result == GlobalErrors.Success)
                {
                    path = GetWebFormPath(ref path, ref printModel);
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return GlobalErrors.Error;
            }
            return result;
        }
    }
}